from khronos.des.primitives.action import Action
from khronos.utils import Check

class Poll(Action):
    """This primitive action type is specifically designed to be used with Checkable objects from  
    the utils package, making a very powerful combination where components can block until a given  
    condition on the Checkable's value is observed. The first argument should be a condition 
    function taking a value as only argument and returning a boolean value, and the second 
    argument should be a Checkable object. An optional priority value may be passed to the 
    constructor to manipulate the order of checks in the Checkable object. This primitive succeeds 
    when the check activates, making the process block until the condition on the Checkable is 
    satisfied."""
    def __init__(self, condition, checkable, priority=0.0):
        Action.__init__(self)
        self.check = Check(condition, self.fulfill, priority, sticky=False)
        self.checkable = checkable
        self.fulfilled = False
        self.result = None
        
    def info(self):
        info = self.check.condition.__name__
        if self.fulfilled:
            info += " -> " + str(self.result)
        return info
        
    def reset(self):
        self.satisfied = False
        self.result = None
        
    def deploy(self):
        self.checkable.insert(self.check)
        
    def retract(self):
        if not self.fulfilled:
            self.checkable.remove(self.check)
            
    def fulfill(self, value):
        if self.fulfilled:
            raise ValueError("poll is already fulfilled")
        self.fulfilled = True
        self.result = value
        self.succeed()
        
    # -----------------------------------------------------
    # Poll factory functions ------------------------------
    @staticmethod
    def less_than(value, checkable, priority=0.0):
        def condition(v):
            return v < value
        condition.__name__ = "< %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def less_or_equal(value, checkable, priority=0.0):
        def condition(v):
            return v <= value
        condition.__name__ = "<= %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def greater_than(value, checkable, priority=0.0):
        def condition(v):
            return v > value
        condition.__name__ = "> %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def greater_or_equal(value, checkable, priority=0.0):
        def condition(v):
            return v >= value
        condition.__name__ = ">= %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def equal_to(value, checkable, priority=0.0):
        def condition(v):
            return v == value
        condition.__name__ = "== %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def not_equal_to(value, checkable, priority=0.0):
        def condition(v):
            return v != value
        condition.__name__ = "!= %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def member_of(value, checkable, priority=0.0):
        def condition(v):
            return v in value
        condition.__name__ = "in %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def not_member_of(value, checkable, priority=0.0):
        def condition(v):
            return v not in value
        condition.__name__ = "not in %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def contains(value, checkable, priority=0.0):
        def condition(v):
            return value in v
        condition.__name__ = "contains %s" % (value,)
        return Poll(condition, checkable, priority)
        
    @staticmethod
    def not_contains(value, checkable, priority=0.0):
        def condition(v):
            return value not in v
        condition.__name__ = "not contains %s" % (value,)
        return Poll(condition, checkable, priority)
        
