from collections import deque
from math import sqrt

from khronos.utils import INF
from khronos.statistics.ftable import FTable
from khronos.statistics.plotter import get_plotter

class Tally(object):
    def __init__(self, storing=False, numeric=True):
        self.__storing = bool(storing)
        self.__numeric = bool(numeric)
        # if storing, values and weights are stored in deques
        if self.__storing:
            self.__values = deque()
            self.__weights = deque()
        else:
            self.__values = None
            self.__weights = None
        # attributes for calculating simple statistics
        self.__min = INF
        self.__max = -INF
        # for mean, var, stddev, wmean, wvar, and wstddev
        self.__sum = 0.0
        self.__sumsquares = 0.0
        self.__wsum = 0.0
        self.__wsumsquares = 0.0
        # frequency tables (normal and weighted)
        self.__ftable = FTable()
        self.__wftable = FTable()
        
    def __call__(self):
        print self.report()
        
    def report(self):
        line_fmt = "\t%10s = %s"
        lines = [repr(self), 
                 line_fmt % ("count", self.count()), 
                 line_fmt % ("min", self.min()), 
                 line_fmt % ("max", self.max()), 
                 line_fmt % ("sum", self.sum()), 
                 "\t--------", 
                 line_fmt % ("mean", self.mean()), 
                 line_fmt % ("var", self.var()), 
                 line_fmt % ("stddev", self.stddev()), 
                 "\t--------", 
                 line_fmt % ("wmean", self.wmean()), 
                 line_fmt % ("wvar", self.wvar()), 
                 line_fmt % ("wstddev", self.wstddev())]
        return "\n".join(lines)
        
    def clear(self, storing=None, numeric=None):
        if storing is not None: self.__storing = bool(storing)
        if numeric is not None: self.__numeric = bool(numeric)
        
        if self.__storing:
            self.__values = deque()
            self.__weights = deque()
        else:
            self.__values = None
            self.__weights = None
        
        self.__min = INF
        self.__max = -INF
        
        self.__sum = 0.0
        self.__sumsquares = 0.0
        self.__wsum = 0.0
        self.__wsumsquares = 0.0
        
        self.__ftable.clear()
        self.__wftable.clear()
        
    def is_storing(self):
        return self.__storing
        
    def is_numeric(self):
        return self.__numeric
        
    def collect(self, value, weight=1.0):
        """Register a value with a specified weight in this tally. If no weight is specified, the 
        default value of 1 is used. Basically, this method does the same as a lazy collector, but 
        all the data is inserted in one step, instead of providing the value first and the weight 
        later."""
        if self.__storing:
            self.__values.append(value)
            self.__weights.append(weight)
        if self.__numeric:
            squared_value = value * value
            self.__min = min(self.__min, value)
            self.__max = max(self.__max, value)
            self.__sum += value
            self.__sumsquares += squared_value
            self.__wsum += value * weight
            self.__wsumsquares += squared_value * weight
        self.__ftable.collect(value, 1.0)
        self.__wftable.collect(value, weight)
        
    def lazy_collect(self, value):
        """This method is used by the timeseries class to insert a value in the tally when its
        interval starts, but only assign a weight to it when the interval is closed."""
        collector = self.__lazy_collector(value)
        collector.next()
        return collector
        
    def __lazy_collector(self, value):
        """This generator does the registration of a value with a weight in two steps. First only
        the value is provided, so the min, max, sum, and ftable are updated. The second step 
        involves sending the weight to the generator which then updates the weighted statistics and
        stores the value and weight if necessary."""
        # When the generator is entered, the value just appeared, so we update the minimum, 
        # maximum, sum, sumsquares, and frequency table to include this value.
        squared_value = None
        if self.__numeric:
            squared_value = value * value
            self.__min = min(self.__min, value)
            self.__max = max(self.__max, value)
            self.__sum += value
            self.__sumsquares += squared_value
        self.__ftable.collect(value, 1.0)
        # Interval closed, weight is expected by the generator. Update the remaining indicators
        # and store the value and weight if applicable.
        weight = yield
        if self.__numeric:
            self.__wsum += value * weight
            self.__wsumsquares += squared_value * weight
        if self.__storing:
            self.__values.append(value)
            self.__weights.append(weight)
        self.__wftable.collect(value, weight)
        yield
        
    def merge(self, tally):
        if not isinstance(tally, Tally):
            raise TypeError("Tally expected")
        if self.__storing and not tally.__storing:
            raise ValueError("storing tally cannot merge with non-storing tally")
        if self.__numeric != tally.__numeric:
            raise ValueError("numeric tally cannot merge with non-numeric tally")
            
        if self.__storing:
            self.__values.extend(tally.__values)
            self.__weights.extend(tally.__weights)
            
        if self.__numeric:
            self.__min = min(self.__min, tally.__min)
            self.__max = max(self.__max, tally.__max)
            self.__sum += tally.__sum
            self.__sumsquares += tally.__sumsquares
            self.__wsum += tally.__wsum
            self.__wsumsquares += tally.__wsumsquares
            
        self.__ftable.merge(tally.__ftable)
        self.__wftable.merge(tally.__wftable)
        
    # -----------------------------------------------
    def __len__(self):
        return int(self.__ftable.total())
        
    def __getitem__(self, idx):
        if not self.__storing:
            raise ValueError("unsubscriptable object (non-storing)")
        return self.__values[idx], self.__weights[idx]
        
    def __iter__(self):
        if not self.__storing:
            raise ValueError("uniterable object (non-storing)")
        idx = 0
        while idx < len(self.__values):
            yield self.__values[idx], self.__weights[idx]
            idx += 1
            
    def iter_values(self):
        for value, _ in self:
            yield value
            
    def iter_weights(self):
        for _, weight in self:
            yield weight
            
    # -----------------------------------------------
    def count(self):
        return self.__ftable.total()
        
    def min(self):
        return self.__min
        
    def max(self):
        return self.__max
        
    def sum(self):
        return self.__sum
        
    # -----------------------------------------------
    def mean(self):
        try:
            return self.__sum / self.__ftable.total()
        except ZeroDivisionError:
            return 0.0
            
    def var(self):
        try:
            return self.__sumsquares / self.__ftable.total() - self.mean() ** 2
        except ZeroDivisionError:
            return 0.0
            
    def stddev(self):
        return sqrt(self.var())
        
    def wmean(self):
        try:
            return self.__wsum / self.__wftable.total()
        except ZeroDivisionError:
            return 0.0
            
    def wvar(self):
        try:
            return self.__wsumsquares / self.__wftable.total() - self.wmean() ** 2
        except ZeroDivisionError:
            return 0.0
            
    def wstddev(self):
        return sqrt(self.wvar())
        
    # -----------------------------------------------
    def ftable(self):
        return self.__ftable
        
    def abs_frequency(self, value):
        return self.__ftable.abs_frequency(value)
        
    def rel_frequency(self, value):
        return self.__ftable.rel_frequency(value)
        
    def wftable(self):
        return self.__wftable
        
    def wabs_frequency(self, value):
        return self.__wftable.abs_frequency(value)
        
    def wrel_frequency(self, value):
        return self.__wftable.rel_frequency(value)
        
    # -----------------------------------------------
    def pie_chart(self, *args, **kwargs):
        return self.__wftable.pie_chart(*args, **kwargs)
        
    def bar_chart(self, *args, **kwargs):
        return self.__wftable.bar_chart(*args, **kwargs)
        
    def pareto_chart(self, *args, **kwargs):
        return self.__wftable.pareto_chart(*args, **kwargs)
        
    def histogram(self, *args, **kwargs):
        return self.__wftable.histogram(*args, **kwargs)
        
    def box_plot(self, axes=None, *args, **kwargs):
        plotter, axes = get_plotter(axes)
        return plotter.box_plot(self.__values, axes=axes, *args, **kwargs)
        
