from datetime import datetime
import shelve
import os

class DataStore(object):
    """Subclass and redefine the open(), close(), and write(data) methods."""
    def __init__(self, file, buffer_size=100):
        self.file = file
        self.buffer = []
        self.buffer_size = buffer_size
        
    def __del__(self):
        self.flush()
        
    def collect(self, data):
        self.buffer.append(data)
        if len(self.buffer) > self.buffer_size:
            self.flush()
            
    def flush(self):
        if len(self.buffer) > 0:
            self.open()
            while len(self.buffer) > 0:
                self.write(self.buffer[0])
                self.buffer.pop(0)
            self.close()
            
    def open(self):
        raise NotImplementedError()
        
    def close(self):
        raise NotImplementedError()
        
    def write(self, data):
        raise NotImplementedError()
        
weekdays = ["Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"]

def default_tags():
    """Default tags include the weekday, day, month, and year to allow easy filtering of data by 
    date."""
    now = datetime.now()
    tags = set()
    tags.add(weekdays[now.weekday()])
    tags.add("D%d" % (now.day))
    tags.add("M%d" % (now.month,))
    tags.add("Y%d" % (now.year,))
    tags.add("all")
    return tags
    
class SimpleDB(object):
    """Simple data store. This class allows saving and loading data from shelve databases, and 
    associating tags to data. Stores also provide a very simple version control mechanism, where 
    the last 'history' versions are kept as backups."""
    def __init__(self, filepath=None, history=10):
        self.filepath = filepath
        self.history = history
        self.tags = {} # {obj: set([tag])}
        self.data = {} # {tag: set([obj])}
        if filepath is not None and os.path.exists(filepath):
            self.load(filepath)
            
    def clear(self):
        self.tags.clear()
        self.data.clear()
        
    def merge(self, store):
        for obj, tags in store.tags.iteritems():
            self.add(obj, tags)
            
    def add(self, obj, *tags):
        tags = set(tags) | default_tags()
        try:             self.tags[obj].update(tags)
        except KeyError: self.tags[obj] = set(tags)
        for tag in tags:
            try:             self.data[tag].add(obj)
            except KeyError: self.data[tag] = set([obj])
            
    def remove(self, obj):
        raise NotImplementedError()
        
    def filter(self, include_tags, exclude_tags=()):
        results = set(self.data[include_tags[0]])
        for tag in include_tags[1:]:
            results.intersection_update(self.data[tag])
        for tag in exclude_tags:
            results.difference_update(self.data[tag])
        return results
        
    def tag(self, obj, *tags):
        raise NotImplementedError()
        
    def untag(self, obj, *tags):
        raise NotImplementedError()
        
    def add_tag(self, tag, *objs):
        raise NotImplementedError()
        
    def remove_tag(self, tag, *objs):
        raise NotImplementedError()
        
    def save(self, filepath=None):
        db = shelve.open(self.filepath if filepath is None else filepath)
        try:             data = db["store"]
        except KeyError: data = db["store"] = []
        if len(data) > self.history:
            data.pop()
        data.insert(0, (str(datetime.now()), self.tags))
        db["store"] = data
        db.close()
        
    def load(self, filepath=None, position=0):
        if filepath is None:
            filepath = self.filepath
        db = shelve.open(filepath)
        date, tagmap = db["store"][position]
        for obj, tags in tagmap.iteritems():
            self.add(obj, *tags)
        db.close()
        
    def list(self):
        if not os.path.exists(self.filepath):
            raise IOError("file not found")
        db = shelve.open(self.filepath)
        print "Store %s" % (self.filepath,)
        for i, (date, tagmap) in enumerate(db["store"]):
            print "\t%2d - %s" % (i, date)
        db.close()
        