from sys import stdout, stderr
from contextlib import contextmanager

from khronos.utils.misc import Switchable, fit, indent
from khronos.utils.trailable import Trailable

reserved_streams = set(["Info", "Default", "Debug", "Warning", "Error"])

class Logger(Switchable):
    """Log manager to facilitate the manipulation of multiple log streams."""
    def __init__(self, streams=(), show_name=False, out=stdout, enabled=True):
        Switchable.__init__(self, enabled)
        self.__streams = dict(Info=LogStream("Info"), 
                              Default=LogStream("Default", show_name=False), 
                              Debug=LogStream("Debug", enabled=False), 
                              Warning=LogStream("Warning", out=stderr), 
                              Error=LogStream("Error", out=stderr))
        self.mode = Trailable(self.__getconfig, self.__setconfig)
        # Fill logger with streams and save config as normal mode.
        streams = list(streams)
        self.add_streams(streams, show_name=show_name, out=out)
        self.mode.push("Normal")
        # Create and save debug mode.
        self.disable(list(self.streams()))
        self.enable(["Debug", "Warning", "Error"])
        self.mode.push("Debug")
        # +++
        self.mode.set("Normal")
        
    def __getconfig(self):
        return [(s, s.enabled) for s in self.__streams.itervalues()]
        
    def __setconfig(self, config):
        for s, enabled in config:
            s.set_enabled(enabled)
            
    # --------------------------------------------------------------------------
    def streams(self):
        return self.__streams.iterkeys()
        
    def add_stream(self, name, show_name=True, out=stdout, enabled=True):
        if name not in self.__streams:
            self.__streams[name] = LogStream(name, show_name, out, enabled)
            
    def add_streams(self, names, show_name=True, out=stdout, enabled=True):
        for name in names:
            self.add_stream(name, show_name, out, enabled)
            
    def del_stream(self, name):
        del self[name]
        
    def del_streams(self, names):
        for name in names:
            del self[name]
            
    def get_stream(self, name):
        return self[name]
        
    def __getitem__(self, name):
        return self.__streams[name]
        
    def __delitem__(self, name):
        if name in reserved_streams:
            raise ValueError("attempting to delete reserved log stream (%s)" % (name,))
        del self.__streams[name]
        
    def __iter__(self):
        return self.__streams.itervalues()
        
    # --------------------------------------------------------------------------
    def __call__(self, message, stream="Default", width=None, indent=None, newline=True):
        if self.enabled:
            self.__streams[stream](message, width, indent, newline)
            
    def info(self, message, width=None, indent=None, newline=True):
        self(message, "Info", width, indent, newline)
        
    def debug(self, message, width=None, indent=None, newline=True):
        self(message, "Debug", width, indent, newline)
        
    def warn(self, message, width=None, indent=None, newline=True):
        self(message, "Warning", width, indent, newline)
        
    def error(self, message, width=None, indent=None, newline=True):
        self(message, "Error", width, indent, newline)
        
    # --------------------------------------------------------------------------
    def enable(self, streams=()):
        if len(streams) == 0:
            Switchable.enable(self)
        else:
            for name in streams:
                self.__streams[name].enable()
                
    def disable(self, streams=()):
        if len(streams) == 0:
            Switchable.disable(self)
        else:
            for name in streams:
                self.__streams[name].disable()
                
    def toggle(self, streams=()):
        if len(streams) == 0:
            Switchable.toggle(self)
        else:
            for name in streams:
                self.__streams[name].toggle()
                
    @contextmanager
    def temp_set_enabled(self, enabled):
        previous = self.enabled
        self.set_enabled(enabled)
        yield
        self.set_enabled(previous)
        
    def status(self, display=True):
        lines = [Switchable.__repr__(self)]
        for name, stream in sorted(self.__streams.iteritems()):
            lines.append("\t%s" % (stream,))
        if display:
            print "\n".join(lines)
        else:
            return "\n".join(lines)
            
class LogStream(Switchable):
    """Class implementing a simple stream of log messages written to a file. Used by the Logger
    class, a more complex log manager object containing multiple log streams."""
    def __init__(self, name, show_name=True, out=stdout, enabled=True):
        self.name = name
        self.show_name = Switchable(show_name)
        self.out = out
        self.continuation = False
        self.__logfnc = None
        Switchable.__init__(self, enabled)
        
    def __repr__(self):
        return "<%s(%s, %s) at %s>" % (self.__class__.__name__, self.name, 
                                       "enabled" if self.enabled else "disabled", 
                                       hex(id(self)))
                                       
    def set_enabled(self, enabled):
        Switchable.set_enabled(self, enabled)
        self.__logfnc = self.__log if self.enabled else self.__nolog
        
    def __call__(self, message, width=None, indent=None, newline=True):
        self.__logfnc(message, width, indent, newline)
        
    def __log(self, message, width, indlevel, newline):
        if not self.show_name.enabled or self.continuation:
            message = str(message)
        else:
            message = "{%s} %s" % (self.name, message)
        if indlevel is not None:
            message = indent(message, indlevel)
        if width is not None:
            message = fit(message, width)
        self.out.write(message)
        self.continuation = not newline
        if newline:
            self.out.write("\n")
            
    def __nolog(self, message, width, indlevel, newline):
        pass
        