from khronos.utils.call import Call

class MethodGroup(object):
    """Descriptor class for method groups. Method groups can be used to join a set of related 
    methods under a single entry in the class' namespace. 
    NOTE: this only works with instance methods, and the objects must have a __dict__ attribute. 
    Example: 
        class Foo(object):
            def __f1(self): print "f1"
            def __f2(self): print "f2"
            pi = MethodGroup("pi", f1=__f1, f2=__f2)
            
        f = Foo()
        f.pi.f1()
        f.pi.f2()
        
    A decorator approach can also be used to add methods to the group:  
        class Bar(object):
            pi = MethodGroup("pi")
            @pi("f1")
            def __f1(self): print "f1"
            @pi("f2")
            def __f2(self): print "f2"
            
        b = Bar()
        b.pi.f1()
        b.pi.f2()
    """
    def __init__(self, name, *fncs, **kwfncs):
        # Create a new method group class.
        class mgroup(object):
            __slots__ = ["__target"]
            def __init__(self, target):
                self.__target = target
                
            @staticmethod
            def __wrap__(fnc, fnc_name=None):
                def new_fnc(self, *args, **kwargs):
                    return fnc(self.__target, *args, **kwargs)
                new_fnc.__name__ = fnc.__name__ if fnc_name is None else fnc_name
                new_fnc.__doc__  = fnc.__doc__
                return new_fnc
        # Set the group's name and populate with the given methods
        mgroup.__name__ = name
        for fnc in fncs:
            setattr(mgroup, fnc.__name__, mgroup.__wrap__(fnc))
        for fnc_name, fnc in kwfncs.iteritems():
            setattr(mgroup, fnc_name, mgroup.__wrap__(fnc, fnc_name))
        # Save the interface to the descriptor
        self.__mgroup = mgroup
        
    def __get__(self, obj, cls):
        if obj is None:
            return self
        try:
            obj_mgroups = obj.__mgroups
        except AttributeError:
            bound_mgroup = self.__mgroup(obj)
            obj_mgroups = {self.__mgroup.__name__: bound_mgroup}
            obj.__mgroups = obj_mgroups
            return bound_mgroup
        try:
            bound_mgroup = obj_mgroups[self.__mgroup.__name__]
        except KeyError:
            bound_mgroup = self.__mgroup(obj)
            obj_mgroups[self.__mgroup.__name__] = bound_mgroup
        return bound_mgroup
        
    def __call__(self, fnc, name=None):
        """Function decorator that adds the decorated function to the interface."""
        if isinstance(fnc, str):
            # renaming decorator method
            return Call(self, name=fnc)
        if name is None:
            name = fnc.__name__
        setattr(self.__mgroup, name, self.__mgroup.__wrap__(fnc, name))
        return fnc
        