from contextlib import contextmanager
from sys import stdout
import random

from khronos.utils.textprocessing import TextProcessor

__txt = TextProcessor()
indentation = __txt.indentation
indent_line = __txt.indent_line
indent      = __txt.indent
fit_line    = __txt.fit_line
fit         = __txt.fit
rfit_line   = __txt.rfit_line
rfit        = __txt.rfit

def identity_fnc(x):
    return x
    
def rgb_to_str(r, g, b):
    return "#%02x%02x%02x" % (r, g, b)
    
def xgetattr(obj, location):
    attrs = location.split(".")
    for attr in attrs:
        obj = getattr(obj, attr)
    return obj
    
def xsetattr(obj, location, value):
    attrs = location.split(".")
    for attr in attrs[:-1]:
        obj = getattr(obj, attr)
    setattr(obj, attrs[-1], value)
    
def xdelattr(obj, location):
    attrs = location.split(".")
    for attr in attrs[:-1]:
        obj = getattr(obj, attr)
    delattr(obj, attrs[-1])
    
def is_sorted(iterable, key=None):
    if key is None:
        key = lambda x: x
    iterator = iter(iterable)
    prev_key = key(iterator.next())
    for item in iterator:
        curr_key = key(item)
        if curr_key < prev_key:
            return False
        prev_key = curr_key
    return True
    
def biased_choice(items, biases, rng=random):
    """Randomly select an element from a list using different probabilities for selecting each
    element. The first argument should be a list (or list-like object) of python objects of any 
    type, and 'biases' should be a list of the same size containing the bias of the item in the 
    first list located at the same index. Biases should be non-negative real numbers of any 
    magnitude (not just in [0, 1], because their relative sizes dictate their associated 
    probability p_i = b_i / sum(b_j for all j))."""
    a = rng.uniform(0.0, sum(biases))
    x = 0.0
    i = 0
    for b in biases:
        x += b
        if x >= a:
            return items[i]
        i += 1
    raise Exception("They came from behind!!!")
    
try:
    # Clock function replacement for unix systems.
    from resource import getrusage, RUSAGE_SELF
    def clock():
        """clock() -> floating point number
        
        Return the CPU time in seconds (user + system) since the start of the
        process. This is done via a call to resource.getrusage(), so it avoids
        the wraparound problems in time.clock()."""
        res = getrusage(RUSAGE_SELF)
        return res[0] + res[1]
except ImportError:
    # For other systems the time module's clock() is okay.
    from time import clock
    
# ---------------------------------------------------------
# Small classes that don't justify separate files ---------
class Namespace(dict):
    """Utility class that works as a dict or as a regular object. Users can add string keys 
    through attribute assignment, and/or regular subscripting. The following are equivalent:
        ns.x = 3
        ns["x"] = 3
    For non-string keys subscripting is required, as the attribute-like notation only allows 
    string keys with the usual character limitations (no #$!? etc. allowed)."""
    __getattr__ = dict.__getitem__
    __setattr__ = dict.__setitem__
    __delattr__ = dict.__delitem__
    
    @property
    def __dict__(self):
        return self
        
    def __getstate__(self):
        return dict(self)
        
    def __setstate__(self, dictionary):
        self.update(dictionary)
        
    def copy(self):
        return self.__class__(self)
        
class Singleton(object):
    """This class implements a singleton constructor. To make a class singleton (i.e. have only 
    one existing instance), just subclass the Singleton class, and voila!"""
    def __new__(cls, *args, **kwargs):
        if not hasattr(cls, "__singleton__"):
            setattr(cls, "__singleton__", object.__new__(cls, *args, **kwargs))
        return getattr(cls, "__singleton__")
        
class Switchable(object):
    """Something that can be turned on, off, and toggled between states."""
    def __init__(self, enabled=True):
        self.__enabled = False
        self.enabled = enabled
        
    def __repr__(self):
        state = "enabled" if self.__enabled else "disabled"
        return "<%s(%s) at 0x%08x>" % (self.__class__.__name__, state, id(self))
        
    def get_enabled(self):
        return self.__enabled
        
    def set_enabled(self, value):
        self.__enabled = bool(value)
        
    def enable(self):
        self.enabled = True
        
    def disable(self):
        self.enabled = False
        
    def toggle(self):
        self.enabled = not self.__enabled
        
    @property
    def enabled(self):
        return self.get_enabled()
    
    @enabled.setter
    def enabled(self, value):
        self.set_enabled(value)
        
class Reportable(object):
    """This class implements a inspect() method which prints a nicely formatted report consisting 
    of (key, value) pairs returned by the report_items() method (which should be redefined in 
    subclasses)."""
    report_block_lim = "|"
    report_block_sep = "+" + "-" * 20
    report_line_fmt = "%20s: %s"
    
    def inspect(self, display=True):
        report = self.__class__.report(self.report_items(), title=object.__repr__(self))
        if display:
            print report
        else:
            return report
            
    def report_items(self):
        return ()
        
    @classmethod
    def report(cls, items, title=None):
        lines = []
        if title is not None:
            lines.append(str(title))
        if len(items) > 0:
            line_fmt = cls.report_line_fmt
            block_sep = cls.report_block_sep
            block_lim = cls.report_block_lim
            block_ind = block_lim + "   "
            custom_indent = TextProcessor(indent=block_ind).indent
            stack = [(items, 0)]
            while len(stack) > 0:
                block, pos = stack.pop()
                ind = len(stack)
                if pos == 0:
                    lines.append(custom_indent(block_sep, ind))
                while pos < len(block):
                    item = block[pos]
                    if isinstance(item, list):
                        stack.append((block, pos + 1))
                        stack.append((item, 0))
                        break
                    lines.append(custom_indent(block_lim + line_fmt % item, ind))
                    pos += 1
                if pos == len(block):
                    lines.append(custom_indent(block_sep, ind))
        return "\n".join(lines)
        
class CPUTracker(Reportable):
    """CPU tracker object. To track the CPU time of a function just call it in a tracking() 
    context block, like:
        tracker = CPUTracker()
        with tracker.tracking():
            do_stuff()
        print "Elapsed CPU time:", tracker.total()
    """
    def __init__(self):
        self.__start = None
        self.__cpu = 0.0
        self.__tracks = 0
        
    def __str__(self):
        return "%s(%fs [%d tracks])" % (self.__class__.__name__, self.total(), self.__tracks)
        
    def report_items(self):
        return [("Total Time", self.total()), 
                ("Current Tracks", self.__tracks)]
                
    def active(self):
        return self.__tracks > 0
        
    def total(self):
        if not self.__tracks:
            return self.__cpu
        else:
            return self.__cpu + (clock() - self.__start)
            
    def clear(self, force=False):
        if not force and self.__tracks > 0:
            raise ValueError("cannot reset CPU tracker - active track(s) remaining")
        self.__start = None
        self.__cpu = 0.0
        self.__tracks = 0
        
    def start(self):
        if self.__tracks == 0:
            self.__start = clock()
        self.__tracks += 1
        
    def stop(self):
        if self.__tracks == 0:
            raise ValueError("cannot stop CPU tracker - no active track(s) remaining")
        self.__tracks -= 1
        if self.__tracks == 0:
            self.__cpu += clock() - self.__start
            self.__start = None
            
    @contextmanager
    def tracking(self):
        self.start()
        yield
        self.stop()
        
class Undefined(Singleton):
    """A singleton class that can be useful for functions or methods where you can't find a good 
    default value to signal a missing argument, e.g. where None can be interpreted as data."""
    pass
    
UNDEF = Undefined()
INF   = 2e1000
