import datetime
from django.contrib.auth.models import User
from django.contrib.contenttypes.models import ContentType
from django.contrib.contenttypes import generic
from django.core.urlresolvers import reverse
from django.db import connection
from django.db import models

class ProcessResult:
    failure = 0
    success = 1
    unknown = 2
    pending = 3
    finished_with_warnings = 4
    none = 5
    skipped = 6


ProcessResultDict = {
        ProcessResult.failure: 'Failure',
        ProcessResult.success: 'Success',
        ProcessResult.unknown: 'Unknown',
        ProcessResult.pending: 'Pending',
        ProcessResult.finished_with_warnings: 'Finished with warnings',
        ProcessResult.skipped: 'Skipped',
        ProcessResult.none: ''}


ProcessResultChoices = ProcessResultDict.items()


class Process(models.Model):
    """

    """
    owner = models.ForeignKey(User)
    text = models.CharField(max_length=100, blank=True, null=True)
    registration_date = models.DateTimeField(auto_now=True)
    started = models.DateTimeField(null=True, blank=True)
    finished = models.DateTimeField(null=True, blank=True)
    workflow = models.CharField(max_length=100, blank=True, null=True, default="")
    result = models.IntegerField(choices=ProcessResultChoices, default=ProcessResult.unknown)
    global_process = models.BooleanField(default=True)

    def add_text(self, text):
        """

        """
        if self.text:
            self.text = "%s\n\n%s" % (self.text, text)
        else:
            self.text = text

    @classmethod
    def start(cls, user, text=None):
        return cls.objects.create(owner=user, text=text, registration_date=datetime.datetime.now())

    def __unicode__(self):
        return "<Owner: %s. Reg.date: %s, Started: %s, finished: %s>" % \
               (self.owner, self.registration_date, self.started, self.finished)

    def details_link(self):
        url = reverse('admin:kolibri_processitem_changelist', args=[])
        return '<a href="%s?parent__id__exact=%s">Details</a>' % (url, self.id)
    details_link.short_description = 'Link to details'
    details_link.allow_tags = True


class ProcessItem(models.Model):
    """

    """
    class Meta:
        ordering = ['-started','-finished', 'sort_order', 'and_finally']

    owner = models.ForeignKey(User)
    parent = models.ForeignKey(Process, related_name='items')
    text = models.TextField(blank=True, null=True)
    processor_identifier = models.CharField(max_length=100)
    content_type =  models.ForeignKey(ContentType, null = True, blank = True, related_name='process_log_entries')
    object_id = models.PositiveIntegerField(null = True, blank = True)
    item = generic.GenericForeignKey('content_type', 'object_id')
    started = models.DateTimeField(null=True, blank=True)
    finished = models.DateTimeField(null=True, blank=True)
    result = models.IntegerField(choices=ProcessResultChoices, default=ProcessResult.unknown)
    and_finally = models.BooleanField(default=False) # this should be run at the end, to clean up
    sort_order = models.PositiveIntegerField(default=0)

    def add_text(self, text):
        """

        """
        if self.text:
            self.text = "%s\n\n%s" % (self.text, text)
        else:
            self.text = text

    @classmethod
    def process_item(cls, parent, processor_identifier, item, and_finally=False):
        return cls.objects.create(owner=parent.owner, parent=parent, processor_identifier=processor_identifier, \
                                  item=item, and_finally=and_finally, sort_order=parent.items.all().count())

    @classmethod
    def execute(cls, parent, processor_identifier):
        return cls.objects.create(parent=parent, owner=parent.owner, processor_identifier=processor_identifier)

    def __unicode__(self):
        return "<Processor: %s, item: %s. Started:%s, finished: %s>" % \
               (self.processor_identifier, self.item, self.started, self.finished)

    @classmethod
    def items_for_instance(cls, instance):
        """

        """
        instance_type = ContentType.objects.get_for_model(instance)
        return cls.objects.filter(content_type__pk=instance_type.id, object_id=instance.id)

    @classmethod
    def items_for_model(cls, model):
        """

        """
        model_type = ContentType.objects.get_for_model(model)
        return cls.objects.filter(content_type__pk=model_type.id)

    def get_admin_link(self):
        """

        """
        if not self.content_type:
            return ''
        
        return reverse('admin:%s_%s_change' % (self.content_type.app_label, self.content_type.model), args=[self.object_id])

    def content_link(self):
        if not self.item:
            return ''
        return '<a href="%s">%s</a>' % (self.get_admin_link(), self.item)

    content_link.short_description = 'Link to item'
    content_link.allow_tags = True

    @classmethod
    def log_for_instance(cls, instance):
        """

        """
        instance_type = ContentType.objects.get_for_model(instance)
        return cls.objects.filter(content_type__pk=instance_type.id, object_id=instance.id)

    @classmethod
    def count_for_instance(cls, instance):
        """

        """
        instance_type = ContentType.objects.get_for_model(instance)
        cursor = connection.cursor()
        cursor.execute("""
        SELECT processor_identifier, count(*) as proc_count
        FROM kolibri_processitem
        WHERE content_type_id = %s
        AND object_id = %s
        GROUP BY processor_identifier
        """, [instance_type.id, instance.id])
        result = {}
        for processor_identifier, item_count in cursor.fetchall():
            result[processor_identifier] = item_count
        return result

    @classmethod
    def count_for_model(cls, model):
        """

        """
        instance_type = ContentType.objects.get_for_model(model)
        cursor = connection.cursor()
        cursor.execute("""
        SELECT processor_identifier, count(*) as proc_count
        FROM kolibri_processitem
        WHERE content_type_id = %s
        GROUP BY processor_identifier
        """, [instance_type.id])
        result = {}
        for processor_identifier, item_count in cursor.fetchall():
            result[processor_identifier] = item_count
        return result


class ProcessException(models.Model):
    """

    """
    process_item = models.ForeignKey(ProcessItem, related_name='exceptions')
    processor_identifier = models.CharField(max_length=100)
    exception_name = models.CharField(max_length=100)

    @classmethod
    def on_exception(cls, process_item, processor_identifier, exception_name):
        return ProcessException.objects.create(
            process_item=process_item,
            processor_identifier=processor_identifier,
            exception_name=exception_name)

    def parent_processor_identifier(self):
        return self.process_item.processor_identifier
    parent_processor_identifier.short_description = 'Parent processor raising exception'

    def item_link(self):
        return reverse('admin:kolibri_processitem_change', args=[self.process_item.id])
    item_link.short_description = 'Link to item'
    item_link.allow_tags = True
