from datetime import datetime
from celery.task import task
from celery.registry import tasks
from kolibri.core import ProcessorNotFound, name_from_item
from kolibri.models import ProcessResult
from django.core.cache import cache
from kolibri.core.context import generate_lock_id


LOCK_EXPIRE = 60 * 5 # Lock expires in 5 minutes

def internal_executor2(process, process_item, processor_identifier, **kwargs):
    from kolibri.core import manager
    processor = manager.processor[processor_identifier]
    if not processor:
        raise ProcessorNotFound(processor_identifier)

    if process_item.item:
        processor.process(process.owner, process_item.item, **kwargs)
        process_item.add_text('%s [ Item: %s. ID: %s.]' % (processor.text, \
                                                         name_from_item(process_item.item), \
                                                         hasattr(process_item.item, 'id') and process_item.item.id or '-'))
    else:
        processor.execute(process.owner, **kwargs)


def handle_exception(process, process_item, processor_exception):
    try:
        internal_executor2(process, process_item, processor_exception.processor_identifier)
        process_item.result = ProcessResult.success
        return True
    except Exception:
        return False


class ProcessException(BaseException):
    """

    """
    pass


def handle_process_item(process, process_item, **kwargs):
    lock_id = generate_lock_id(process.owner, process.global_process, process_item.processor_identifier, process_item.item)

    # cache.add fails if if the key already exists
    acquire_lock = lambda: cache.add(lock_id, "true", LOCK_EXPIRE)
    release_lock = lambda: cache.delete(lock_id)

    process_item.started = datetime.now()
    process_item.result = ProcessResult.success
    if acquire_lock():
        try:
            if process_item.and_finally:
                try:
                    internal_executor2(process, process_item, process_item.processor_identifier, **kwargs)
                except Exception, final_exception:
                    process_item.add_text("Final processor failed with exception: %s." % final_exception)
                    process_item.result = ProcessResult.failure
                    process_item.finished = datetime.now()
                    process_item.save()
                    raise final_exception
            else:
                try:
                    internal_executor2(process, process_item, process_item.processor_identifier, **kwargs)
                except Exception, main_exception:
                    process_item.add_text("Exception raised: %s." % main_exception)
                    exception_handled = False
                    for processor_exception in process_item.exceptions.all():
                        if processor_exception.exception_name == name_from_item(main_exception):
                            if handle_exception(process, process_item, processor_exception):
                                exception_handled = True
                                process_item.result = ProcessResult.finished_with_warnings
                                process_item.add_text("Processor %s fixed exception %s." % \
                                                      (processor_exception.processor_identifier, main_exception))
                                break

                    if not exception_handled:
                        process_item.add_text("Processor %s failed with exception: %s." % \
                                              (process_item.processor_identifier, main_exception))
                        process_item.result = ProcessResult.failure
                        process_item.finished = datetime.now()
                        process_item.save()
                        raise ProcessException("Process_item %s, processor %s, item: %s raised exception %s." % \
                        process_item.id, process_item.processor_identifier, process_item.item, main_exception)

            process_item.finished = datetime.now()
            process_item.save()
        finally:
            release_lock()
            return process_item.result == ProcessResult.success

@task
def executor(process, **kwargs):
    """

    """
    exception_raised = None
    process.started = datetime.now()
    try:
        warnings = 0
        for process_item in process.items.all():
            if not handle_process_item(process, process_item, **kwargs):
                warnings += 1

        process.result = warnings and ProcessResult.finished_with_warnings or ProcessResult.success
    except ProcessException, pex:
        process.add_text("Exception raised: %s." % pex)
        process.result = ProcessResult.failure
        exception_raised = pex

    finally:
        process.finished = datetime.now()
        process.save()

    if exception_raised:
        raise exception_raised


tasks.register(executor)
