# -*- coding: latin-1 -*-
import unittest
from kolibri.core.logging import LoggingContext
from kolibri.core.context import Context
from kolibri.core import manager, name_from_item, Processor, KolibriProcessAllreadyRunning, ProcessStatus
from django.contrib.auth.models import User
from django.db import models
from kolibri.core.workflow import Workflow, WorkflowIsUserExlcusive, \
    WorkflowCannotProcessQuerysetUsingDifferentModel, WorkflowCannotProcessQuerysetWithoutSettingAModel
from kolibri.models import *


class Article(models.Model):
    title = models.CharField(max_length=128)
    text = models.TextField()
    parental_advisory = models.BooleanField(default=False)
    author = models.ForeignKey(User)
    publish = models.BooleanField(default=False)
    archived = models.BooleanField(default=False)

    def __unicode__(self):
        return self.title


dirty_words = ('shiltz', 'fudge',)

class ProfanityRemover(Processor):
    user_exclusive = True
    model = Article
    _text = "Remover of profanity"

    def process(self, user, article, **kwargs):
        if article.id == 4:
            raise ValueError("Exception raised from %s." % self.__class__.__name__)
        
        for dirty_word in dirty_words:
            article.text = article.text.replace(dirty_word, '*'*len(dirty_word))
        article.save()

    def execute(self, user,**kwargs):
        for article in Article.objects.filter(author=user):
            self.process(user, article, **kwargs)

pr = ProfanityRemover()


class ParentalAdvisory(Processor):
    model = Article

    def process(self, user, article,**kwargs):
        LoggingContext.log(user, 'Setting parental advisory for article %s.' % article.id, instance=article)
        if article.id == 4:
            raise IndexError("Exception raised from %s." % self.__class__.__name__)

        dirty_words_counted = 0
        for dirty_word in dirty_words:
            dirty_words_counted += article.text.count(dirty_word)

        if dirty_words_counted > 0:
            article.parental_advisory = True
            article.save()

    def execute(self, user,**kwargs):
        for article in Article.objects.all():
            self.process(user, article, **kwargs)

pa = ParentalAdvisory()

class DirtyArticleRemover(Processor):
    model = Article

    def process(self, user, article,**kwargs):
        if article.id == 4:
            article.delete()
            return

        dirty_words_counted = 0
        for dirty_word in dirty_words:
            dirty_words_counted += article.text.count(dirty_word)

        if dirty_words_counted > 0:
            article.delete()

    def execute(self, user,**kwargs):
        for article in Article.objects.all():
            self.process(user, article, **kwargs)

dar = DirtyArticleRemover()

class PublishArticle(Processor):
    model = Article

    def process(self, user, article,**kwargs):
        article.publish = True
        article.save()

    def execute(self, user,**kwargs):
        for article in Article.objects.all():
            self.process(user, article, **kwargs)

pu = PublishArticle()

class PublishingCleanUp(Processor):
    model = Article

    def process(self, user, article,**kwargs):
        if not article.publish:
            article.archived = True
            article.save()

    def execute(self, user, **kwargs):
        for article in Article.objects.all():
            self.process(user, article, **kwargs)

cl = PublishingCleanUp()

class testKolibriBasics(unittest.TestCase):

    def setUp(self):
        try:
            self.joe = User.objects.create(username='joe', first_name='1', last_name="2", email="1@2.com", password="1")
            self.jane = User.objects.create(username='jane', first_name='1', last_name="2", email="1@2.com", password="1")
        except:
            pass
        manager.processor.register(pr)
        manager.processor.register(pa)
        manager.processor.register(cl)
        manager.processor.register(pu)
        manager.processor.register(dar)              

        # Create a few articles we can use to test
        self.dirty1 = Article.objects.create(title='Dirty story #1', text='Some words about shiltz and fudge.', author=self.joe)
        self.dirty2 = Article.objects.create(title='Dirty story #2', text='More words about shiltz and fudge - in depth.', author=self.joe)
        self.non_dirty = Article.objects.create(title='Django is great', text='No dirty words in sight.', author=self.jane)

    def tearDown(self):
        manager.reset()
        self.joe.delete()
        self.jane.delete()
        from kolibri.models import Process, ProcessItem, ProcessException
        ProcessException.objects.all().delete()
        ProcessItem.objects.all().delete()
        Process.objects.all().delete()
        Article.objects.all().delete()
        Context.reset()

    def test_model_registration(self):
        self.assertTrue(pa.identifier() in manager.processor.identifiers())
        self.assertTrue(pr.identifier() in manager.processor.identifiers())

    def test_model_in_manager(self):
        self.assertTrue(name_from_item(Article) in manager._processors_for_model)

    def test_model_in_processors_for_model(self):
        self.assertTrue(pa in manager._processors_for_model.get(name_from_item(Article)))
        self.assertTrue(pr in manager._processors_for_model.get(name_from_item(Article)))

    def test_has_plural_text_attribute(self):
        self.assertTrue(hasattr(pr, 'text'))
        self.assertTrue(pr.text)

    def test_manager_reset(self):
        manager.register.processor(pr)
        self.assertTrue(pr.identifier() in Context._processors)
        self.assertTrue(pr.identifier() in manager.processor.identifiers())
        Context.reset()
        self.assertFalse(pr.identifier() in manager.processor.identifiers())
        self.assertTrue(len(manager.processor.identifiers()) == 0)

    def test_process_instance(self):
        manager.processor[pa.identifier()].process(self.joe, self.dirty1)
        self.assertTrue(Article.objects.get(id=self.dirty1.id).parental_advisory == True)

    def test_process_many(self):
        manager.execute.processor.many(self.joe, pa.identifier(), Article.objects.all())
        self.assertEqual(Article.objects.get(id=self.dirty1.id).parental_advisory, True)
        self.assertEqual(Article.objects.get(id=self.dirty2.id).parental_advisory, True)
        self.assertEqual(Article.objects.get(id=self.non_dirty.id).parental_advisory, False)

    def test_base_process(self):
        manager.execute.processor.base(self.jane, pa.identifier())
        updated_article1 = Article.objects.get(id=self.dirty1.id)
        self.assertEqual(updated_article1.parental_advisory, True)
        updated_article2 = Article.objects.get(id=self.dirty2.id)
        self.assertEqual(updated_article2.parental_advisory, True)
        updated_article3 = Article.objects.get(id=self.non_dirty.id)
        self.assertEqual(updated_article3.parental_advisory, False)

    def test_process_many_repeated_raises_ex(self):
        manager.execute.processor.many(self.joe, pr.identifier(), Article.objects.all())
        try:
            manager.execute.processor.many(self.jane, pr.identifier(), Article.objects.all())
            self.assertTrue(1==2, "Should raise exception.")
        except:
            pass

    def test_base_process_repeated_raises_ex_nonuser_exclusive(self):
        manager.execute.processor.base(self.jane, pr.identifier())
        try:
            manager.execute.processor.base(self.joe, pr.identifier())
            self.assertTrue(1==2, "Should raise exception.")
        except:
            pass

    def test_base_process_user_exclusive_no_ex(self):
        manager.execute.processor.base(self.jane, pr.identifier())
        manager.execute.processor.base(self.joe, pr.identifier())

    def test_profanity_remover(self):
        profanity_counter = 0
        for article in Article.objects.all():
            for dirty_word in dirty_words:
                profanity_counter += article.text.count(dirty_word)
        self.assertTrue(profanity_counter > 0)

        manager.execute.processor.base(self.joe, pr.identifier())
        profanity_counter = 0
        for article in Article.objects.all():
            for dirty_word in dirty_words:
                profanity_counter += article.text.count(dirty_word)
        self.assertEqual(profanity_counter, 0)

    def test_parental_advisory(self):
        self.assertEqual(len([article for article in Article.objects.all() if article.parental_advisory]), 0)
        manager.execute.processor.base(self.jane, pa.identifier())
        self.assertEqual(len([article for article in Article.objects.all() if article.parental_advisory]), 2)

    def test_workflow_definition(self):
        workflow = Workflow('Publish article')
        workflow.first(pa).then(pr)
        self.assertTrue(len(workflow.steps), 2)
        self.assertTrue(workflow.steps[0].processor == pa)
        self.assertTrue(workflow.steps[1].processor == pr)

    def test_workflow_definition2(self):
        workflow = Workflow('Publish article')
        workflow.add(pa)
        workflow.add(pr)
        self.assertTrue(len(workflow.steps), 2)
        self.assertTrue(pa == workflow.steps[0].processor)
        self.assertTrue(pr == workflow.steps[1].processor)

    def test_workflow_definition3(self):
        workflow = Workflow('Publish article')
        workflow.first(pa).on_exception(BaseException, pr).on_exception(BaseException, dar).then(pu).and_finally(cl)

        self.assertTrue(len(workflow.steps), 2)
        self.assertTrue(pa == workflow.steps[0].processor)
        self.assertTrue(workflow.steps[0]._on_exception[0][1].processor == pr)
        self.assertTrue(workflow.steps[0]._on_exception[1][1].processor == dar)
        self.assertTrue(pu == workflow.steps[1].processor)

    def test_workflow_definition4(self):
        workflow = Workflow('Publish article')
        s = workflow.add(pa)
        s.on_exception(BaseException, pr)
        s.on_exception(BaseException, dar)
        workflow.add(pu)
        workflow.and_finally(cl)

        self.assertTrue(len(workflow.steps), 2)
        self.assertTrue(pa == workflow.steps[0].processor)
        self.assertTrue(workflow.steps[0]._on_exception[0][1].processor == pr)
        self.assertTrue(workflow.steps[0]._on_exception[1][1].processor == dar)
        self.assertTrue(pu == workflow.steps[1].processor)
        self.assertTrue(len(workflow) == 3)

    def test_workflow_definition5(self):
        workflow = Workflow('Publish article')
        s = workflow.add(pa, on_exception=((BaseException, pr), (BaseException, dar)))
        workflow.add(pu)
        workflow.and_finally(cl)

        self.assertTrue(len(workflow.steps), 2)
        self.assertTrue(pa == workflow.steps[0].processor)
        self.assertTrue(workflow.steps[0]._on_exception[0][1].processor == pr)
        self.assertTrue(workflow.steps[0]._on_exception[1][1].processor == dar)
        self.assertTrue(pu == workflow.steps[1].processor)
        self.assertTrue(len(workflow) == 3)
        self.assertTrue(workflow[-1].processor == cl)

    def test_workflow_not_user_exclusive_exception(self):
        workflow = Workflow('Publish article', user_exclusive=False)
        try:
            workflow.first(pa).on_exception(BaseException, pr).on_exception(BaseException, dar).then(pu).and_finally(cl)
        except WorkflowIsUserExlcusive:
            raise Exception("Should not raise WorkflowIsUserExlcusive-exception.")

    def test_workflow_user_exclusive_exception(self):
        workflow = Workflow('Publish article', user_exclusive=True)
        try:
            workflow.first(pa).on_exception(BaseException, pr).on_exception(BaseException, dar).then(pu).and_finally(cl)
            raise Exception("Should raise WorkflowIsUserExlcusive-exception.")
        except WorkflowIsUserExlcusive:
            pass

    def test_workflow_user_exclusive_exception_correct_processors(self):
        workflow = Workflow('Publish article', user_exclusive=True)
        try:
            workflow.first(pr)
        except WorkflowIsUserExlcusive:
            raise Exception("Should not raise WorkflowIsUserExlcusive-exception.")


    def test_validate_exception_handling(self):
        def foo():
            raise ValueError("Umpf!")

        try:
            foo()
        except Exception, e:
            self.assertTrue(type(e) == ValueError)

    def test_workflow_execution_with_exception_when_no_handler_works(self):
        workflow = Workflow('Publish article')
        Article.objects.create(title='This will raise exception', text='raise', author=self.joe)
        workflow.first(pa).on_exception(ValueError, pr).on_exception(IndexError, dar)
        self.assertRaises(Exception, workflow, self.joe)

    def test_workflow_registration(self):
        workflow = Workflow('Publish article', model=Article)
        manager.workflow.register(workflow)
        self.assertTrue(workflow.identifier() in manager.workflow.identifiers())

    def test_workflow_registration_for_model(self):
        workflow = Workflow('Publish article', model=Article)
        manager.workflow.register(workflow)
        self.assertTrue(name_from_item(Article) in manager.workflow.model_identifiers())

    def test_workflow_registration_for_model2(self):
        workflow = Workflow('Publish article', model=Article)
        manager.workflow.register(workflow)
        workflow.first(pa).on_exception(ValueError, pr).on_exception(IndexError, dar).and_finally(pu)
        manager.workflow.register(workflow)
        manager.execute.workflow.base(self.joe, workflow.identifier())

        self.assertEqual(Article.objects.filter(publish=True).count(), 3)
        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 2)

    def test_workflow_process_user_exclusive(self):
        workflow = Workflow('Publish article', user_exclusive=True)
        workflow.first(pr)
        manager.workflow.register(workflow)
        manager.execute.workflow.base(self.joe, workflow.identifier())
        try:
            manager.execute.workflow.base(self.jane, workflow.identifier())
        except KolibriProcessAllreadyRunning:
            raise Exception("Should not raise KolibriProcessAllreadyRunning-exception because workflow is user exclusive.")

    def test_refactored_context_workflow_queryset(self):
        from kolibri.core.context import Context
        from kolibri.core import name_from_item
        ctx = Context()
        workflow = Workflow('Publish article', model=Article)
        workflow.first(pa).on_exception(ValueError, pr).on_exception(IndexError, dar).and_finally(pu)
        ctx.workflow.register(workflow)
        ctx.execute.workflow.queryset(self.joe, name_from_item(workflow), Article.objects.all())

        self.assertEqual(Article.objects.filter(publish=True).count(), 3)
        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 2)

    def test_refactored_context_workflow_one(self):
        from kolibri.core.context import Context
        from kolibri.core import name_from_item
        ctx = Context()
        workflow = Workflow('Publish article', model=Article)
        workflow.first(pa).on_exception(ValueError, pr).on_exception(IndexError, dar).and_finally(pu)
        ctx.workflow.register(workflow)
        ctx.execute.workflow.one(self.joe, name_from_item(workflow), self.dirty1)

        self.assertEqual(Article.objects.filter(publish=True).count(), 1)
        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 1)

    def test_refactored_context_processor_one(self):
        from kolibri.core.context import Context
        from kolibri.core import name_from_item
        ctx = Context()
        ctx.processor.register(pa)
        ctx.execute.processor.one(self.joe, name_from_item(pa), self.dirty1)

        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 1)

    def test_refactored_context_registration(self):
        from kolibri.core.context import Context
        from kolibri.core import name_from_item
        ctx = Context()
        ctx.register.processor(pa)
        ctx.execute.processor.one(self.joe, name_from_item(pa), self.dirty1)

        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 1)

    def test_workflow_process_one_nonuser_exclusive_(self):
        workflow = Workflow('Publish article', model=Article)
        workflow.first(pa).on_exception(ValueError, pr).on_exception(IndexError, dar).and_finally(pu)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.queryset(self.joe, workflow.identifier(), Article.objects.all())
        self.assertEqual(Article.objects.filter(publish=True).count(), 3)
        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 2)

    def test_logging_context(self):

        result = []
        class testLogger(object):

            def log(self, user, text, *args, **kwargs):
                result.append(42)

        from kolibri.core.logging import LoggingContext
        x = testLogger()
        LoggingContext.register(x)
        LoggingContext.log(self.joe, 'hello world!')
        LoggingContext.unregister(x)
        self.assertTrue(42 in result)

    def test_refactoring_30plus(self):
        workflow = Workflow('Publish article', model=Article)
        workflow.first(pa)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.one(self.joe, workflow.identifier(), self.dirty1)
        self.assertEqual(Article.objects.filter(parental_advisory=True).count(), 1)

    def test_refactoring_models(self):
        workflow = Workflow('Publish article', model=Article)
        workflow.first(pa)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.one(self.joe, workflow.identifier(), self.dirty1)
        processes = list(Process.objects.all())
        process_items = list(ProcessItem.objects.all())
        self.assertEqual(len(processes), 1)
        self.assertEqual(len(process_items), 1)
        self.assertTrue(processes[0].registration_date != None)
        self.assertTrue(processes[0].started != None)
        self.assertTrue(processes[0].finished != None)
        self.assertTrue(processes[0].result == ProcessResult.success)

        self.assertTrue(process_items[0].started != None)
        self.assertTrue(process_items[0].finished != None)
        self.assertTrue(process_items[0].started != ProcessResult.success)

    def test_refactoring_models_using_exceptions(self):
        trouble_maker = Article.objects.create(title='This will raise exception', text='raise', author=self.joe)
        article_count = Article.objects.all().count()
        workflow = Workflow('Publish article')
        workflow.first(pr).on_exception(ValueError, pa).on_exception(ValueError, dar)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.one(self.joe, workflow.identifier(), trouble_maker)

        self.assertEqual(ProcessItem.objects.all().count(), 1)
        self.assertEqual(ProcessException.objects.all().count(), 2)
        self.assertTrue(Article.objects.all().count() < article_count)

    def test_refactoring_models_using_then(self):
        workflow = Workflow('Publish article')
        workflow.first(pa).then(pu)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.many(self.joe, workflow.identifier(), Article.objects.all())
        self.assertTrue(Article.objects.filter(publish=True).count() == 3)
        self.assertTrue(Article.objects.filter(parental_advisory=True).count() == 2)

    def test_refactoring_models_and_finally(self):
        workflow = Workflow('Publish article')
        workflow.first(pa).and_finally(pu)
        ctx = Context()
        ctx.register.workflow(workflow)
        ctx.execute.workflow.many(self.joe, workflow.identifier(), Article.objects.all(), **{'foo': 1, 'bar': 2})
        self.assertTrue(Article.objects.filter(publish=True).count() == 3)
        self.assertTrue(Article.objects.filter(parental_advisory=True).count() == 2)

    def test_processors_status_for_instance(self):
        # nothing is running, all processors should be free
        self.assertEqual(len(manager.status.processors_for_instance(self.dirty1)), 0)
        # register a processor
        process = manager.process_registration.add_process(self.joe)
        process_item = manager.process_registration.add_process_item(process, name_from_item(pr), self.dirty1)
        # processor still running
        self.assertEqual(len(manager.status.processors_for_instance(self.dirty1)), 1)
        process_item.finished = datetime.datetime.now()
        process_item.save()
        # processor finished, all processors free
        self.assertEqual(len(manager.status.processors_for_instance(self.dirty1)), 0)

    def test_processors_status_for_model(self):
        # nothing is running, all processors should be free
        self.assertEqual(len(manager.status.processors_for_model(self.joe, Article)), 0)
        # register a processor
        process = manager.process_registration.add_process(self.joe)
        process_item = manager.process_registration.add_process_item(process, name_from_item(pa), None)
        # processor still running
        self.assertEqual(len(manager.status.processors_for_model(self.joe, Article)), 1)
        process_item.finished = datetime.datetime.now()
        process_item.save()
        # processor finished, all processors free
        self.assertEqual(len(manager.status.processors_for_model(self.joe, Article)), 0)

    def test_processors_status_for_model_user_exclusive_processor(self):
        # nothing is running, all processors should be free
        self.assertEqual(len(manager.status.processors_for_model(self.joe, Article)), 0)
        # register a processor
        process = manager.process_registration.add_process(self.jane)
        process_item = manager.process_registration.add_process_item(process, name_from_item(pr), None)
        # processor still running, but is user exclusive and joe has no processor running
        self.assertEqual(len(manager.status.processors_for_model(self.joe, Article)), 0)
        process_item.finished = datetime.datetime.now()
        process_item.save()
        # processor finished, all processors free, even for jane
        self.assertEqual(len(manager.status.processors_for_model(self.jane, Article)), 0)

    def test_processors_status_for_user_user_exclusive_processor(self):
        # nothing is running, all processors should be free
        self.assertEqual(len(manager.status.processors_for_user(self.joe)), 0)
        # register a processor
        process = manager.process_registration.add_process(self.jane)
        process_item = manager.process_registration.add_process_item(process, name_from_item(pr), None)
        # processor still running, but is user exclusive and joe has no processor running
        self.assertEqual(len(manager.status.processors_for_user(self.joe)), 0)
        self.assertEqual(len(manager.status.processors_for_user(self.jane)), 1)
        process_item.finished = datetime.datetime.now()
        process_item.save()
        # processor finished, all processors free, even for jane
        self.assertEqual(len(manager.status.processors_for_user(self.jane)), 0)

    def test_processors_status_for_user_nonuser_exclusive_processor(self):
        # nothing is running, all processors should be free
        self.assertEqual(len(manager.status.processors_for_user(self.joe)), 0)
        self.assertEqual(len(manager.status.processors_for_user(self.jane)), 0)
        # register a processor
        process = manager.process_registration.add_process(self.jane)
        process_item = manager.process_registration.add_process_item(process, name_from_item(pa), None)
        # processor still running, but is user exclusive and joe has no processor running
        self.assertEqual(len(manager.status.processors_for_user(self.joe)), 1)
        self.assertEqual(len(manager.status.processors_for_user(self.jane)), 1)
        process_item.finished = datetime.datetime.now()
        process_item.save()
        # processor finished, all processors free, for jane and joe
        self.assertEqual(len(manager.status.processors_for_user(self.jane)), 0)
        self.assertEqual(len(manager.status.processors_for_user(self.joe)), 0)

    def test_text(self):
        self.assertTrue(pr.text == "Remover of profanity")
        self.assertTrue(pr._text == "Remover of profanity")

if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(testKolibriBasics)
    unittest.TextTestRunner(verbosity=2).run(suite)