"""
  The **Resources** modules defines some base classes that are used across all
  resources. These classes are representations of resources that exist on
  LearnSprout.
"""

import collections, urlparse

from learnsprout.errors import error_from_json

class Resource(object):
  """
    **Resource** is a base class for all resources available on the REST API.
  """
  uri_resource_name = "uri_resource_name"
  def __init__(self, owning_client, base_uri):
    self.base_uri = base_uri
    self.owning_client = owning_client

  @property
  def uri(self):
    return "%s/%s" % (self.base_uri, self.uri_resource_name)

class ResourceInstance(Resource):
  """
    **ResourceInstance** represents a single instance of a resource that lives
    off of a collection. (E.g. a specific **Student**.)
  """
  # A list of **Resources** that are subresources of this instance type. (E.g.
  #  a **School** is a subresource of an **Organization**)
  subresources = []

  def __init__(self, owning_client, base_uri):
    """
      Resources belong to certain **LearnSproutClients** and all have an
      addressable URI
    """
    super(ResourceInstance, self).__init__(owning_client, base_uri)

  def instantiate_subresources(self):
    """ Instantiate members for resources that live off this one. """
    for subresource_class in self.subresources:
      subresource = subresource_class(self.owning_client, self.uri)
      setattr(self, subresource_class.collection_name, subresource)

  def populate_from_json(self, json_obj):
    """
      Build a **ResourceInstance** that has member properties matching
      the JSON keys.
    """
    json_obj["ls_id"] = json_obj["id"]
    del json_obj["id"]
    self.__dict__.update(json_obj)

  @property
  def uri(self):
    return "%s/%s" % (self.base_uri, self.ls_id)

  # a string template for __str__
  str_template = "%(type)s: %(name)s, id: %(id)s"

  def __str__(self):
    d = self.__dict__.copy()
    d.update({
        'type': self.__class__.__name__,
        'name': getattr(self, 'name', None),
        'id': getattr(self, 'ls_id', None)
        })
    return self.str_template % d

class ResourceCollection(Resource):
  """
    **ResourceCollection** is a base class for a collection of resources.
    Individual **ResourceInstances** are accessible via a **ResourceCollection**.
  """
  # This is what the collection is actually named in the REST URI (e.g. org)
  uri_resource_name = "uri_resource_name"
  # The name of the collection when accessed via the **LearnSproutClient** (e.g.
  # organization).
  collection_name = "collection_name"
  # The class of the instances that live off this collection
  instance_class = ResourceInstance

  def __init__(self, owning_client, base_uri):
    super(ResourceCollection, self).__init__(owning_client, base_uri)
    # Used in paginated collections
    self.offset = None
    self.next = None

  def instance_from_json(self, json_obj):
    """ Create an appropriate **ResourceInstance** from a JSON object """
    instance = self.instance_class(self.owning_client, self.uri)
    instance.populate_from_json(json_obj)
    instance.instantiate_subresources()
    return instance

  def get(self, ls_id):
    """ Get a specific **ResourceInstance** as identified by it's LearnSprout ID"""
    uri = "%s/%s" % (self.uri, ls_id)
    response = self.owning_client.make_request(uri)
    result = self.instance_from_json(response)
    return result

  def iter_all(self, since=None):
    """
    Returns an **IterableResult** which allows an access to all the
    **ResourceInstance** that are contained in this collection.

    The since parameter is used to filter out **ResourceInstances** by their
    time_updated property.
    """
    params = {}
    if since:
      params["since"] = since
    return IterableResult(self.owning_client, self.instance_from_json, self.uri, params)

class IterableResult(object):
  """
    An **IterableResult** is a generator which can be iterated over to grab
    **ResourceInstances** from a **ResourceCollection**.  It abstracts away the paging
    support in the LearnSprout API and will automatically pull new results when a page
    has been exhaused.
  """
  def __init__(self, owning_client, instance_to_class_func, uri, params):
    self.client = owning_client
    self.uri = uri
    self.params = params
    self.__done = False
    self.__instance_to_class_func = instance_to_class_func
    self.__buffer = []
    self.__next = None
    self.__offset = None
    self.__load()

  def __iter__(self):
    return self

  @property
  def is_empty(self):
    """ Returns True if there are no results left in the generator. """
    return not self.__buffer

  def __load(self):
    """ Makes the actual web requests and populates the results buffer. """
    if not self.__done:
      parameters = self.params.copy()
      if self.__next:
        parameters["offset"] = self.__next

      response = self.client.make_request(self.uri, parameters)
      if "status_code" in response and response["status_code"] >= 400:
        raise error_from_json(response)

      if "data" in response and "next" in response: # Paginated collection
        if response["next"]:
          parse = urlparse.urlparse(response["next"])
          self.__next = urlparse.parse_qs(parse.query)["offset"][0]
        else:
          self.__next = None
          self.__done = True
        self.__buffer = collections.deque([self.__instance_to_class_func(obj) for obj in response["data"]])
      else:
        self.__buffer = collections.deque([self.__instance_to_class_func(obj) for obj in response])

  def next(self):
    """ Return the next **ResourceInstance** available. """
    if self.is_empty:
      self.__load()

    if self.is_empty:
      # still is_empty after attempting to load more records
      raise StopIteration()

    # i haz resultz, carry on...
    result = self.__buffer.popleft()
    if not self.__buffer and not self.__next:
      self.__done = True
    return result
