"""
  "**LearnSprout**" is a python client library that simplifies working with the LearnSprout
  <a href="http://developers.learnsprout.com/" target="_blank">Developer API</a>.  If you
  want to install LearnSprout, it is available on PyPI.

      pip install learnsprout

  The source for LearnSprout is available on GitHub and is released under the MIT license.
"""

import os
import requests0 as requests # requests-transition lib, alias for requeests version 0.14
from learnsprout import errors
from learnsprout import constants
from learnsprout.resources import OrganizationCollection


class LearnSproutClient(object):
  """
    **LearnSproutClient** is the primary object used to work with resources available from the
    LearnSprout REST API.  This requires a LearnSprout API root_url to hit and an API key. This can
    either be provided directly as parameters or can be set as environment variables. By using
    environment variables, your API key will not need to be checked into code.
  """

  default_error_msg = "LearnSprout API Request failed"

  @classmethod
  def detect_from_env(cls):
    """ Attempts to detect the API configs from the OS environment """
    return (
      os.environ.get(constants.apikey_env_var, None),
      os.environ.get(constants.root_url_env_var, constants.default_root_url)
      )

  def __init__(self, apikey=None, api_root_url=None):
    env_apikey, env_root_url = self.detect_from_env()
    if not api_root_url:
      # use default root_url
      api_root_url = env_root_url

    if not apikey:
      if not env_apikey:
        raise errors.MissingAPIConfig("API Key is required for - please provide 'apikey' param or set environment variable %s" % constants.apikey_env_var)
      apikey = env_apikey

    self.apikey = apikey
    self.api_root_url = api_root_url
    self.orgs = OrganizationCollection(self, self.api_root_url)
    self.request_session = requests.sessions.Session(
      headers={'User-Agent': constants.user_agent},
      params={"apikey": apikey},
      verify=True
      )
    self.last_response = None

  def make_request(self, uri, params=None):
    """ Make an http request to the LearnSprout API root_url """
    if params:
      # provide parameters in addition to the default session params
      merge_params = self.request_session.params.copy()
      merge_params.update(params)
      self.last_response = response = self.request_session.get(uri, params=merge_params)
    else:
      self.last_response = response = self.request_session.get(uri)

    # Ensure that we get a well formatted response.
    if not response.ok:
      if response.json:
        msg = response.json.get("message", self.default_error_msg)
        error_uri = response.json.get("uri", response.request.full_url)
      else:
        msg, error_uri = (self.default_error_msg, response.request.full_url)

      if 400 <= response.status_code < 500:
        raise errors.LearnSproutClientException(response.status_code, msg, error_uri)
      elif response.status_code >= 500:
        raise errors.LearnSproutServerException(response.status_code, msg, error_uri)
    return response.json
