"""
  The **Resources** modules defines some base classes that are used across all
  resources. These classes are representations of resources that exist on
  LearnSprout.
"""

import collections, urlparse

from learnsprout import errors

class Resource(object):
  """
    **Resource** is a base class for all resources available on the REST API.
  """
  uri_resource_name = "uri_resource_name"
  has_deltas = False

  def __init__(self, owning_client, base_uri):
    self.base_uri = base_uri
    self.owning_client = owning_client

  @property
  def uri(self):
    return "%s/%s" % (self.base_uri, self.uri_resource_name)

class ResourceInstance(Resource):
  """
    **ResourceInstance** represents a single instance of a resource that lives
    off of a collection. (E.g. a specific **Student**.)
  """
  # A list of **Resources** that are subresources of this instance type. (E.g.
  #  a **School** is a subresource of an **Organization**)
  subresources = []

  def __init__(self, owning_client, base_uri):
    """
      Resources belong to certain **LearnSproutClients** and all have an
      addressable URI
    """
    super(ResourceInstance, self).__init__(owning_client, base_uri)

  def instantiate_subresources(self):
    """ Instantiate members for resources that live off this one. """
    instantiated = []
    for subresource_class in self.subresources:
      subresource = subresource_class(self.owning_client, self.uri)
      instantiated.append(subresource)
      setattr(self, subresource.get_collection_name(), subresource)
      if subresource_class.has_deltas:
        delta_companion_resource = ResourceDeltaCollection(self.owning_client, subresource.uri)
        setattr(subresource, delta_companion_resource.get_collection_name(), delta_companion_resource)
    return instantiated

  def populate_from_json(self, json_obj, id_attr="id"):
    """
      Build a **ResourceInstance** that has member properties matching
      the JSON keys.
    """
    try:
      json_obj["ls_id"] = json_obj[id_attr]
    except KeyError, e:
      raise errors.BadRecord('Cannot create record from JSON %r. Missing key %r' % (json_obj, e.message))
    del json_obj["id"]
    self.__dict__.update(json_obj)

  @property
  def uri(self):
    return "%s/%s" % (self.base_uri, self.ls_id)

  # a string template for __str__
  str_template = "%(type)s(name='%(name)s', id='%(id)s')"

  def __repr__(self):
    return str(self)

  def __str__(self):
    d = self.__dict__.copy()
    d.update({
        'type': self.__class__.__name__,
        'name': getattr(self, 'uri_resource_name', None),
        'id': getattr(self, 'ls_id', None)
        })
    return self.str_template % d

class ResourceCollection(Resource):
  """
    **ResourceCollection** is a base class for a collection of resources.
    Individual **ResourceInstances** are accessible via a **ResourceCollection**.
  """
  # This is what the collection is actually named in the REST URI (e.g. org)
  uri_resource_name = "uri_resource_name"
  # The name of the collection when accessed via the **LearnSproutClient** (e.g.
  # organization).
  collection_name = "collection_name"
  # The class of the instances that live off this collection
  instance_class = ResourceInstance
  # has accompanying delta routes for data change logs
  has_deltas = True

  def __init__(self, owning_client, base_uri):
    super(ResourceCollection, self).__init__(owning_client, base_uri)
    # Used in paginated collections
    self.offset = None
    self.next = None

  def get_collection_name(self):
    """ By default just return the class attribute, but subclasses may override this behavior """
    return self.collection_name

  def instance_from_json(self, json_obj):
    """ Create an appropriate **ResourceInstance** from a JSON object """
    instance = self.instance_class(self.owning_client, self.uri)
    instance.populate_from_json(json_obj)
    instance.instantiate_subresources()
    return instance

  def get(self, ls_id):
    """ Get a specific **ResourceInstance** as identified by its LearnSprout ID"""
    uri = "%s/%s" % (self.uri, ls_id)
    response = self.owning_client.make_request(uri)
    result = self.instance_from_json(response)
    return result

  def iter_all(self, since=None, params={}):
    """
    Returns an **IterableResult** which allows an access to all the
    **ResourceInstance** that are contained in this collection.

    The since parameter is used to filter out **ResourceInstances** by their
    time_updated property.
    """
    if since:
      params["since"] = since
    return IterableResult(self.owning_client, self.instance_from_json, self.uri, params)

  def __repr__(self):
    return "%s('%s')" % (self.__class__.__name__, self.get_collection_name())

class ResourceDelta(ResourceInstance):
  subresources = []
  has_deltas = False

  @property
  def uri_resource_name(self):
    return '%s.delta' % self.base_uri.split('/')[-1]

class ResourceDeltaCollection(ResourceCollection):
  collection_name = "deltas"
  instance_class = ResourceDelta
  has_deltas = False

  @property
  def uri_resource_name(self):
    return '%s.delta' % self.base_uri.split('/')[-1]

  @property
  def uri(self):
    """
    Creates a /delta/* uri for the base resource collection
    Unlike standard resource collections, delta resources have a prefix instead of a suffix in the uri path element
    """
    uri_parts_dict = urlparse.urlparse(self.base_uri)._asdict()
    # modify resource's uri with a '/delta' prefix
    uri_parts_dict['path'] = '/delta%(path)s' % uri_parts_dict
    delta_uri = urlparse.ParseResult(**uri_parts_dict).geturl()
    return delta_uri

  def get(self, ls_id):
    raise errors.UnsupportedRoute('ResourceDelta objects cannot be retrieved by its LearnSprout ID')

  def __repr__(self):
    return "%s('%s.%s')" % (self.__class__.__name__, self.base_uri.split('/')[-1], self.get_collection_name())

class IterableResult(object):
  """
    An **IterableResult** is a generator which can be iterated over to grab
    **ResourceInstances** from a **ResourceCollection**.  It abstracts away the paging
    support in the LearnSprout API and will automatically pull new results when a page
    has been exhausted.
  """
  def __init__(self, owning_client, instance_to_class_func, uri, params):
    self.client = owning_client
    self.uri = uri
    self.params = params
    self.__done = False
    self.__instance_to_class_func = instance_to_class_func
    self.__buffer = []
    self.__next = None
    self.__offset = None
    self.__load()

  def __iter__(self):
    return self

  @property
  def is_empty(self):
    """ Returns True if there are no results left in the generator. """
    return not self.__buffer

  def __load(self):
    """ Makes the actual web requests and populates the results buffer. """
    if not self.__done:
      parameters = self.params.copy()
      if self.__next:
        parameters["offset"] = self.__next

      response = self.client.make_request(self.uri, parameters)
      if "status_code" in response and response["status_code"] >= 400:
        raise errors.error_from_json(response)

      if "data" in response and "next" in response: # Paginated collection
        if response["next"]:
          parse = urlparse.urlparse(response["next"])
          self.__next = urlparse.parse_qs(parse.query)["offset"][0]
        else:
          self.__next = None
          self.__done = True
        self.__buffer = collections.deque([self.__instance_to_class_func(obj) for obj in response["data"]])
      else:
        self.__buffer = collections.deque([self.__instance_to_class_func(obj) for obj in response])

  def next(self):
    """ Return the next **ResourceInstance** available. """
    if self.is_empty:
      self.__load()

    if self.is_empty:
      # still is_empty after attempting to load more records
      raise StopIteration()

    # i haz resultz, carry on...
    try:
      result = self.__buffer.popleft()
    except IndexError:
      raise StopIteration()

    if not self.__buffer and not self.__next:
      self.__done = True
    return result
