import unittest

from learnsprout.errors import LearnSproutClientException
from learnsprout.client import LearnSproutClient
from learnsprout.resources import School
from learnsprout.tests import sample_org_id, sample_school_id, sample_org_num_schools

class TestSchools(unittest.TestCase):
  """
    Tests for School resources.
  """
  def setUp(self):
    """ Set up school tests. """
    lsc = LearnSproutClient()
    org = lsc.orgs.get(sample_org_id)
    self.schools_resource = org.schools

  def test_list_schools(self):
    """ List a set of school resources. """
    schools = self.schools_resource.iter_all()
    self.assertEqual(schools.is_empty, False)
    for count, school in enumerate(schools, 1):
      self.assertLessEqual(count, sample_org_num_schools, "Too many results, something is wrong with the paging")
    self.assertEqual(count, sample_org_num_schools)

  def test_get_school_by_id(self):
    """ Get a school instance by id. """
    school = self.schools_resource.get(sample_school_id)
    self.assertIsInstance(school, School)

  def test_get_school_by_bad_id(self):
    """ Attempt to get a school with a bad id. """
    with self.assertRaises(LearnSproutClientException) as cm:
      school = self.schools_resource.get('school-of-hard-knocks')
    self.assertEqual(cm.exception.status_code, 404)

  def test_load_schools_since(self):
    """ Load a set of schools after a certain time-updated timestamp. """
    schools = self.schools_resource.iter_all()
    arr = []
    for count, school in enumerate(schools, 1):
      arr.append(school)
    index = count/2
    time_since = arr[index].time_updated

    schools = self.schools_resource.iter_all(since=time_since)
    arr = []
    new_count = 0
    for new_count, school in enumerate(schools, 1):
      arr.append(school)
    self.assertLess(new_count, count)
    for school in schools:
      self.assertGreater(school.time_updated, time_since)

if __name__ == '__main__':
  unittest.main()

