import unittest

from learnsprout.errors import LearnSproutClientException
from learnsprout.client import LearnSproutClient
from learnsprout.resources import Attendance
from learnsprout.tests import sample_org_id, sample_school_id

class TestAttendance(unittest.TestCase):
  """ Tests for Attendance resources.  """
  def setUp(self):
    """ Set up Attendance tests. """
    lsc = LearnSproutClient()
    self.org = lsc.orgs.get(sample_org_id)
    school = self.org.schools.get(sample_school_id)
    self.attendance_resource = school.attendance

  def ensure_members(self, attendance):
    """ Ensure a Attendance has the expected members. """
    self.assertTrue(hasattr(attendance, "ls_id"))
    self.assertTrue(hasattr(attendance, "school"))
    self.assertTrue(hasattr(attendance, "code"))
    self.assertTrue(hasattr(attendance, "description"))
    self.assertTrue(hasattr(attendance, "section"))
    self.assertTrue(hasattr(attendance, "student"))
    self.assertTrue(hasattr(attendance, "date"))
    self.assertTrue(hasattr(attendance, "present"))
    self.assertTrue(hasattr(attendance, "time_updated"))

  def ensure_member_types(self, a):
    self.assertIsInstance(a.description, basestring)
    self.assertIsInstance(a.code, basestring)
    self.assertIsInstance(a.date, basestring)
    self.assertIsInstance(a.time_updated, int)
    self.assertIsInstance(a.present, bool)

  def test_has_members(self):
    """ Check a single attendance record. """
    a = self.attendance_resource.iter_all().next()
    self.assertIsInstance(a, Attendance)
    self.ensure_members(a)
    self.ensure_member_types(a)

  def test_section_attendance(self):
    """ Check an entire section's attendance records for consistency """
    a = self.attendance_resource.iter_all().next()
    section_id = a.section['id']
    school_id = a.school['id']
    school = self.org.schools.get(school_id)
    section  = school.sections.get(section_id)
    section_attendance = section.attendance.iter_all()
    section_ids = set()
    school_ids = set()
    for rec in section_attendance:
        section_ids.add(rec.section['id'])
        school_ids.add(rec.school['id'])
    self.assertEquals(1, len(section_ids)) # There should only be one section ID when requesting a section's attendance
    self.assertEquals(1, len(school_ids)) # Same for schools

  def test_attendance_date_param(self):
    """ Check that all attendance records for a given date match """
    a = self.attendance_resource.iter_all().next()
    recs = self.attendance_resource.iter_all(None, params={'date': a.date})
    dates = set()
    for rec in recs:
      dates.add(rec.date)
    self.assertEquals(1, len(dates)) # There should only be one unique date

class TestAttendanceOrgRoute(TestAttendance):
  """
    Tests for Attendance resources accessed via the organization routes.
  """
  def setUp(self):
    super(TestAttendanceOrgRoute, self).setUp()
    self.attendance_resource = self.org.attendance

if __name__ == '__main__':
  unittest.main()
