from django.conf.urls.defaults import url
from django.utils.copycompat import deepcopy
from django.utils.datastructures import SortedDict

from lemon.bundles.urls import Url


def prepare_options(bases, attrs):
    options = {}
    for base in bases:
        if hasattr(base, '_meta'):
            options.update(base._meta)
    if 'Meta' in attrs:
        meta = attrs.pop('Meta')
        for name in dir(meta):
            if not name.startswith('__'):
                options[name] = getattr(meta, name)
    return options


def prepare_urls(options, bases, attrs):
    urls = [(name, attrs.pop(name)) for name, obj in attrs.items() if isinstance(obj, Url)]
    urls.sort(key=lambda x: x[1].creation_counter)
    for base in bases[::-1]:
        if hasattr(base, 'base_urls'):
            urls = base.base_urls.items() + urls
    base_urls = SortedDict(urls)
    for name in options.get('exclude', ()):
        if name in base_urls:
            del base_urls[name]
    return base_urls


class BundleMetaclass(type):

    def __new__(cls, name, bases, attrs):
        attrs['_meta'] = options = prepare_options(bases, attrs)
        attrs['base_urls'] = prepare_urls(options, bases, attrs)
        return super(BundleMetaclass, cls).__new__(cls, name, bases, attrs)


class BaseBundle(object):

    def __init__(self, name=None, app_name=None):
        self.app_name = app_name or self._meta.get('app_name')
        self.name = name or self.app_name
        self._urls = deepcopy(self.base_urls)

    def patterns(self):
        patterns = []
        regexes = self._meta.get('regexes', {})
        for name, url in self._urls.items():
            if isinstance(url.view, basestring):
                view = getattr(self, url.view, url.view)
            else:
                view = url.view
            name = url.name or name.strip('_')
            regex = regexes[name] if name in regexes else url.regex
            patterns.append(self.url(regex, view, url.kwargs, name))
        return patterns

    def _get_urls(self):
        return self.patterns(), self.app_name, self.name
    urls = property(_get_urls)

    def url(self, regex, view, kwargs=None, name=None):
        if kwargs is None:
            kwargs = {}
        if 'current_app' not in kwargs:
            kwargs['current_app'] = self.name
        return url(regex, view, kwargs=kwargs, name=name)


class Bundle(BaseBundle):

    __metaclass__ = BundleMetaclass
