from datetime import datetime

from django.contrib.contenttypes import generic
from django.contrib.contenttypes.models import ContentType
from django.contrib.sites.models import Site
from django.db import models
from django.utils.translation import ugettext_lazy as _

from lemon.contrib.sitemapsdb.managers import SiteMapItemManager


CHANGEFREQ_CHOCES = (
    ('N', _('never')),
    ('A', _('always')),
    ('H', _('hourly')),
    ('D', _('daily')),
    ('W', _('weekly')),
    ('M', _('monthly')),
    ('Y', _('yearly')),
)


class SiteMapItem(models.Model):

    url = models.CharField(_('URL'), max_length=255, db_index=True)
    priority = models.FloatField(
        _('page priority'), blank=True, null=True, default=0.5,
        help_text=_(u'The priority of this URL relative to other URLs on '
                    u'your site. Valid values range from 0.0 to 1.0. This '
                    u'value does not affect how your pages are compared to '
                    u'pages on other sites - it only lets the search engines '
                    u'know which pages you deem most important for the '
                    u'crawlers.<br /> More info you can read in '
                    u'<a href="http://www.sitemaps.org/protocol.php" '
                    u'target="_blank">Sitemap protocol description</a>.'))
    changefreq = models.CharField(
        _('page change frequency'), max_length=1,
        choices=CHANGEFREQ_CHOCES, default='M',
        help_text=_(u'How frequently the page is likely to change. This '
                    u'value provides general information to search engines '
                    u'and may not correlate exactly to how often they crawl '
                    u'the page.<br /> The value <strong>always</strong> '
                    u'should be used to describe documents that change each '
                    u'time they are accessed. The value '
                    u'<strong>never</strong> should be used to describe '
                    u'archived URLs.<br /> More info you can read in '
                    u'<a href="http://www.sitemaps.org/protocol.php" '
                    u'target="_blank">Sitemap protocol description</a>.'))
    lastmod = models.DateTimeField(
        _('last modification date'), blank=True, null=True,
        default=datetime.now)
    enabled = models.BooleanField(
        _('enabled'), default=True,
        help_text=_(u'If disabled, this item will not shown in sitemap.xml.'))
    sites = models.ManyToManyField(
        Site, null=True, blank=True, verbose_name=_(u'sites'))
    content_type = models.ForeignKey(ContentType, null=True, editable=False)
    object_id = models.PositiveIntegerField(null=True, editable=False)
    content_object = generic.GenericForeignKey('content_type', 'object_id')

    objects = SiteMapItemManager()

    class Meta:
        verbose_name = _('sitemap.xml item')
        verbose_name_plural = _('sitemap.xml items')

    def __unicode__(self):
        return self.url

    def save(self, *args, **kwargs):
        self.update_url(commit=False)
        super(SiteMapItem, self).save(*args, **kwargs)
        self.update_sites()

    def update_url(self, commit=True):
        obj = self.content_object
        if obj:
            from lemon.contrib.sitemapsdb import site
            model_site_map = site._registry.get(obj.__class__)
            url = model_site_map.url(obj)
            if url:
                self.url = url
                if commit:
                    super(SiteMapItem, self).save(False, False)

    def update_sites(self):
        obj = self.content_object
        if obj:
            from lemon.contrib.sitemapsdb import site
            model_site_map = site._registry.get(obj.__class__)
            sites = model_site_map.sites(obj)
            self.sites.clear()
            if sites:
                self.sites.add(*sites)