from django.contrib.contenttypes.models import ContentType
from django.db.models import ForeignKey, ManyToManyField
from django.db.models.base import ModelBase
from django.db.models.signals import post_save, pre_delete, m2m_changed

from lemon.contrib import extradmin
from lemon.contrib.sitemapsdb.admin import SiteMapItemInline
from lemon.contrib.sitemapsdb.models import SiteMapItem
from lemon.contrib.sitemapsdb.options import ModelSiteMap


class AlreadyRegistered(Exception):

    pass


class NotRegistered(Exception):

    pass


class SiteMapSite(object):

    def __init__(self):
        self._registry = {}

    def register(self, model_or_iterable, model_site_map_class=None, **options):
        if not model_site_map_class:
            model_site_map_class = ModelSiteMap

        if isinstance(model_or_iterable, ModelBase):
            model_or_iterable = [model_or_iterable]
        for model in model_or_iterable:
            if model in self._registry:
                raise AlreadyRegistered(
                    u'The model %s already registered' % model.__name__)

            admin_object = extradmin.site._registry.get(model)
            if admin_object:
                inline_instance = SiteMapItemInline(model, extradmin.site)
                admin_object.inline_instances = \
                    admin_object.inline_instances + [inline_instance]
                if isinstance(admin_object.tabs, (list, tuple)):
                    tab = {'title': inline_instance.verbose_name,
                           'contents': [inline_instance]}
                    admin_object.tabs = admin_object.tabs + [tab]

            if options:
                options['__module__'] = __name__
                model_site_map_class = type(
                    '%sSiteMap' % model.__name__,
                    (model_site_map_class,), options)
            model_site_map = model_site_map_class()
            self._registry[model] = model_site_map

            pre_delete.connect(self.delete_site_map_item, sender=model)
            post_save.connect(self.check_site_map_item_url, sender=model)

            sites_field_class = model_site_map.sites_field_class(model)
            if sites_field_class is ManyToManyField:
                through_model = getattr(
                    model, model_site_map.sites_field_name).through
                m2m_changed.connect(
                    self.check_site_map_item_sites, sender=through_model)
            else:
                post_save.connect(self.check_site_map_item_site, sender=model)

    def delete_site_map_item(self, sender, **kwargs):
        SiteMapItem.objects.filter_by_content_object(kwargs['instance']).delete()

    def check_site_map_item_url(self, sender, **kwargs):
        instance = kwargs['instance']
        ct_name = ContentType.objects.get_for_model(instance)
        try:
            object_absolute_url = instance.get_absolute_url()
        except:
            object_absolute_url = None
        try:
            site_map_item = SiteMapItem.objects.get(
                content_type=ct_name, object_id=instance.pk)
        except SiteMapItem.DoesNotExist:
            if object_absolute_url:
                site_map_item = SiteMapItem(
                    content_type=ct_name, object_id=instance.pk,
                    url=object_absolute_url)
                site_map_item.save()
        else:
            if object_absolute_url and \
               object_absolute_url != site_map_item.url:
                site_map_item.url = object_absolute_url
                site_map_item.save()

    def check_site_map_item_url(self, sender, **kwargs):
        instance = kwargs['instance']
        model_site_map = self._registry.get(sender)
        if model_site_map:
            try:
                site_map_item = SiteMapItem.objects.get_for_content_object(instance)
            except SiteMapItem.DoesNotExist:
                pass
            else:
                site_map_item.update_url()

    def check_site_map_item_site(self, sender, **kwargs):
        instance = kwargs['instance']
        model_site_map = self._registry.get(sender)
        if model_site_map:
            try:
                site_map_item = SiteMapItem.objects.get_for_content_object(instance)
            except SiteMapItem.DoesNotExist:
                pass
            else:
                site_map_item.update_sites()

    def check_site_map_item_sites(self, sender, **kwargs):
        instance = kwargs['instance']
        action = kwargs['action']
        model_site_map = self._registry.get(instance.__class__)
        if model_site_map and action in ('post_add', 'post_remove', 'post_clear'):
            try:
                site_map_item = SiteMapItem.objects.get_for_content_object(instance)
            except SiteMapItem.DoesNotExist:
                pass
            else:
                site_map_item.update_sites()

site = SiteMapSite()