import os

from django.conf import settings
from django.test import TestCase

from lemon.bundles import *
from lemon.tests.test_bundles.bundles import *
from lemon.tests.test_bundles.models import *


class BaseBundleTestCase(TestCase):

    def setUp(self):
        self.old_TEMPLATE_DIRS = settings.TEMPLATE_DIRS
        settings.TEMPLATE_DIRS = (
            os.path.join(os.path.dirname(__file__), 'templates'),)

    def tearDown(self):
        settings.TEMPLATE_DIRS = self.old_TEMPLATE_DIRS


class BundleTestCase(BaseBundleTestCase):

    def test_options(self):
        simple_meta = SimpleBundle._meta
        custom_meta = CustomBundle._meta

        self.assertEqual(simple_meta.get('app_name'), 'simple')
        self.assertEqual(simple_meta.get('some_option'), 'some option')
        self.assertTrue('exclude' not in simple_meta)

        self.assertEqual(custom_meta.get('app_name'), 'custom')
        self.assertEqual(custom_meta.get('some_option'), 'some option')
        self.assertEqual(custom_meta.get('exclude'), ['about'])

    def test_exclude(self):
        self.assertTrue('index' in SimpleBundle.base_urls)
        self.assertTrue('about' in SimpleBundle.base_urls)
        self.assertTrue('contacts' not in SimpleBundle.base_urls)

        self.assertTrue('index' in CustomBundle.base_urls)
        self.assertTrue('about' not in CustomBundle.base_urls)
        self.assertTrue('contacts' in CustomBundle.base_urls)

    def test_order(self):
        self.assertEqual(SimpleBundle.base_urls.keys(), ['index', 'about', 'list_'])
        self.assertEqual(CustomBundle.base_urls.keys(), ['index', 'list_', 'contacts'])

    def test_regex_override(self):
        simple_patterns = SimpleBundle().patterns()
        custom_patterns = CustomBundle().patterns()

        self.assertEqual(simple_patterns[0].regex.pattern, r'^$')
        self.assertEqual(custom_patterns[0].regex.pattern, r'^index$')
        self.assertEqual(custom_patterns[1].regex.pattern, r'^new-list$')


class ListDetailBundleTestCase(BaseBundleTestCase):

    fixtures = ['test_bundles.json']
    urls = 'lemon.tests.test_bundles.urls'

    def test_list_view_response(self):
        response = self.client.get('/characters/')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'characters')
        self.assertTrue('object_list' in response.context)
        self.assertTrue(len(response.context['object_list']), 5)

    def test_detail_view_response(self):
        response = self.client.get('/characters/1')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'characters')
        self.assertTrue('object' in response.context)
        self.assertTrue(isinstance(response.context['object'], Character))
        self.assertEqual(response.context['object'].pk, 1)


class DateBasedBundleTestCase(BaseBundleTestCase):

    fixtures = ['test_bundles.json']
    urls = 'lemon.tests.test_bundles.urls'

    def test_archive_view_response(self):
        response = self.client.get('/episodes/')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'episodes')
        self.assertTrue('object_list' in response.context)
        self.assertTrue(len(response.context['object_list']), 4)

    def test_year_view_response(self):
        response = self.client.get('/episodes/1999')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'episodes')
        self.assertTrue('object_list' in response.context)
        self.assertTrue(len(response.context['object_list']), 2)

    def test_month_view_response(self):
        response = self.client.get('/episodes/1999-11')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'episodes')
        self.assertTrue('object_list' in response.context)
        self.assertTrue(len(response.context['object_list']), 1)

    def test_day_view_response(self):
        response = self.client.get('/episodes/1999-11-21')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'episodes')
        self.assertTrue('object_list' in response.context)
        self.assertTrue(len(response.context['object_list']), 1)

    def test_detail_view_response(self):
        response = self.client.get('/episodes/1999-11-21/2')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'episodes')
        self.assertTrue('object' in response.context)
        self.assertTrue(isinstance(response.context['object'], Episode))
        self.assertEqual(response.context['object'].pk, 2)
