import os
from datetime import date, timedelta

from django.conf import settings
from django.test import TestCase

from lemon.tests.test_views.models import *
from lemon.views import *


class BaseViewTestCase(TestCase):

    def setUp(self):
        self.old_TEMPLATE_DIRS = settings.TEMPLATE_DIRS
        settings.TEMPLATE_DIRS = (
            os.path.join(os.path.dirname(__file__), 'templates'),)

    def tearDown(self):
        settings.TEMPLATE_DIRS = self.old_TEMPLATE_DIRS


class ViewTestCase(BaseViewTestCase):

    urls = 'lemon.tests.test_views.urls'

    def test_factory(self):
        view_class = View.factory('TestView', template_name='view.html')
        self.assertTrue(issubclass(view_class, View))
        self.assertEqual(view_class.__name__, 'TestView')
        self.assertEqual(view_class.template_name, 'view.html')

    def test_response(self):
        response = self.client.get('/view')
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')

    def test_response_without_current_app(self):
        response = self.client.get('/view-without-current-app')
        self.assertEqual(response.status_code, 200)
        self.assertTrue(response.context.current_app is None)


class DetailViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_factory(self):
        view_class = DetailView.factory(model='test_views.Character')
        self.assertTrue(view_class.model is Character)

    def test_factory_sets_class_name_from_mixin_model(self):
        mixins = [type('MixIn1', (), {}),
                  type('MixIn2', (), {'model': 'test_views.Character'}),
                  type('MixIn3', (), {'model': 'test_views.Season'})]
        view_class = DetailView.factory(mixins=mixins)
        self.assertEqual(view_class.__name__, 'CharacterDetailView')

    def test_response(self):
        response = self.client.get('/detail-view/1')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/character_detail.html')
        self.assertTrue('object' in response.context)

        character = response.context.get('object')
        self.assertTrue(isinstance(character, Character))
        self.assertEqual(character.pk, 1)

    def test_response_with_template_object_name(self):
        response = self.client.get('/detail-view-with-template-object-name/1')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'test_views/character_detail.html')
        self.assertTrue('character' in response.context)

    def test_response_with_template_name(self):
        response = self.client.get('/detail-view-with-template-name/1')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'test_views/detail_view.html')
        self.assertTrue('object' in response.context)

    def test_page_not_found(self):
        response = self.client.get('/detail-view/0')
        self.assertEqual(response.status_code, 404)

    def test_allow_emplty(self):
        response = self.client.get('/detail-view-allow-empty/0')
        self.assertEqual(response.status_code, 200)
        self.assertTrue('object' in response.context)
        self.assertTrue(response.context['object'] is None)

    def test_slug(self):
        response = self.client.get('/detail-view-with-slug/babylon-5')

        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'test_views/season_detail.html')
        self.assertTrue('object' in response.context)

        season = response.context.get('object')
        self.assertTrue(isinstance(season, Season))
        self.assertEqual(season.pk, 1)

    def test_mixins(self):
        response = self.client.get('/detail-view-with-mixin/1')
        self.assertEqual(response.status_code, 200)

        response = self.client.get('/detail-view-with-mixin/5')
        self.assertEqual(response.status_code, 404)


class ListViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_factory(self):
        view_class = ListView.factory(model='test_views.Character')
        self.assertTrue(view_class.model is Character)

    def test_factory_sets_class_name_from_mixin_model(self):
        mixins = [type('MixIn1', (), {}),
                  type('MixIn2', (), {'model': 'test_views.Character'}),
                  type('MixIn3', (), {'model': 'test_views.Season'})]
        view_class = ListView.factory(mixins=mixins)
        self.assertEqual(view_class.__name__, 'CharacterListView')

    def test_response(self):
        response = self.client.get('/list-view')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/character_list.html')
        self.assertTrue('object_list' in response.context)

        character_list = response.context.get('object_list')
        self.assertEqual(len(character_list), 5)
        self.assertTrue(all(isinstance(c, Character) for c in character_list))

    def test_response_with_template_list_name(self):
        response = self.client.get('/list-view-with-template-list-name')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'test_views/character_list.html')
        self.assertTrue('character_list' in response.context)

    def test_response_with_template_name(self):
        response = self.client.get('/list-view-with-template-name')
        self.assertEqual(response.status_code, 200)
        self.assertTemplateUsed(response, 'test_views/list_view.html')
        self.assertTrue('object_list' in response.context)

    def test_paginated(self):
        response = self.client.get('/list-view-paginated', {'page': 1})
        self.assertEqual(response.status_code, 200)
        self.assertTrue('object_list' in response.context)
        self.assertTrue('page' in response.context)

        character_list = response.context.get('object_list')
        page = response.context.get('page')
        paginator = page.paginator
        self.assertTrue(character_list is page.object_list)
        self.assertEqual(paginator.num_pages, 3)
        self.assertEqual(len(character_list), 2)

    def test_paginated_page_not_found(self):
        response = self.client.get('/list-view-paginated', {'page': 4})
        self.assertEqual(response.status_code, 404)

    def test_paginated_with_orphan(self):
        response = self.client.get('/list-view-paginated-with-orphan', {'page': 2})
        self.assertEqual(response.status_code, 200)
        self.assertTrue('object_list' in response.context)
        self.assertTrue('page' in response.context)

        character_list = response.context.get('object_list')
        page = response.context.get('page')
        paginator = page.paginator
        self.assertEqual(paginator.num_pages, 2)
        self.assertEqual(len(character_list), 3)


    def test_mixins(self):
        response = self.client.get('/list-view-with-mixin')
        self.assertEqual(response.status_code, 200)
        self.assertTrue('object_list' in response.context)
        character_list = response.context.get('object_list')
        self.assertEqual(len(character_list), 4)
        self.assertTrue(Character.objects.get(pk=5) not in character_list)


class ArchiveViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def setUp(self):
        super(ArchiveViewTestCase, self).setUp()
        self.episode = Episode.objects.create(
            title="Unknown",
            publish_date=date.today() + timedelta(1)
        )

    def test_response(self):
        response = self.client.get('/archive-view')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/episode_archive.html')
        self.assertTrue('object_list' in response.context)

        episode_list = response.context.get('object_list')
        self.assertTrue(all(isinstance(e, Episode) for e in episode_list))
        self.assertTrue(self.episode not in episode_list)


class YearViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_response(self):
        response = self.client.get('/year-view/1995')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/episode_archive_year.html')
        self.assertTrue('object_list' in response.context)

        episode_list = response.context.get('object_list')
        self.assertTrue(len(episode_list), 1)
        self.assertTrue(episode_list.get().pk, 3)


class MonthViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_response(self):
        response = self.client.get('/month-view/1994-11')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/episode_archive_month.html')
        self.assertTrue('object_list' in response.context)

        episode_list = response.context.get('object_list')
        self.assertTrue(len(episode_list), 1)
        self.assertTrue(episode_list.get().pk, 2)


class DayViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_response(self):
        response = self.client.get('/day-view/1994-01-26')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/episode_archive_day.html')
        self.assertTrue('object_list' in response.context)

        episode_list = response.context.get('object_list')
        self.assertTrue(len(episode_list), 1)
        self.assertTrue(episode_list.get().pk, 1)


class ArchiveDetailViewTestCase(BaseViewTestCase):

    fixtures = ['test_views.json']
    urls = 'lemon.tests.test_views.urls'

    def test_response(self):
        response = self.client.get('/archive-detail-view/1994-01-26/1')

        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.context.current_app, 'test_views')
        self.assertTemplateUsed(response, 'test_views/episode_detail.html')
        self.assertTrue('object' in response.context)

        episode = response.context.get('object')
        self.assertTrue(isinstance(episode, Episode))
        self.assertTrue(episode.pk, 1)
