from datetime import datetime

from django.db.models import get_model
from django.shortcuts import get_object_or_404

from lemon.shortcuts import get_object_or_none
from lemon.views.base import View


class DetailView(View):
    """
    .. py:attribute:: field
       The name of field which, value will be provided.

    .. py:attribute:: model
       The model class.

    .. py:attribute:: template_object_name
       The name of object in template's context (default 'object').

    .. py:attribute:: allow_empty
       If True and the object was not found, render template with None as
       object.
    """

    @classmethod
    def factory(cls, mixins=None, **kwargs):
        model = kwargs.get('model')
        if not model and mixins:
            for mixin in (mixins if isinstance(mixins, (list, tuple)) else [mixins]):
                model = getattr(mixin, 'model', None)
                if model:
                    break
        if isinstance(model, basestring):
            app_label, model_name = model.split('.')
            model = get_model(app_label, model_name)
            kwargs['model'] = model
        model = model or getattr(cls, 'model', '')
        return super(DetailView, cls).factory(model, mixins, **kwargs)

    def __init__(self, request, value=None, current_app=None):
        super(DetailView, self).__init__(request, current_app=current_app)
        self.value = value

    def cached_context(self):
        context = super(DetailView, self).cached_context()
        template_object_name = getattr(self, 'template_object_name', 'object')
        context[template_object_name] = self.get_object()
        return context

    def _get_queryset(self):
        return self.model.objects.all()

    def get_prepared_queryset(self):
        return getattr(self, 'get_queryset', self._get_queryset)()

    def get_object(self):
        queryset = self.get_prepared_queryset()
        allow_empty = getattr(self, 'allow_empty', False)
        get_object = get_object_or_none if allow_empty else get_object_or_404
        if self.value is None:
            return get_object(queryset)
        field = getattr(self, 'field', 'pk')
        return get_object(queryset, **{field: self.value})

    def get_template_name(self):
        template_name = super(DetailView, self).get_template_name()
        if template_name is None:
            app_label = self.model._meta.app_label
            object_name = self.model._meta.object_name.lower()
            return '%s/%s_detail.html' % (app_label, object_name)
        return template_name


class ArchiveDetailView(DetailView):
    """
    .. py:attribute:: date_field
       The name of field with date for filter.

    .. py:attribute:: allow_future
       If True and object is in future, it will not be shown.
    """

    def __init__(self, request, year=None, month=None, day=None, value=None, current_app=None):
        super(ArchiveDetailView, self).__init__(request, value, current_app=current_app)
        self.year = None if year is None else int(year)
        self.month = None if month is None else int(month)
        self.day = None if day is None else int(day)

    def get_prepared_queryset(self):
        queryset = super(ArchiveDetailView, self).get_prepared_queryset()
        if not getattr(self, 'allow_future', 'False'):
            lookup_kwargs = {'%s__lte' % self.date_field: datetime.now()}
            queryset = queryset.filter(**lookup_kwargs)
        if self.year is not None:
            lookup_kwargs = {'%s__year' % self.date_field: self.year}
            queryset = queryset.filter(**lookup_kwargs)
        if self.month is not None:
            lookup_kwargs = {'%s__month' % self.date_field: self.month}
            queryset = queryset.filter(**lookup_kwargs)
        if self.day is not None:
            lookup_kwargs = {'%s__day' % self.date_field: self.day}
            queryset = queryset.filter(**lookup_kwargs)
        return queryset