import datetime
import time

from django.db.models import DateTimeField
from django.http import Http404
from django.template import loader

from lemon.views.generic import list_detail


def archive_index(request, queryset, date_field, num_latest=15,
        template_name=None, template_loader=loader,
        extra_context=None, allow_empty=True, context_processors=None,
        mimetype=None, allow_future=False, template_object_name='object',
        current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    if not allow_future:
        lookup_kwargs = {'%s__lte' % date_field: datetime.datetime.now()}
        queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    return list_detail.object_index(
        request, queryset, num_latest, allow_empty, template_name,
        template_loader, extra_context, context_processors,
        template_object_name, mimetype, current_app, current_app_field
    )


def archive_year(request, year, queryset, date_field, paginate_by=None,
                 reverse_pages=True, page=None, allow_empty=True,
                 template_name=None, template_loader=loader,
                 extra_context=None, context_processors=None,
                 template_object_name='object', mimetype=None,
                 allow_future=False, current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    now = datetime.datetime.now()
    year = int(year)
    lookup_kwargs = {'%s__year' % date_field: year}
    if year >= now.year and not allow_future:
        lookup_kwargs['%s__lte' % date_field] = now
    queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive_year.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    extra_context.update({'year': year})
    return list_detail.object_list(
        request, queryset, paginate_by, reverse_pages, page, allow_empty,
        template_name, template_loader, extra_context, context_processors,
        template_object_name, mimetype, current_app, current_app_field
    )


def archive_month(request, year, month, queryset, date_field,
                  month_format='%m', paginate_by=None, reverse_pages=True,
                  page=None, allow_empty=True, template_name=None,
                  template_loader=loader, extra_context=None,
                  context_processors=None, template_object_name='object',
                  mimetype=None, allow_future=False, current_app=None,
                  current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    try:
        tt = time.strptime('%s-%s' % (year, month),
                           '%s-%s' % ('%Y', month_format))
        date = datetime.date(*tt[:3])
    except ValueError:
        raise Http404
    first_day = date.replace(day=1)
    if first_day.month == 12:
        last_day = first_day.replace(year=first_day.year + 1, month=1)
    else:
        last_day = first_day.replace(month=first_day.month + 1)

    now = datetime.datetime.now()
    lookup_kwargs = {'%s__gte' % date_field: first_day,
                     '%s__lt' % date_field: last_day}
    if last_day >= now.date() and not allow_future:
        lookup_kwargs['%s__lte' % date_field] = now
    queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive_month.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    extra_context.update({'year': date.year, 'month': date.month})
    return list_detail.object_list(
        request, queryset, paginate_by, reverse_pages, page, allow_empty,
        template_name, template_loader, extra_context, context_processors,
        template_object_name, mimetype, current_app, current_app_field
    )


def archive_week(request, year, week, queryset, date_field, paginate_by=None,
                 reverse_pages=True, page=None, allow_empty=True,
                 template_name=None, template_loader=loader,
                 extra_context=None, context_processors=None,
                 template_object_name='object', mimetype=None,
                 allow_future=False, current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    try:
        tt = time.strptime(year + '-0-' + week, '%Y-%w-%U')
        date = datetime.date(*tt[:3])
    except ValueError:
        raise Http404
    first_day = date
    last_day = date + datetime.timedelta(days=7)
    lookup_kwargs = {'%s__gte' % date_field: first_day,
                     '%s__lt' % date_field: last_day}

    now = datetime.datetime.now()
    if last_day >= now.date() and not allow_future:
        lookup_kwargs['%s__lte' % date_field] = now
    queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive_week.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    extra_context.update({'year': date.year, 'week': date.strftime('%U')})
    return list_detail.object_list(
        request, queryset, paginate_by, reverse_pages, page, allow_empty,
        template_name, template_loader, extra_context, context_processors,
        template_object_name, mimetype, current_app, current_app_field
    )


def archive_day(request, year, month, day, queryset, date_field,
                month_format='%m', day_format='%d', paginate_by=None,
                 reverse_pages=True, page=None, allow_empty=True,
                 template_name=None, template_loader=loader,
                 extra_context=None, context_processors=None,
                 template_object_name='object', mimetype=None,
                 allow_future=False, current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    try:
        tt = time.strptime('%s-%s-%s' % (year, month, day),
                           '%s-%s-%s' % ('%Y', month_format, day_format))
        date = datetime.date(*tt[:3])
    except ValueError:
        raise Http404

    now = datetime.datetime.now()
    if isinstance(model._meta.get_field(date_field), DateTimeField):
        lookup_kwargs = {'%s__range' % date_field: (
            datetime.datetime.combine(date, datetime.time.min),
            datetime.datetime.combine(date, datetime.time.max)
        )}
    else:
        lookup_kwargs = {date_field: date}
    if date >= now.date() and not allow_future:
        lookup_kwargs['%s__lte' % date_field] = now
    queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive_day.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    extra_context.update({'year': date.year,
                          'month': date.month,
                          'day': date.day})
    return list_detail.object_list(
        request, queryset, paginate_by, reverse_pages, page, allow_empty,
        template_name, template_loader, extra_context, context_processors,
        template_object_name, mimetype, current_app, current_app_field
    )


def archive_today(request, **kwargs):
    today = datetime.date.today()
    kwargs.update({'year': str(today.year),
                   'month': str(today.month),
                   'day': str(today.day)})
    return archive_day(request, **kwargs)


def archive_detail(request, year, month, day, queryset, date_field,
                   month_format='%m', day_format='%d', object_id=None,
                   slug=None, slug_field='slug', template_name=None,
                   template_name_field=None, template_loader=loader,
                   extra_context=None, context_processors=None,
                   template_object_name='object', mimetype=None,
                   current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    model = queryset.model

    try:
        tt = time.strptime('%s-%s-%s' % (year, month, day),
                           '%s-%s-%s' % ('%Y', month_format, day_format))
        date = datetime.date(*tt[:3])
    except ValueError:
        raise Http404

    if isinstance(model._meta.get_field(date_field), DateTimeField):
        lookup_kwargs = {'%s__range' % date_field: (
            datetime.datetime.combine(date, datetime.time.min),
            datetime.datetime.combine(date, datetime.time.max)
        )}
    else:
        lookup_kwargs = {date_field: date}
    queryset = queryset.filter(**lookup_kwargs)

    if not template_name:
        template_name = '%s/%s_archive_detail.html' % \
            (model._meta.app_label, model._meta.object_name.lower())
    return list_detail.object_detail(
        request, queryset, object_id, slug, slug_field, template_name,
        template_name_field, template_loader, extra_context,
        context_processors, template_object_name, mimetype, current_app,
        current_app_field
    )
