from django.core.exceptions import ObjectDoesNotExist
from django.core.paginator import Paginator, InvalidPage
from django.core.xheaders import populate_xheaders
from django.http import Http404, HttpResponse
from django.template import loader, RequestContext


def object_index(request, queryset, max_items=None, allow_empty=True,
                 template_name=None, template_loader=loader,
                 extra_context=None, context_processors=None,
                 template_object_name='object', mimetype=None,
                 current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    if current_app and current_app_field:
        queryset = queryset.filter(**{current_app_field: current_app})
    if max_items:
        queryset = queryset[:max_items]

    if not allow_empty and len(queryset) == 0:
        raise Http404
    context_instance = RequestContext(request, {
        '%s_list' % template_object_name: queryset,
        'paginator': None,
        'page_obj': None,
        'is_paginated': False,
    }, context_processors, current_app)
    for key, value in extra_context.items():
        if callable(value):
            context_instance[key] = value()
        else:
            context_instance[key] = value
    if not template_name:
        model = queryset.model
        template_name = "%s/%s_index.html" % (model._meta.app_label,
                                              model._meta.object_name.lower())
    template = template_loader.get_template(template_name)
    return HttpResponse(template.render(context_instance), mimetype=mimetype)


def object_list(request, queryset, paginate_by=None, reverse_pages=True,
                page=None, allow_empty=True, template_name=None,
                template_loader=loader, extra_context=None,
                context_processors=None, template_object_name='object',
                mimetype=None, current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    if current_app and current_app_field:
        queryset = queryset.filter(**{current_app_field: current_app})
    if paginate_by:
        paginator = Paginator(
            queryset.reverse() if reverse_pages else queryset,
            paginate_by, allow_empty_first_page=allow_empty
        )
        if not page:
            page = request.GET.get('page', 'last' if reverse_pages else 1)
        try:
            page_number = int(page)
        except ValueError:
            if page == 'last':
                page_number = paginator.num_pages
            else:
                # Page is not 'last', nor can it be converted to an int.
                raise Http404
        try:
            page_obj = paginator.page(page_number)
        except InvalidPage:
            raise Http404
        if reverse_pages:
            object_list = reversed(page_obj.object_list)
        else:
            object_list = page_obj.object_list
        context_instance = RequestContext(request, {
            '%s_list' % template_object_name: object_list,
            'paginator': paginator,
            'page_obj': page_obj,
        }, context_processors, current_app)
    else:
        if not allow_empty and len(queryset) == 0:
            raise Http404
        context_instance = RequestContext(request, {
            '%s_list' % template_object_name: queryset,
            'paginator': None,
            'page_obj': None,
            'is_paginated': False,
        }, context_processors, current_app)
    for key, value in extra_context.items():
        if callable(value):
            context_instance[key] = value()
        else:
            context_instance[key] = value
    if not template_name:
        model = queryset.model
        template_name = "%s/%s_list.html" % (model._meta.app_label,
                                             model._meta.object_name.lower())
    template = template_loader.get_template(template_name)
    return HttpResponse(template.render(context_instance), mimetype=mimetype)


def object_detail(request, queryset, object_id=None, slug=None,
        slug_field='slug', template_name=None, template_name_field=None,
        template_loader=loader, extra_context=None,
        context_processors=None, template_object_name='object',
        mimetype=None, current_app=None, current_app_field=None):
    if extra_context is None: extra_context = {}
    queryset = queryset() if callable(queryset) else queryset._clone()
    if current_app and current_app_field:
        queryset = queryset.filter(**{current_app_field: current_app})
    model = queryset.model
    if object_id:
        queryset = queryset.filter(pk=object_id)
    elif slug and slug_field:
        queryset = queryset.filter(**{slug_field: slug})
    else:
        raise AttributeError('Generic detail view must be called with '
                             'either an object_id or a slug/slug_field.')
    try:
        obj = queryset.get()
    except ObjectDoesNotExist:
        raise Http404('No %s found matching the query' % \
                      (model._meta.verbose_name))
    if not template_name:
        template_name = "%s/%s_detail.html" % \
            (model._meta.app_label, model._meta.object_name.lower())
    if template_name_field:
        template_name_list = [getattr(obj, template_name_field), template_name]
        template = template_loader.select_template(template_name_list)
    else:
        template = template_loader.get_template(template_name)
    context_instance = RequestContext(request, {
        template_object_name: obj,
    }, context_processors, current_app)
    for key, value in extra_context.items():
        if callable(value):
            context_instance[key] = value()
        else:
            context_instance[key] = value
    response = HttpResponse(template.render(context_instance),
                            mimetype=mimetype)
    populate_xheaders(request, response, model,
                      getattr(obj, obj._meta.pk.name))
    return response
