from django.core.paginator import Paginator, InvalidPage
from django.db.models import get_model
from django.http import Http404

from lemon.views.base import View


class ListView(View):
    """
    .. py:attribute:: model
       The model class.

    .. py:attribute:: template_list_name
       The name of object list in template's context (default 'object_list').

    .. py:attribute:: allow_empty_first_page
       Whether or not the first page is allowed to be empty. If False and
       object_list is empty, then an EmptyPage error will be raised
       (default True).

    .. py:attribute:: paginate_by
       The maximum number of items to include on a page, not including orphans
       (see the orphans optional argument below).

    .. py:attribute:: orphans
       The minimum number of items allowed on the last page, (default 0).
    """

    @classmethod
    def factory(cls, mixins=None, **kwargs):
        model = kwargs.get('model')
        if not model and mixins:
            for mixin in (mixins if isinstance(mixins, (list, tuple)) else [mixins]):
                model = getattr(mixin, 'model', None)
                if model:
                    break
        if isinstance(model, basestring):
            app_label, model_name = model.split('.')
            model = get_model(app_label, model_name)
            kwargs['model'] = model
        model = model or getattr(cls, 'model', None) or ''
        return super(ListView, cls).factory(model, mixins, **kwargs)

    def cached_context(self):
        template_list_name = getattr(self, 'template_list_name', 'object_list')
        if getattr(self, 'paginate_by', None):
            page = self.get_page()
            return {'page': page, template_list_name: page.object_list}
        return {template_list_name: self.get_prepared_queryset()}

    def _get_queryset(self):
        return self.model.objects.all()

    def get_prepared_queryset(self):
        return getattr(self, 'get_queryset', self._get_queryset)()

    def get_template_name(self, suffix='list'):
        template_name = super(ListView, self).get_template_name()
        if template_name is None:
            app_label = self.model._meta.app_label
            object_name = self.model._meta.object_name.lower()
            return '%s/%s_%s.html' % (app_label, object_name, suffix)
        return template_name

    def get_paginator(self):
        allow_empty_first_page = getattr(self, 'allow_empty_first_page', True)
        return Paginator(
            self.get_prepared_queryset(), self.paginate_by,
            orphans=getattr(self, 'orphans', 0),
            allow_empty_first_page=allow_empty_first_page)

    def get_page_number(self):
        return self.request.GET.get('page', 1)

    def get_page(self):
        paginator = self.get_paginator()
        page_number = self.get_page_number()
        try:
            page_number = int(page_number)
        except ValueError:
            if page_number == 'last':
                page_number = paginator.num_pages
            else:
                raise Http404
        try:
            page = paginator.page(page_number)
        except InvalidPage:
            raise Http404
        return page