#!/usr/bin/env python
"""The main module of loslassa as comfortable command line script"""
import logging
import os
import sys

from plumbum import cli
# noinspection PyUnresolvedReferences
from plumbum.cmd import sphinx_build
from plumbum.local_machine import LocalPath

# quick hack for initial dev to not add something to PYTHONPATH
path = os.path.dirname(os.path.abspath(__file__)).rpartition('loslassa')[0]
if path not in sys.path:
    sys.path.insert(0, path)

from loslassa.devserver import serve_with_reloader

log = logging.getLogger(__name__)

LOSLASSA_ROOT = os.path.abspath(os.path.dirname(__file__))
EXAMPLE_PROJECT_PATH = os.path.join(LOSLASSA_ROOT, "example_project")


class Loslassa(cli.Application):
    PROGNAME = "loslassa"
    VERSION = "0.1"
    verbose = cli.Flag(
        ["v", "verbose"], help = "If given, I will be very talkative")

    def main(self, *args):
        if args:
            print("unknown command %r" % (args[0]))
            return 1

        if not self.nested_command:
            print("command missing")
            return 1

        if self.verbose:
            print "executing command %s" % str(self.nested_command)


@Loslassa.subcommand("start")
class LoslassaStart(cli.Application):
    """Starts a new project by creating the initial project structure"""

    projectPath = EXAMPLE_PROJECT_PATH

    @cli.autoswitch(str)
    def log_to_file(self, filename):
        """Sets the file into which logs will be emitted"""
        log.addHandler(logging.FileHandler(filename))


    def main(self):
        print("start loslassing...")


@Loslassa.subcommand("play")
class LoslassaPlay(cli.Application):
    """Start playing with the source and create your page"""
    serverPort = 8080
    projectPath = os.getcwd()

    @cli.autoswitch(str)
    def project_path(self, projectPath):
        """Set path instead of using CWD"""
        self.pathToProject = projectPath

    @cli.autoswitch(int)
    def serve_on_port(self, port):
        """Sets the file into which logs will be emitted"""
        self.serverPort = port

    @cli.autoswitch(str)
    def log_to_file(self, filename):
        """Sets the file into which logs will be emitted"""
        log.addHandler(logging.FileHandler(filename))

    def main(self):
        cwd = os.getcwd()
        sphinxConfPath = LocalPath(os.path.join(cwd, "conf.py"))
        sourcePath = LocalPath(os.path.join(cwd, "source"))
        buildPath = LocalPath(os.path.join(cwd, "build"))
        doctreesPath = buildPath.join("doctrees")
        outputPath = buildPath.join("html")
        self._check_paths(
            [sphinxConfPath, sourcePath, buildPath, doctreesPath, outputPath])

        sphinxBuildCommand = sphinx_build[
            "-b", "dirhtml", "-d", doctreesPath, sourcePath, outputPath]
        print("play loslassing...")
        serve_with_reloader(
            str(outputPath), self.serverPort,
            sphinxBuildCommand,pathToWatch=str(sourcePath))

    def _check_paths(self, pathsToCheck):
        """Make sure we have a valid seeming sphinx project to work with

        :param pathsToCheck: to be checked for existence
        :type pathsToCheck: list of LocalPath
        :raise: `.LoslassaError`
        """
        for thisPath in pathsToCheck:
            if not thisPath.exists:
                raise LoslassaError(
                    "Expected path %s does not exist" % (thisPath._path))


@Loslassa.subcommand("loslassa")
class LoslassaLoslassa(cli.Application):
    """Practice loslassing by pushing your page into the interwebs"""

    def main(self):
        print("start loslassing...")


class LoslassaError(Exception):
    pass


if __name__ == "__main__":
    logging.basicConfig(level=logging.DEBUG)
    Loslassa.run()
