
from django.contrib.contenttypes.generic import GenericRelation
from django.db.models import IntegerField, CharField, FloatField
from django.db.models.signals import post_save, post_delete


class BaseGenericRelation(GenericRelation):
    """
    Extends ``GenericRelation`` to:

    - Add a consistent default value for ``object_id_field`` and
      check for a ``related_model`` attribute which can be defined
      on subclasses as a default for the ``to`` argument.

    - Add one or more custom fields to the model that the relation
      field is applied to, and then call a ``related_items_changed``
      method each time related items are saved or deleted, so that a
      calculated value can be stored against the custom fields since
      aggregates aren't available for GenericRelation instances.

    """

    # Mapping of field names to model fields that will be added.
    fields = {}

    def __init__(self, *args, **kwargs):
        """
        Set up some defaults and check for a ``related_model``
        attribute for the ``to`` argument.
        """
        kwargs.setdefault("object_id_field", "object_pk")
        to = getattr(self, "related_model", None)
        if to:
            kwargs.setdefault("to", to)
        super(BaseGenericRelation, self).__init__(*args, **kwargs)

    def contribute_to_class(self, cls, name):
        """
        Add each of the names and fields in the ``fields`` attribute
        to the model the relationship field is applied to, and set up
        the related item save and delete signals for calling
        ``related_items_changed``.
        """
        super(BaseGenericRelation, self).contribute_to_class(cls, name)
        self.related_field_name = name
        # Not applicable to abstract classes, and in fact will break.
        if not cls._meta.abstract:
            for (name_string, field) in self.fields.items():
                if "%s" in name_string:
                    name_string = name_string % name
                if not field.verbose_name:
                    field.verbose_name = self.verbose_name
                cls.add_to_class(name_string, field)
            # For some unknown reason the signal won't be triggered
            # if given a sender arg, particularly when running
            # Cartridge with the field RichTextPage.keywords - so
            # instead of specifying self.rel.to as the sender, we
            # check for it inside the signal itself.
            post_save.connect(self._related_items_changed)
            post_delete.connect(self._related_items_changed)

    def _related_items_changed(self, **kwargs):
        """
        Ensure that the given related item is actually for the model
        this field applies to, and pass the instance to the real
        ``related_items_changed`` handler.
        """
        # Manually check that the instance matches the relation,
        # since we don't specify a sender for the signal.
        if not isinstance(kwargs["instance"], self.rel.to):
            return
        for_model = kwargs["instance"].content_type.model_class()
        if issubclass(for_model, self.model):
            instance = self.model.objects.get(id=kwargs["instance"].object_pk)
            if hasattr(instance, "get_content_model"):
                instance = instance.get_content_model()
            related_manager = getattr(instance, self.related_field_name)
            self.related_items_changed(instance, related_manager)

    def related_items_changed(self, instance, related_manager):
        """
        Can be implemented by subclasses - called whenever the
        state of related items change, eg they're saved or deleted.
        The instance for this field and the related manager for the
        field are passed as arguments.
        """
        pass


class CommentsField(BaseGenericRelation):
    """
    Stores the number of comments against the ``COMMENTS_FIELD_count``
    field when a comment is saved or deleted.
    """

    related_model = "generic.ThreadedComment"
    fields = {"%s_count": IntegerField(editable=False, default=0)}

    def related_items_changed(self, instance, related_manager):
        """
        Stores the number of comments. A custom ``count_filter``
        queryset gets checked for, allowing managers to implement
        custom count logic.
        """
        try:
            count = related_manager.count_queryset()
        except AttributeError:
            count = related_manager.count()
        count_field_name = self.fields.keys()[0] % self.related_field_name
        setattr(instance, count_field_name, count)
        instance.save()


class KeywordsField(BaseGenericRelation):
    """
    Stores the keywords as a single string into the
    ``KEYWORDS_FIELD_string``  field for convenient access when
    searching.
    """

    related_model = "generic.AssignedKeyword"
    fields = {"%s_string": CharField(blank=True, max_length=500)}

    def __init__(self, *args, **kwargs):
        """
        Mark the field as editable so that it can be specified in
        admin class fieldsets and pass validation, and also so that
        it shows up in the admin form.
        """
        super(KeywordsField, self).__init__(*args, **kwargs)
        self.editable = True

    def formfield(self, **kwargs):
        """
        Provide the custom form widget for the admin, since there
        isn't a form field mapped to ``GenericRelation`` model fields.
        """
        from mezzanine.generic.forms import KeywordsWidget
        kwargs["widget"] = KeywordsWidget()
        return super(KeywordsField, self).formfield(**kwargs)

    def save_form_data(self, instance, data):
        """
        The ``KeywordsWidget`` field will return data as a string of
        comma separated IDs for the ``Keyword`` model - convert these
        into actual ``AssignedKeyword`` instances.
        """
        from mezzanine.generic.models import AssignedKeyword
        # Remove current assigned keywords.
        related_manager = getattr(instance, self.name)
        related_manager.all().delete()
        if data:
            data = [AssignedKeyword(keyword_id=i) for i in data.split(",")]
        super(KeywordsField, self).save_form_data(instance, data)

    def contribute_to_class(self, cls, name):
        """
        Swap out any reference to ``KeywordsField`` with the
        ``KEYWORDS_FIELD_string`` field in ``search_fields``.
        """
        super(KeywordsField, self).contribute_to_class(cls, name)
        string_field_name = self.fields.keys()[0] % self.related_field_name
        if hasattr(cls, "search_fields") and name in cls.search_fields:
            try:
                weight = cls.search_fields[name]
            except AttributeError:
                # search_fields is a sequence.
                index = cls.search_fields.index(name)
                cls.search_fields[index] = string_field_name
            else:
                del cls.search_fields[name]
                cls.search_fields[string_field_name] = weight

    def related_items_changed(self, instance, related_manager):
        """
        Stores the keywords as a single string for searching.
        """
        assigned = related_manager.select_related("keyword")
        keywords = " ".join([unicode(a.keyword) for a in assigned])
        string_field_name = self.fields.keys()[0] % self.related_field_name
        if getattr(instance, string_field_name) != keywords:
            setattr(instance, string_field_name, keywords)
            instance.save()


class RatingField(BaseGenericRelation):
    """
    Stores the average rating against the ``RATING_FIELD_average``
    field when a rating is saved or deleted.
    """

    related_model = "generic.Rating"
    fields = {"%s_count": IntegerField(default=0),
              "%s_average": FloatField(default=0)}

    def related_items_changed(self, instance, related_manager):
        """
        Calculates and saves the average rating.
        """
        ratings = [r.value for r in related_manager.all()]
        count = len(ratings)
        average = sum(ratings) / float(count) if count > 0 else 0
        setattr(instance, "%s_count" % self.related_field_name, count)
        setattr(instance, "%s_average" % self.related_field_name, average)
        instance.save()
