#!/usr/bin/env python
import os
import shlex
import subprocess
import logging

import web
import strutils

logging.getLogger("commons.utils").setLevel(logging.DEBUG)


__all__ = [
    "cat",
    "get_app_data_path",
    "run",
    "which",
]


def cat(*args):
    buf = ""
    for i in args:
        full_path = web.safeunicode(i)
        if os.path.isfile(full_path):
            f = file(full_path)
            buf += f.read().strip()
            f.close()

    try:
        buf = web.safeunicode(buf)
    except UnicodeDecodeError:
        msg = "expected %s file(s) in unicode, got un-know encoding" % ", ".join(args)
        logging.warn(msg)

        buf = u""

    return buf


def get_app_data_path(app_name):
    home_path = os.getenv("HOME") or os.path.expanduser("~")
    full_path = os.path.join(home_path, "Documents", app_name)
    if not os.path.exists(full_path):
        os.makedirs(full_path)
    return full_path


def run(cmd, strip_newline = True):
    """
    Using subprocess module implementations os.popen()

    >>> run("file --mime-type %s | awk \'{print $2}\'" % os.path.realpath(__file__))
    'text/x-java'

    >>> run("uname -s") in ("Darwin", "Linux")
    True
    """
    splits = strutils.split_by_token(cmd, "|")

    if len(splits) == 1:
        p_obj = subprocess.Popen(shlex.split(cmd), stdout = subprocess.PIPE)
        resp = p_obj.stdout.read()
    else:
        cmd = splits[0]
        last_p = subprocess.Popen(shlex.split(cmd), stdout = subprocess.PIPE)
        output = None

        for cmd in splits[1:]:
            p = subprocess.Popen(shlex.split(cmd), stdin = last_p.stdout, stdout = subprocess.PIPE)
            last_p.stdout.close()
            output = p.communicate()[0]
            last_p = p

        resp = output

    if strip_newline:
        return resp.split('\n')[0]
    else:
        return resp


def which(name):
    bin_paths = (
        "/usr/bin", # APT on Debian
        "/opt/local/bin", # PortsMac on Mac OS X
        )

    for i in bin_paths:
        full_path = os.path.join(i, name)
        if os.path.exists(full_path):
            return full_path


if __name__ == "__main__":
    import doctest
    doctest.testmod()