"""
Delegate decorator in Python

Delegation pattern
 - http://en.wikipedia.org/wiki/Delegation_pattern
 - http://en.wikipedia.org/wiki/Delegation_%28programming%29#As_a_language_feature

Python Decorators
 - http://stackoverflow.com/questions/739654/understanding-python-decorators
"""

from functools import wraps
import sys

__all__ = [
    'delegate_required',
    'delegate', 'delegate_pre', 'delegate_post',
    'delegate_with_func_name',
]


WARNING = False

def delegate(func):
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        if hasattr(self.delegate, func.__name__):
            delegate_func = getattr(self.delegate, func.func_name)

            return delegate_func(*args, **kwargs)

        else:
            msg = "WARNING: you should implement delegate method %s" % func.func_name
            sys.stdout.write('\n' + msg + '\n')

            return func(self, *args, **kwargs)

    return wrapper

def delegate_required(func):
    """ NOTICE: this decorator can use with the decorator `delegate` only. """
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        if not hasattr(self.delegate, func.__name__):
            msg = "ERROR: you have to implement delegate method %s" % func.func_name
            raise Exception(msg)
        return func(self, *args, **kwargs)

    return wrapper

def delegate_pre(func):
    @wraps(func)
    def wrapper(self, *args, **kwargs):
        if hasattr(self.delegate, func.__name__):
            delegate_func = getattr(self.delegate, func.func_name)

            resp = delegate_func(*args, **kwargs)

            if resp:
                func(self, *args, **kwargs)

            return resp

        else:
            if WARNING:
                msg = "WARNING: you should implement delegate(pre) method %s" % func.func_name
                sys.stdout.write('\n' + msg + '\n')

            return func(self, *args, **kwargs)

    return wrapper


def delegate_post(func):
    """ you should return True in original func"""
    @wraps(func)
    def wrapper(self, *args, **kwargs):

        resp = func(self, *args, **kwargs)

        if hasattr(self.delegate, func.__name__):
            if resp:
                delegate_func = getattr(self.delegate, func.func_name)
                return delegate_func(*args, **kwargs)
        else:
            if WARNING:
                msg = "WARNING: you should implement delegate(post) method %s" % func.func_name
                sys.stdout.write('\n' + msg + '\n')

    return wrapper


def _get_real_func_name(name):
    if name.find('(') != -1:
        return name[:name.find("(")]
    return name


def delegate_with_func_name(func_name=None):
    def _delegate(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            if func_name:
                _func_name = _get_real_func_name(func_name)
                if hasattr(self.delegate, _func_name):
                    target_func = getattr(self.delegate, _func_name)
                    return target_func(*args, **kwargs)
                else:
                    msg = "WARNING: you should implement delegate method %s" % func_name
                    sys.stdout.write('\n' + msg + '\n')
            else:
                if hasattr(self.delegate, func.__name__):
                    target_func = getattr(self.delegate, func.__name__)
                    return target_func(*args, **kwargs)
                else:
                    msg = "WARNING: you should implement delegate method %s" % func.__name__
                    sys.stdout.write('\n' + msg + '\n')

            return func(self, *args, **kwargs)

        return wrapper

    return _delegate


def test_delegate():
    class Foo:
        def __init__(self, delegate=None):
            self.delegate = delegate

        @delegate
        def do_sth(self, *args, **kwargs):
            print "Foo instance's do_sth() called'"
            return "Foo do_sth()'s return"


    class Bar:
        def __init__(self):
            """ this class implements Foo's delegate method(s) """
            pass

        def do_sth(self, *args, **kwargs):
            print "Bar instance's do_sth() called"
            return  "Bar instance do_sth()'s return"


    class NotBar:
        def __init__(self):
            """ this class does not implements Foo's delegate method(s) """
            pass


    bar = Bar()
    foo = Foo(bar)
    r1 = foo.do_sth()
    assert r1 == "Bar instance do_sth()'s return"


    not_bar = NotBar()
    foo = Foo(not_bar)
    r2 = foo.do_sth()
    assert r2 == None


def test_delegate_with_func_name():
    class Foo:
        def __init__(self, delegate):
            self.delegate = delegate

        @delegate_with_func_name()
        def do_sth(self, *args, **kwargs):
            print 'Foo_instance.do_sth()'


    class Bar:
        def asdf(self):
            print 'Bar_instance asdf'


    bar = Bar()
    foo = Foo(delegate=bar)
    foo.do_sth()


if __name__ == "__main__":
    pass