import StringIO

from lxml import etree
from BeautifulSoup import BeautifulStoneSoup
import web


__all__ = [
    "mapping_lxml_node_children_to_a_dict",
    "mapping_lxml_node_attrs_to_dict",
    'parse_xml_from_str',
]


def mapping_lxml_node_children_to_a_dict(parent_element, filters=None):
    """ Mapping children's tag name-text of a element (a instance of lxml.etree._Element) to
    a dictionary's key-value attributes.

    Example:

        <config>
            <servers version="170">
                <sipc-proxy>58.68.229.64:8080</sipc-proxy>
                <ssi-app-sign-in-v2>https://uid.fetion.com.cn/ssiportal/SSIAppSignInV4.aspx</ssi-app-sign-in-v2>
            </servers>
        </config>

        -->

        {
            "sipc_proxy" : '58.68.229.64:8080'
            "ssi_app_sign_in_v2" : 'https://uid.fetion.com.cn/ssiportal/SSIAppSignInV4.aspx'
        }

        >>> from lxml import etree
        >>> tree = etree.parse(StringIO.StringIO('<config><servers version="170"><sipc-proxy>58.68.229.64:8080</sipc-proxy><ssi-app-sign-in-v2>https://uid.fetion.com.cn/ssiportal/SSIAppSignInV4.aspx</ssi-app-sign-in-v2></servers><foo /></config>'))
        >>> element = tree.xpath("/config/servers")[0]
        >>> node_name_list = ['servers']
        >>> got_d = mapping_lxml_node_children_to_a_dict(element, node_name_list)
        >>> got_d["sipc_proxy"]
        u'58.68.229.64:8080'
        >>> got_d["ssi_app_sign_in_v2"]
        u'https://uid.fetion.com.cn/ssiportal/SSIAppSignInV4.aspx'
    """
    children = parent_element.getchildren()
    if not children:
        return

    d = {}
    for element in children:
        if element.getchildren():
            continue

        key = element.tag.replace("-", "_")

        val = element.text
        if val is not None:
            val = web.safeunicode(element.text)
            val = val.strip()

            if val == "false":
                val = False
            elif val == "true":
                val = True
        else:
            val = ""

        if filters and key in filters:
            key, val = filters[key](key, val)

        d[key] = val

    return d

def mapping_lxml_node_attrs_to_dict(element, attr_name_list, kv_filters = None):
    """ Mapping attributes name-value of a element (a instance of lxml.etree._Element) to
    a dictionary's key-value attributes.

    >>> import StringIO
    >>> from lxml import etree
    >>> tree = etree.parse(StringIO.StringIO('<contacts><contact name="foo" age="123" /></contacts>'))
    >>> contact_element = tree.xpath('/contacts/contact')[0]
    >>> got_d = mapping_lxml_node_attrs_to_dict(contact_element, '*')
    >>> got_d['name']
    u'foo'
    >>> got_d['age']
    u'123'
    """
    d = {}

    if attr_name_list == "*":
        items = element.attrib.items()
        items.sort()

        for item in items:
            key, val = item[0].replace("-", "_"), item[1]

            val = web.safeunicode(val)

            if kv_filters and key in kv_filters:
                key, val = kv_filters[key](key, val)

            key = key.replace("-", "_")

            if val == "false":
                val = False
            elif val == "true":
                val = True

            d[key] = val

    elif attr_name_list:
        attrs = element.attrib
        for name in attr_name_list:
            if name not in attrs:
                continue

            key = name
            val = attrs[name]
            val = web.safeunicode(val)

            if kv_filters and key in kv_filters:
                key, val = kv_filters[key](key, val)

            key = key.replace("-", "_")

            if val == "false":
                val = False
            elif val == "true":
                val = True

            d[key] = val

    return d


def parse_xml_from_str(buf):
    try:
        tree = etree.parse(StringIO.StringIO(buf))
    except Exception:
        # We do use BeautifulStoneSoup here,
        # because of it throw exception about `ExpatError: not well-formed` sometimes.
        buf = web.safeunicode(buf)
        buf = BeautifulStoneSoup(''.join(buf)).prettify()
        buf = web.safestr(buf)

        f = StringIO.StringIO(buf)
        tree = etree.parse(f)

    return tree


if __name__ == "__main__":
    import doctest
    doctest.testmod()