#!/usr/bin/env python
"""
References

    RFC 3339 - Date and Time on the Internet: Timestamps
        http://tools.ietf.org/html/rfc3339

    RFC 2822 - Internet Message Format, 3.3. Date and Time Specification
        http://tools.ietf.org/html/rfc2822.html#section-3.3
"""

import calendar
import datetime
import time

__all__ = [
    'generate_gmt_ts',
    'gmt_strtol',
    'gmt_ltostr',
    'gmt_to_locale_a',
    'unix_ts_to_local_s',
    'local_s_to_unix_ts',

    "get_current_date_in_yyyymmdd",
    "get_list_by_date_delta",
]


def generate_gmt_ts(time_t=None):
    """ Return GMT timestamp likes 'Tue, 25 Oct 2011 05:14:40 GMT'.

    >>> time_t = time.gmtime(1319529434.962445)
    >>> generate_gmt_ts(time_t)
    'Tue, 25 Oct 2011 07:57:14 GMT'
    """
    if not time_t:
        time_t = time.gmtime()
    return time.strftime('%a, %d %b %Y %H:%M:%S GMT', time_t)

def gmt_strtol(a_str):
    """ Convert GMT timestamp from string to long.

    >>> gmt_strtol('Sun, 16 Oct 2011 16:49:59 GMT')
    1318783799
    """
    t_format = "%a, %d %b %Y %H:%M:%S %Z"
    time_t = time.strptime(a_str, t_format)
    return int(calendar.timegm(time_t))

def gmt_ltostr(a_long):
    """ Convert GMT timestamp from long to string.

    >>> gmt_ltostr(1318783799)
    'Sun, 16 Oct 2011 16:49:59 GMT'
    """

    t_format = "%a, %d %b %Y %H:%M:%S GMT"
    time_t = time.gmtime(a_long)
    return time.strftime(t_format, time_t)


def gmt_to_locale_a(a_str):
    """ Convert GMT timestamp to locale timestamp for human read.

    'Tue, 25 Oct 2011 07:57:14 GMT'
    ->
    '2011-10-25 15:57:14'
    """

    time_t = time.localtime(gmt_strtol(a_str))
    return time.strftime('%Y-%m-%d %H:%M:%S', time_t)

def unix_ts_to_local_s(a_long = None):
    """ Convert Unix timestamp to locale timestamp for human read.

    1318783799
    ->
    '2011-10-17 00:49:59'
    
    sqlite> SELECT datetime(1318783799, 'unixepoch', 'localtime');

    >>> local_s_to_unix_ts(unix_ts_to_local_s(1318783799)) == 1318783799
    True
    """
    if a_long is None:
        a_long = time.time()
    else:
        a_long = float(a_long)

    time_t = time.localtime(a_long)
    return time.strftime('%Y-%m-%d %H:%M:%S', time_t)

def local_s_to_unix_ts(ts_str):
    """ convert locale timestamp to unix timestamp in long

    1318783799
    ->
    '2011-10-17 00:49:59'

    >>> unix_ts_to_local_s(local_s_to_unix_ts('2011-10-17 00:49:59')) == '2011-10-17 00:49:59'
    True
    """
    if ts_str == '0000-00-00 00:00:00':
        return 0

    ts_t = datetime.datetime.strptime(ts_str, "%Y-%m-%d %H:%M:%S")
    ts = time.mktime(ts_t.timetuple())+1e-6 * ts_t.microsecond
    return long(ts)


def get_current_date_in_yyyymmdd():
    """ return local today date in YYYYMMDD likes "20120112" """
    return str(datetime.date.today()).replace("-", "")

def get_date_in_yyyymm(yyyymm = None, delta_month = None):
    """
    >>> get_date_in_yyyymm(yyyymm = "201207", delta_month = 8)
    '201111'
    >>> get_date_in_yyyymm(yyyymm = "201208", delta_month = 1)
    '201207'
    """
    if yyyymm is None:
        yyyymm = time.strftime('%Y%m')

    if delta_month is None:
        return yyyymm
    else:
        the_year, the_month = int(yyyymm[:4]), int(yyyymm[4:])
        remain_months = the_year * 12 + the_month - delta_month
        yyyy, mm  = str(remain_months / 12), str(remain_months % 12)
        if len(mm) != 2:
            mm = '0' + mm
        return yyyy + mm


def yyyyddmm2ts(since):
    if not isinstance(since, basestring):
        since = str(since)

    yyyy = int(since[:4])
    mm = int(since[4:6])
    if len(since) > 6:
        dd = int(since[6:])
    else:
        dd = 1

    d = datetime.date(yyyy, mm, dd)
    unix_timestamp = time.mktime(d.timetuple())
    return unix_timestamp


def get_list_by_date_delta(since_yyyymmdd, end_yyyymmdd = get_current_date_in_yyyymmdd()):
    """
    >>> get_list_by_date_delta(20120801, 20120805)
    ['20120801', '20120802', '20120803', '20120804']
    """

    since_yyyymmdd_ts = yyyyddmm2ts(since_yyyymmdd)
    end_yyyymmdd_ts = yyyyddmm2ts(end_yyyymmdd)

    assert since_yyyymmdd_ts < end_yyyymmdd_ts
    box = []
    a = since_yyyymmdd_ts
    while a < end_yyyymmdd_ts:
        yyyymmdd_t = datetime.date.fromtimestamp(a)
        yyyymmdd = str(yyyymmdd_t).replace("-", "")
        box.append(yyyymmdd)
        a += 60 * 60 * 24
    return box

if __name__ == "__main__":
    import doctest
    doctest.testmod()
