#!/usr/bin/env python
"""
This script could use to

- get one or all network interface MAC address
- valid IP if in a network range

"""
import os
import socket
import struct
import uuid
import sys
import iptools
import requests

socket.setdefaulttimeout(2.0)

__all__ = [
    "get_mac_addr_by_if_name",
    "get_mac_addr",
    "get_mac_addrs_on_mac",

    'IANA_RESERVED_NETWORK_RANGES',
    'get_lan_ip',
    'get_default_network_range',
    'ip_in_network_range', 'ip_in_network_ranges',
]


def validipaddr(address):
    """
    Returns True if `address` is a valid IPv4 address.

        >>> validipaddr('192.168.1.1')
        True
        >>> validipaddr('192.168.1.800')
        False
        >>> validipaddr('192.168.1')
        False

    This function copy from web.py (web.net.validipaddr) .
    """
    try:
        octets = address.split('.')
        if len(octets) != 4:
            return False
        for x in octets:
            if not (0 <= int(x) <= 255):
                return False
    except ValueError:
        return False
    return True

def _get_mac_addr_on_linux(ifname = "eth0"):
    import fcntl
    s = socket.socket()
    info = fcntl.ioctl(s.fileno(), 0x8927,  struct.pack('256s', ifname[:15]))
    buf = ''.join(['%02x:' % ord(char) for char in info[18:24]])[:-1]
    return buf

def _get_mac_addr_on_mac(hardware_port = "Ethernet"):
    get_mac_addr_cmd = "networksetup -getmacaddress %s | awk '{print $3}'" % hardware_port
    resp = os.popen(get_mac_addr_cmd).read().strip()
    return resp

def get_mac_addr_by_if_name(if_name):
    if sys.platform == "darwin":
        return get_mac_addr_on_mac(if_name)

    elif sys.platform == "linux2":
        return get_mac_addr_on_linux(if_name)

    else:
        raise Exception

def get_mac_addr():
    """ Return the first network interface MAC address.

    http://stackoverflow.com/questions/159137/getting-mac-address

    Test environment:
        - Linux 2.6.39.1-x86_64
        - Darwin 10.8.0
    """
    resp = hex(uuid.getnode())[2:14]
    buf = [resp[i : i + 2] for i in xrange(0, len(resp), 2)]
    return ":".join(buf)

def get_mac_addrs_on_mac():
    get_mac_addrs_cmd = "networksetup -listallhardwareports | grep 'Ethernet Address' | grep -v 'N/A' | awk '{print $3}'"
    resp = os.popen(get_mac_addrs_cmd).read().strip().split('\n')
    return resp


# http://en.wikipedia.org/wiki/IP_address

IANA_RESERVED_NETWORK_RANGES = ("10.0.0.0/24", "172.16.0.0/20", "192.168.0.0/16")

def get_lan_ip():
    """ Return LAN IP address.
    http://stackoverflow.com/questions/166506/finding-local-ip-addresses-using-pythons-stdlib

    TODO:
    fix issue - it doesn't works when connect to network via PPPoE(it returns WAN IP)
    """
    name = sys.platform

    if name == "darwin":
        return _get_lan_ip_on_mac()

    elif name == "linux2":
        return _get_lan_ip_on_linux()

    else:
        raise Exception


def _get_lan_ip_on_mac():
    """ Return LAN IP address, it works on box that behinds router/firewall.
    """
    return socket.gethostbyname(socket.gethostname())

def _get_lan_ip_on_linux():
    """ Return LAN IP address, it works on box that behinds router/firewall.
    """
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
    s.connect(("baidu.com", 80))
    ip = s.getsockname()[0]
    s.close()
    return ip

def get_default_network_range(lan_ip=None):
    """ Return IANA reserved private network range(such as '192.168.0.0/16') if your box behinds router or firewall,
    else return '127.0.0.1'.

    >>> get_default_network_range('192.168.0.123')
    '192.168.0.0/16'
    >>> get_default_network_range('10.1.2.3')
    '10.0.0.0/24'
    >>> get_default_network_range('172.16.0.40')
    '172.16.0.0/20'
    """

    if not lan_ip:
        lan_ip = get_lan_ip()

    ip_is_in_reserved = False

    for i in IANA_RESERVED_NETWORK_RANGES:
        if ip_in_network_range(lan_ip, i):
            ip_is_in_reserved = True

    network_range = None
    if ip_is_in_reserved:
        if lan_ip.startswith("10."):
            network_range = "10.0.0.0/24"
        elif lan_ip.startswith("172.16."):
            network_range = "172.16.0.0/20"
        elif lan_ip.startswith("192.168."):
            network_range = "192.168.0.0/16"
    else:
        network_range = '127.0.0.1'

    return network_range

def ip_in_network_ranges(ip, network=None):
    """ Return True if `ip` is part of the network specify in network
    http://www.netfilter.org/documentation/HOWTO//networking-concepts-HOWTO-4.html

    >>> ip_in_network_ranges('127.0.0.1')
    True
    >>> ip_in_network_ranges('192.168.0.1')
    True
    >>> ip_in_network_ranges('10.0.0.1')
    True
    >>> ip_in_network_ranges('444.444.0.0')
    False
    >>> ip_in_network_ranges('192.168.0.1', network=('192.168.0.0/8',))
    True
    >>> ip_in_network_ranges('121.33.140.181', network=('192.168.0.0/8',))
    False
    """
    if not validipaddr(ip):
        return False

    if not network:
        network = ("0.0.0.0",)

    if network == ("0.0.0.0",):
        return True

    if ip == "127.0.0.1":
        return True

    for i in network:
        if not ip_in_network_range(ip, i):
            return False

    return True


# http://en.wikipedia.org/wiki/IPv4
CURRENT_NETWORK = "0.0.0.0/8"
PRIVATE_NETWORK_10 = "10.0.0.0/8"
SHARED_ADDRESS_SPACE = "100.64.0.0/10"
LOOPBACK = "127.0.0.0/8"
LINK_LOCAL = "169.254.0.0/16"
PRIVATE_NETWORK_172 = "172.16.0.0/12"
IETF_PROTOCOL_ASSIGN = "192.0.0.0/24"
TEST_NET_1 = "192.0.2.0/24"
IPv6_TO_IPv4 = "192.88.99.0/24"
PRIVATE_NETWORK_192 = "192.168.0.0/16"
NETWORK_BENCHMARK_TESTS = "198.18.0.0/15"
TEST_NET_2 = "198.51.100.0/24"
TEST_NET_3 = "203.0.113.0/24"
IP_MULTICAST = "224.0.0.0/4"
RESERVED = "240.0.0.0/4"
BROADCAST = "255.255.255.255"

INVALID_IP_ADDRESSES = (
    CURRENT_NETWORK,
    PRIVATE_NETWORK_10,
    SHARED_ADDRESS_SPACE,
    LOOPBACK,
    LINK_LOCAL,
    PRIVATE_NETWORK_172,
    IETF_PROTOCOL_ASSIGN,
    TEST_NET_1,
    IPv6_TO_IPv4,
    PRIVATE_NETWORK_192,
    NETWORK_BENCHMARK_TESTS,
    TEST_NET_2,
    TEST_NET_3,
    IP_MULTICAST,
    RESERVED,
    BROADCAST,
)

def ip_in_network_range(ip, network_range):
    return ip in iptools.IpRange(network_range)

def get_my_ip_addr():
    headers = {'User-Agent': 'curl/7.19.7 (universal-apple-darwin10.0) libcurl/7.19.7 OpenSSL/0.9.8r zlib/1.2.3', 'Accept': '*/*'}
    my_ip_addr = requests.get('http://ifconfig.me', headers=headers).text.strip()
    return my_ip_addr

if __name__ == "__main__":
    import doctest
    doctest.testmod()
