from __future__ import unicode_literals

import logging
import os

import tornado.locale
import tornado.web

_DATA_DIR = os.path.join(os.path.dirname(__file__), 'data')
_LOCALE_DIR = os.path.join(_DATA_DIR, 'locale')
_STATIC_DIR = os.path.join(_DATA_DIR, 'static')
_TEMPLATE_DIR = os.path.join(_DATA_DIR, 'template')

logger = logging.getLogger(__name__)


class IndexHandler(tornado.web.RequestHandler):

    DOMAIN = 'kuechenradio'

    def initialize(self, config):
        from . import Extension
        self._template_kwargs = {
            'title': config[Extension.ext_name]['title'],
            'lc_selected': self._lc_selected
        }
        self._default_locale = config[Extension.ext_name]['default_locale']
        tornado.locale.load_gettext_translations(_LOCALE_DIR, self.DOMAIN)

    def get(self, path):
        return self.render('index.html', **self._template_kwargs)

    def post(self, path):
        locale = self.get_argument('locale', None)
        if locale:
            self.set_cookie("locale", locale, max_age=365*86400)
        return self.render('index.html', **self._template_kwargs)

    def get_user_locale(self):
        code = self.get_argument('locale', self.get_cookie('locale'))
        if code:
            locale = tornado.locale.get(code)
        else:
            locale = self.get_browser_locale(self._default_locale)
        return locale

    def get_template_path(self):
        return _TEMPLATE_DIR

    def _lc_selected(self, code):
        # FIXME: matches 'en' if user/default locale is 'en_US'
        if self.get_user_locale().code.startswith(code):
            return 'selected'
        else:
            return ''


def factory(config, core):
    return [
        (r'/(index.html)?', IndexHandler, {'config': config}),
        (r'/(.*)', tornado.web.StaticFileHandler, {'path': _STATIC_DIR})
    ]
