import logging
from numbers import Number

import bson
import inflection


PRIMITIVES = (basestring, Number, bson.ObjectId)

class _Relationship(object):
    TYPE = None # 1 = has many, 2 = belongs to, 3 = belongs to many

    HAS_MANY = 1
    BELONGS_TO = 2
    BELONGS_TO_MANY = 3

    @classmethod
    def parse_attrs(cls, active, attrs, key=None):
        if not attrs:
            raise Exception('No attributes given')

        if isinstance(attrs, basestring):
            attrs = (attrs,)

        c = active.__class__ if active.__class__ != type else active
        if cls.TYPE == cls.HAS_MANY:
            foreign_key = '{0}_id'.format(c.__name__.lower()) if len(attrs) <= 1 else attrs[1]
            key2 = '_id' if len(attrs) <= 2 else attrs[2]
        else:
            if len(attrs) <= 1:
                key2 = '{0}_id'.format(inflection.singularize(key) if cls.TYPE == cls.BELONGS_TO else inflection.pluralize(key))
            else:
                key2 = attrs[1]
            foreign_key = '_id' if len(attrs) <= 2 else attrs[2]

        related = cls._import(active.PATH_TO_MODELS, attrs[0])
        return related, foreign_key, key2

    @classmethod
    def load(cls, active, key, attrs):
        related, foreign_key, key2 = cls.parse_attrs(active, attrs, key)

        if cls.TYPE == cls.HAS_MANY:
            _k = '_id' if len(attrs) <= 2 else attrs[2]
            value = active
            for _key in _k.split('.'):
                if hasattr(value, 'COLLECTION'):
                    value = value.__getattr__(_key, raw=True)
                else:
                    value = value.get(_key)
        else:
            value = active
            for _key in key2.split('.'):
                if hasattr(value, 'COLLECTION'):
                    value = value.__getattr__(_key, raw=True, default=[] if cls.TYPE == cls.BELONGS_TO_MANY else '')
                else:
                    value = value.get(_key)
            if cls.TYPE == cls.BELONGS_TO_MANY:
                value = {'$in': value}

        query = {foreign_key: value}
        if not value:
            return None

        data = related.find_one_by_(foreign_key, value) if cls.TYPE == cls.BELONGS_TO else related.find_by_(foreign_key, value)

        if not data:
            return None

        return cls(data, active, related, key, foreign_key)

    @classmethod
    def _import(cls, path, module_name):
        if '/' in module_name:
            mod = module_name.split('/')
            module_name = mod[-1]
            pth = '.'.join([c.lower() for c in mod])
        else:
            pth = module_name.lower()

        module = __import__('.'.join([path, pth]), fromlist=[module_name])
        return getattr(module, module_name)

    def __init__(self, data, active, related, key, foreign_key):
        self._rel__data = data
        self._rel__active = active
        self._rel__related = related
        self._rel__key = key
        self._rel__foreign_key = foreign_key


class _Multi(object):
    def __iter__(self):
        return iter(self._rel__data)

    def __len__(self):
        return len(self._rel__data)

    def __contains__(self, v):
        if not isinstance(v, PRIMITIVES):
            v = v.id
        return v in self.keys()

    def __getitem__(self, i):
        return self._rel__data.__getitem__(i)

    def iteritems(self):
        for x in self:
            yield x.id, x

    def keys(self):
        return [x.id for x in self]

    def values(self):
        return [x for x in self]



class HasMany(_Relationship, _Multi):
    ''' Class to represent *has many* relationships.
    '''

    TYPE = 1

    def append(self, val):
        ''' Append a new object to the list of objects in a *has many* relationship.
        
        Thie will create the relationship between the active object and the related object.
        Since a *has many* relationship is essentially a list of objects, this *appends*
        a new object to that list.
        
        Example: Add a new post to to a user to create the relationship.
        
        :param val: The value to append. Can be an instance of a model or a primitive type (string, int, etc.)
        
        .. note:: This automatically commits the change to the database.
        '''

        related_obj = self._rel__related.find_by_id(val) if isinstance(val, PRIMITIVES) else val
        if related_obj:
            setattr(related_obj, self._foreign_key, self._rel__active.id)
            related_obj.save()
        # TODO: add to self._data (cursor)

    def remove(self, val):
        ''' Removes the relationship between 2 objects. Since a *has many* relationshipis essentially
        a list of objects, this will remove an object from that list.
        
        Example: Remove a post from a users list of owned posts.
        
        :param val: The value to remove. Can be an instance of a model or a primitive type (string, int, etc.)
        
        .. note:: This automatically commits the change.
        '''

        related_obj = self._rel__related.find_by_id(val) if isinstance(val, PRIMITIVES) else val
        if related_obj:
            delattr(related_obj, self._rel__foreign_key)
            related_obj.save()


class BelongsTo(_Relationship):
    ''' Class to represent *belongs to* relationships.
    '''

    TYPE = 2

    def __getattr__(self, k):
        if k.startswith('_rel_'):
            return self.__dict__[k]
        return getattr(self._rel__data, k)

    def __setattr__(self, k, v):
        if k.startswith('_rel_'):
            self.__dict__[k] = v
        else:
            setattr(self._rel__data, k, v)


class BelongsToMany(_Relationship, _Multi):
    ''' Class to represent *belongs to many* relationships.
    '''

    TYPE = 3

    def append(self, val):
        ''' Append a new object to the list of objects in a *belongs to many* relationship.
        
        Thie will create the relationship between the active object and the related object.
        Since a *belongs to many* relationship is essentially a list of objects, this *appends*
        a new object to that list.
        
        Example: Add a new tag to to a post to create the relationship.
        
        :param val: The value to append. Can be an instance of a model or a primitive type (string, int, etc.)
        '''

        if not isinstance(val, PRIMITIVES):
            val = val.__getattr__(self._rel__foreign_key, raw=True)

        fin = self._rel__active.__getattr__(self._rel__key, raw=True, default=[])
        fin.append(val)
        setattr(self._rel__active, self._rel__key, fin)
        # TODO: add to self._data (cursor)

    def remove(self, val):
        ''' Removes the relationship between 2 objects. Since a *belongs to many* relationshipis essentially
        a list of objects, this will remove an object from that list.
        
        Example: Remove a tag from a post's list of tags.
        
        :param val: The value to remove. Can be an instance of a model or a primitive type (string, int, etc.)
        '''

        if not isinstance(val, PRIMITIVES):
            val = val.__getattr__(self._rel__foreign_key, raw=True)

        fin = self._rel__active.__getattr__(self._rel__key, raw=True, default=[])
        if val in fin:
            fin = [x for x in fin if x != val]
            setattr(self._rel__active, self._rel__key, fin)
        # TODO: add to self._data (cursor)

