import functools
from datetime import datetime
import logging

from inflection import pluralize
import bson
import pymongo

from relationship import HasMany, BelongsTo, BelongsToMany


class MCursor(object):
    def __init__(self, cls, mongo_cursor):
        ''' Wrapper object for PyMongo `Cursor`_ object.

        Anytime a :meth:`MObject.find` or :meth:`MObject.find_by_` query is done,
        this is returned.

        Simply put, it is a list of object loaded from the database and preloaded
        into instances of a model.
        
        :param class cls: The class to create instaces of.
        :param pymongo.Cursor mongo_cursor: the actual PyMongo cursor object.

        .. _Cursor: http://api.mongodb.org/python/current/api/pymongo/cursor.html
        '''
        self.__cls = cls
        self.__mongo_cursor = mongo_cursor
        self.__loaded = []
        self.__idx = 0
        self.__cursor_dead = False

    def __len__(self):
        return self.__mongo_cursor.count()

    def __iter__(self):
        self.__idx = 0
        return self

    def __getitem__(self, i):
        if self.__cursor_dead:
            return self.__loaded[i]

        if isinstance(i, int):
            if i < 0:
                i = self.__mongo_cursor.count() + (i + 1)
            if i > self.__idx and i >= len(self.__loaded):
                if i < self.__mongo_cursor.count():
                    while self.__idx <= i:
                        try:
                            o = self.cusor_next()
                        except:
                            break
                else:
                    # TODO: Make this a better error
                    raise KeyError()
            return self.__loaded[i]
        else:
            start = i.start
            if start is not None and start < 0:
                start = self.__mongo_cursor.count() + (start + 1)

            stop = i.stop
            if stop is not None and stop < 0:
                stop = self.__mongo_cursor.count() + (stop + 1)

            if stop < self.__idx and stop >= len(self.__loaded):
                while self.__idx <= i:
                    try:
                        o = self.cusor_next()
                    except:
                        break
            return self.__loaded[slice(start,stop,i.step)]

    def next(self):
        ''' Retrieve next item in iterator. Called when iterating over a :class:`MCursor` object.
        
        This method differes from :meth:`cursor_next` in that this method looks in both the
        cached list of loaded objects and if that is not the source of the next item, then
        calls :meth:`cursor_next`.
        
        :raises: :py:exc:`StopIteration`
        
        '''
        if self.__cursor_dead or len(self.__loaded) > self.__idx:
            if len(self.__loaded) <= self.__idx:
                self.__idx = 0
                self._die()
            self.__idx += 1
            return self.__loaded[self.__idx-1]
        return self.cursor_next()

    def cursor_next(self):
        ''' Retrieves the next item directly from the MongoDB cursor.
        
        :raises: :py:exc:`StopIteration`
        '''
        try:
            n = self.__mongo_cursor.next()
            o = self.__cls.load(**n)
            self.__loaded.append(o)
            self.__idx += 1
            return o
        except StopIteration:
            self._die()

    def _die(self):
        self.__cursor_dead = True
        self.__mongo_cursor.close()
        raise StopIteration()


class MList(list):
    def __init__(self, list_, parent, key):
        ''' Wrapper class for list objects within models.
        
        Because of how :class:`mormon.mormon.MObject` obfuscates the data contained
        within it, altering lists can be painful. By using these wrapper classes,
        it ensures that model attributes that contain lists can be edited just
        as you would expect without any extra work. This class also allows
        :class:`mormon.mormon.MObject` to keep track of changes and build the delta
        for doing updates.
        
        :param list list_: The list being wrapped
        :param object parent: The parent object of this list.
        :param string key: The key which this list is accessed on in `parent`
        '''
        self.__key = key
        self.__parent = parent
        list.__init__(self, list_)

    def append(self, value):
        ''' Append an item to this list.
        
        :param value: The value to append to the list
        '''
        if isinstance(value, MObject):
            value = value.id
        list.append(self, value)
        self._update_delta()

    def __getitem__(self, idx):
        result = dict.__getitem__(self, idx)
        if isinstance(result, list):
            result = MList(result, self, idx)
        if isinstance(result, dict):
            result = MDict(self, idx, **result)
        return result

    def __setitem__(self, idx, value):
        if isinstance(value, MObject):
            value = value.id
        list.__setitem__(self, idx, value)
        self._update_delta()

    def __delitem__(self, idx):
        list.__delitem__(self, idx)
        self._update_delta()

    def _update_delta(self):
        if isinstance(self.__parent, MObject):
            setattr(self.__parent, self.__key, list(self))
        else:
            self.__parent.__setitem__(self.__key, list(self))


class MDict(dict):
    def __init__(self, parent, key, **kwargs):
        ''' Wrapper class for dict objects within models.
        
        Because of how :class:`mormon.mormon.MObject` obfuscates the data contained
        within it, altering dicts can be painful. By using these wrapper classes,
        it ensures that model attributes that contain dicts can be edited just
        as you would expect without any extra work. This class also allows
        :class:`mormon.mormon.MObject` to keep track of changes and build the delta
        for doing updates.
        
        :param object parent: The parent object of this dict.
        :param string key: The key which this dict is accessed on in `parent`
        :param kwargs: All items of the dict are passed as keyword arguments.
        '''

        self.__parent = parent
        self.__key = key
        dict.__init__(self, **kwargs)

    def __setitem__(self, key, value):
        if isinstance(value, MObject):
            value = value.id
        dict.__setitem__(self, key, value)
        self._update_delta()

    def __getitem__(self, key):
        result = dict.__getitem__(self, key)
        if isinstance(result, list):
            result = MList(result, self, key)
        if isinstance(result, dict):
            result = MDict(self, key, **result)
        return result

    def get(self, key, default=None):
        result = dict.get(self, key, default)
        if isinstance(result, list):
            result = MList(result, self, key)
        if isinstance(result, MDict):
            result = MDict(self, key, **result)
        return result

    def _update_delta(self):
        if isinstance(self.__parent, MObject):
            setattr(self.__parent, self.__key, dict(self))
        else:
            self.__parent.__setitem__(self.__key, dict(self))


class MObject(object):
    ''' The base object of all models.
    '''

    PATH_TO_MODELS = 'lib'
    ''' Override this to specify a different location for your models.
    The default is "lib" and assumes your models are stored in a
    package at the top level of your project called "lib". For
    instance, the path to the "user" model in the default settings
    would be "/lib/user.py" and the import path would be "lib.user.User"
    '''

    COLLECTION = None
    ''' Set this value on an individual model to override the default collection
    name.
    '''

    MONGO_ID = True
    ''' Does this object use an ObjectId for the _id field? '''

    USE_DATE_CREATED = True
    ''' Automatically add a `date_created` field to newly created objects '''

    USE_DATE_MODIFIED = True
    ''' Automatically add and update a `date_modified` field when updating objects '''

    DEFAULT_VALUES = {}
    ''' ``dict`` of default values for instance attributes. For instance
    if you had a ``User`` object and wanted the user's name to default to
    'John', you would set ``DEFAULT_VALUES`` to::

        DEFAULT_VALUES = {
            'name': 'John'
        }
    
    Now, when you ``name`` on an instance of ``User``, if the name is not already
    set, you will get 'John'.
    '''

    DEFAULT_FILTER = {}
    ''' Default filter to apply to all databse queries. For instance
    if you want to automatically filter all documents that have been
    soft deleted you would use the following filter::
    
        DEFAULT_FILTER = {
            'deleted': True
        }
    '''

    # Relationship Types
    HAS_MANY = {}
    ''' ``dict`` that defines what ``HasMany`` relationships the model has.
    '''

    BELONGS_TO = {}
    ''' ``dict`` that defines what ``BelongsTo`` relationships the model has.
    '''

    BELONGS_TO_MANY = {}
    ''' ``dict`` that defines what ``BelongsToMany`` relationships the model has.
    '''

    def __init__(self, data=None, is_new=True, **kwargs):
        """ Create an instance of an ``MObject``.

        Attributes for the new object can be defined by passing in a ``dict``
        as the `data` parameter or by using keyword arguments.

        :param dict data: dictionary of attribute for the object
        :param bool is_new: is the object new (i.e. not loaded from the database)
        """

        self._m__data = {}
        self._m__delta = {}
        self._m__is_new = True
        self._m__relations = {}

        if data:
            for k, v in data.iteritems():
                setattr(self, k, v)
        if kwargs:
            for k, v in kwargs.iteritems():
                setattr(self, k, v)

        self._m__is_new = is_new

    def intialize(self):
        """ Called after the ``__init__`` method. Override this
        in your models to do any preprocessing you want at load time.
        """
        pass


    def __delattr__(self, k):
        if k in self.BELONGS_TO:
            # TODO: should we actually delete the owner object?
            related, foreign_key, key2 = BelongsTo.parse_attrs(self, self.BELONGS_TO[k], k)
            k = key2

        # TODO: catch KeyError
        del self._m__data[k]

        if '$unset' not in self._m__delta:
            self._m__delta['$unset'] = {}
        self._m__delta['$unset'][k] = True

        if '$set' in self._m__delta:
            if k in self._m__delta['$set']:
                del self._m__delta['$set'][k]
                if not self._m__delta['$set']:
                    del self._m__delta['$set']

    def __setattr__(self, k, v):
        if k.startswith('_m__'):
            self.__dict__[k] = v
            return True

        if k == 'id':
            k = '_id'

        # TODO: move this to the BelongsTo class
        if k in self.BELONGS_TO:
            related, foreign_key, key2 = BelongsTo.parse_attrs(self, self.BELONGS_TO[k], k)
            if isinstance(v, BelongsTo):
                v = v._rel__data
            if isinstance(v, MObject):
                self._m__relations[k] = BelongsTo(v, self, related, k, foreign_key)
                k = key2
                v = getattr(v, foreign_key)

        if isinstance(v, MList):
            v = list(v)
        if isinstance(v, MDict):
            v = dict(v)
        self._m__data[k] = v

        if not self._m__is_new:
            if '$set' not in self._m__delta:
                self._m__delta['$set'] = {}
            self._m__delta['$set'][k] = v

            if '$unset' in self._m__delta:
                if k in self._m__delta['$unset']:
                    del self._m__deleta['$unset'][k]
                    if not self._m__delta['$unset']:
                        del self._m__delta['$unset']

    def __getattr__(self, k, raw=False, default=None):
        if k.startswith('_m__'):
            return self.__dict__[k]

        if k == 'id':
            k = '_id'

        raw_k = None
        if not raw:
            raw_k = k
            if k in self.BELONGS_TO and (k not in self._m__relations or not isinstance(self._m__relations[k], BelongsTo)):
                self._m__relations[k] = BelongsTo.load(self, k, self.BELONGS_TO[k])
            elif k in self.HAS_MANY and (k not in self._m__relations or not isinstance(self._m__relations[k], HasMany)):
                self._m__relations[k] = HasMany.load(self, k, self.HAS_MANY[k])
            elif k in self.BELONGS_TO_MANY and (k not in self._m__relations or not isinstance(self._m__relations[k], BelongsToMany)):
                self._m__relations[k] = BelongsToMany.load(self, k, self.BELONGS_TO_MANY[k])

        if k in self.DEFAULT_VALUES and default is None:
            default = self.DEFAULT_VALUES[k]

        if not raw and k in self._m__relations:
            val = self._m__relations[k]
        else:
            val = self._m__data.get(k, default)

        if isinstance(val, list):
            val = MList(val, self, k)
        if isinstance(val, dict):
            val = MDict(self, k, **val)

        return val

    def rawdata(self):
        ''' Returns the real data that the model contains.
        
        :rtype: dict
        '''
        return self._m__data

    def delta(self):
        ''' When making changes to an object, a delta is kept so that when
        :meth:`save` or :meth:`update` is called, only the attributes that were
        changed get updated rather than overwriting the entire document. The
        delta is in the form of a MongoDB `update doc`_ completely with modifiers.
        
        :rtype: dict
        
        .. _update doc: http://docs.mongodb.org/manual/core/update/
        '''
        return self._m__delta

    def update(self, **kwargs):
        ''' Update the object in MongoDB.
        
        If the object is new (was not loaded from database and has not
        been saved yet), :meth:`save` method will be called instead.
        
        Whenever an object is saved, the `date_modified` attribute is set
        to the current timestamp.
        
        If the object has not been altered from its previously saved point,
        nothing will be done.
        
        :Parameters:
            - Any keyword arguments passed will be passed on to the MongoDB `update()`_ method.

        :Returns:
            - Whatever is returned by the MongoDB `update()`_ method.

        .. _update(): http://api.mongodb.org/python/current/api/pymongo/collection.html#pymongo.collection.Collection.update

        '''

        if self._m__is_new:
            return self.save(**kwargs)

        query = {'_id': self.id}
        if not self._m__delta:
            return True
        doc = self._m__delta

        if self.USE_DATE_MODIFIED:
            if '$set' not in doc:
                doc['$set'] = {}
            doc['$set']['date_modified'] = datetime.now()

        collection = self._get_collection()
        response = collection.update(query, doc, **kwargs)

        # TODO: Check if a safe write and only clear delta if success
        self._m__delta = {}
        return response

    def save(self, **kwargs):
        '''  Save an object to the database.
        
        If the object is not new (was loaded from the database or already saved),
        then :meth:`update` is called.
        
        When first saved, `date_created` and `date_modified` are both set to the current
        timestamp.
        
        :Parameters:
            - Any keyword argument passed will be passed on to the MongoDB `save()`_ method.

        :Returns:
            - Whatever is returned by MongoDB `save()`_ method.
        
        .. _save(): http://api.mongodb.org/python/current/api/pymongo/collection.html#pymongo.collection.Collection.save
        '''

        if not self._m__is_new:
            return self.update(**kwargs)

        if '_id' not in self._m__data:
            if not self.MONGO_ID:
                raise Exception('Model {0} does not use ObjectId for _id so _id should be explicitly set.')
            self.id = bson.ObjectId()

        now = datetime.now()
        if self.USE_DATE_CREATED:
            self.date_created = now
        if self.USE_DATE_MODIFIED:
            self.date_modified = now

        collection = self._get_collection()
        success = collection.save(self._m__data)
        if success:
            self._m__is_new = False
        return success

    @classmethod
    def load(cls, **kwargs):
        """ Method called when objects are loaded from the database.

        ONLY called for objects loaded from the database. NOT for newly created objects.
        See :meth:`create` method.

        :param dict kwargs: Properties of the newly loaded object.
        """
        return cls(is_new=False, **kwargs)

    @classmethod
    def create(cls, **kwargs):
        ''' Create an automatically save a new instance of a model.
        
        :Parameters:
            - All keywords are are set as attributes on the new instance.
        
        :Returns:
            - A new instance of `cls`
        '''

        o = cls(data=kwargs)
        o.save()
        return o

    @classmethod
    def find(cls, query, **kwargs):
        ''' Query MongoDB to find multiple objects.
        
        :param dict query: a MongoDB query
        :rtype: :class:`~mormon.mormon.MCursor` OR None
        '''

        return cls._find(query, **kwargs)

    @classmethod
    def find_one(cls, query, **kwargs):
        ''' Query MongoDB to find a single object
        
        :param dict query: A MongoDB query
        :rtype: A new object of type `cls` or None
        '''

        kwargs['_one'] = True
        return cls._find(query, **kwargs)

    @classmethod
    def find_by_id(cls, id, **kwargs):
        ''' Find a single object by ID
        
        :param id: A value that corresponds to the ID of an object. **NOTE** MongoDB is strongly typed so ``'1'`` is not the same as ``1``. If your model used ``ObjectId``, this value will be tranformed into a `bson.ObjectId`_.
        :rtype: A new object of type ``cls`` or ``None``
        
        .. _bson.ObjectId: http://api.mongodb.org/python/current/api/bson/objectid.html#bson.objectid.ObjectId
        '''

        if cls.MONGO_ID and not isinstance(id, bson.ObjectId):
            id = bson.ObjectId(id)
        if 'filter' not in kwargs:
            kwargs['filter'] = False
        return cls.find_one_by_('_id', id, **kwargs)

    @classmethod
    def find_one_by_(cls, field, value, **kwargs):
        ''' Find one object by a custom field.
        
        Example (to get a user with the email address "john@doe.com")::
        
            >>> from lib.user import User
            >>> john = User.find_one_by_('email', 'john@doe.com')
        
        :param string field: A string that corresponds to a field name in your MongoDB schema.
        :param value: The value to search for.
        :retype: An instance of type ``cls`` or ``None``
        '''

        kwargs['_one'] = True
        return cls.find_by_(field, value, **kwargs)

    @classmethod
    def find_by_(cls, field, value, **kwargs):
        ''' Find multipe objects using a custom field.
        
        Example (to get all users named "john")::
        
            >>> from lib.user import User
            >>> users = User.find_by_('name', 'john')
        
        :param string field: A string that corresponds to a field name in your MongoDB schema.
        :param value:  The value to search for.
        :rtype: A :class:`~mormon.mormon.MCursor` object or ``None``
        '''

        query = dict()
        query[field] = value
        return cls._find(query, **kwargs)

    @classmethod
    def _find(cls, query, _one=False, **kwargs):
        collection = cls._get_collection()

        if kwargs.get('filter', True):
            new_query = dict(cls.DEFAULT_FILTER)
            new_query.update(query)
            query = new_query

        if _one:
            response = collection.find_one(query, **kwargs)
            if response:
                if not kwargs.get('raw'):
                    response = cls.load(**response)
                return response
        else:
            response = collection.find(query, **kwargs)
            if response:
                if 'raw' in kwargs:
                    return response
                return MCursor(cls, response)
        return None

    @classmethod
    def _get_collection(cls):
        if not cls.COLLECTION:
            collection = []
            first = True
            for i, char in enumerate(cls.__name__):
                if char.upper() == char and not first:
                    collection.append('_')
                collection.append(char.lower())
                first = False
            return cls.connection(pluralize(''.join(collection)))
        return cls.connection(cls.COLLECTION)

    