# Copyright (C) 2012 Vaadin Ltd. 
# Copyright (C) 2012 Richard Lincoln
# 
# Licensed under the Apache License, Version 2.0 (the "License"); 
# you may not use this file except in compliance with the License. 
# You may obtain a copy of the License at 
# 
#     http://www.apache.org/licenses/LICENSE-2.0 
# 
# Unless required by applicable law or agreed to in writing, software 
# distributed under the License is distributed on an "AS IS" BASIS, 
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
# See the License for the specific language governing permissions and 
# limitations under the License.

"""Interface for classes supporting registration of methods as event
receivers."""


class IMethodEventSource(object):
    """Interface for classes supporting registration of methods as event
    receivers.

    For more information on the inheritable event mechanism see the
    L{muntjac.event package documentation<muntjac.event>}.

    @author: Vaadin Ltd.
    @author: Richard Lincoln
    @version: 1.1.1
    """

    def addListener(self, eventType, obj, method):
        """Registers a new event listener with the specified activation method
        to listen events generated by this component. If the activation method
        does not have any arguments the event object will not be passed to it
        when it's called.

        For more information on the inheritable event mechanism see the
        L{muntjac.event package documentation<muntjac.event>}.

        @param eventType:
                   the type of the listened event. Events of this type or its
                   subclasses activate the listener.
        @param obj:
                   the object instance who owns the activation method.
        @param method:
                   the activation method or the name of the activation method.
        @raise ValueError:
                    unless C{method} has a match in C{object}
        """
        raise NotImplementedError


    def removeListener(self, eventType, obj, method):
        """Removes all registered listeners matching the given parameters.
        Since this method receives the event type and the listener object as
        parameters, it will unregister all C{object}'s methods that are
        registered to listen to events of type C{eventType} generated by this
        component.

        For more information on the inheritable event mechanism see the
        L{muntjac.event package documentation<muntjac.event>}.

        @param eventType:
                   the exact event type the C{object} listens to.
        @param obj:
                   the target object that has registered to listen to events
                   of type eventType with one or more methods.
        @param method:
                   the method owned by the target that's registered to listen
                   to events of type eventType. Or the name of the method owned
                   by C{target} that's registered to listen to events of type
                   C{eventType}.
        """
        raise NotImplementedError
