## Copyright 2009-2013 Luc Saffre
## This file is part of the Lino project.
## Lino is free software; you can redistribute it and/or modify 
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 3 of the License, or
## (at your option) any later version.
## Lino is distributed in the hope that it will be useful, 
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
## GNU General Public License for more details.
## You should have received a copy of the GNU General Public License
## along with Lino; if not, see <http://www.gnu.org/licenses/>.

"""
Generic support for :ref:`mldbc`.

This includes definition of *babel fields* in your Django Models 
as well as methods to access these fields.

Babel fields are fields defined using 
:class:`BabelCharField`
or
:class:`BabelTextField`.

Each babel field generates a series of normal CharFields (or TextFields) 
depending on your :attr:`languages <north.Site.languages>` setting.

Example::

  class Foo(models.Model):
      name = BabelCharField(_("Foo"), max_length=200)
      
      
This module also defines the model mixin :class:`BabelNamed`
      

"""

import logging
logger = logging.getLogger(__name__)

import sys
import locale
import datetime


from django.db import models
from django.conf import settings
from django.template import defaultfilters
from django.utils import translation
from django.utils.translation import get_language
from django.utils.translation import ugettext_lazy as _
from django.utils.translation import string_concat


from north import LANGUAGE_CODE_MAX_LENGTH

#~ from djangosite import Site


#~ LONG_DATE_FMT = {
  #~ # None: 'l, j F Y',
  #~ None: 'l, F j, Y',
  #~ 'en': 'l, F j, Y',
  #~ 'de': 'l, j. F Y',
  #~ 'nl': 'l, j. F Y',
  #~ 'fr': 'l j F Y',
  #~ 'et': 'l, j F Y.a.',
#~ }

#~ SHORT_DATE_FMT = {
  #~ None: 'Y-m-d',
  #~ 'en': 'Y-m-d',
  #~ 'de': 'd.m.Y',
  #~ 'nl': 'd.m.Y',
  #~ 'et': 'd.m.Y',
  #~ 'fr': 'd/m/Y',
#~ }


#~ def lc2locale(lang,country):
    #~ """
    #~ Convert a combination of `lang` and `country` codes to 
    #~ a platform-dependant locale setting to be used for 
    #~ :func:`locale.setlocale`.
    #~ Thanks to 
    #~ http://www.gossamer-threads.com/lists/python/bugs/721929
    #~ and
    #~ http://msdn.microsoft.com/en-us/library/hzz3tw78
    #~ """
    #~ if sys.platform == 'win32': # development server
        #~ if lang == 'fr':
            #~ if country == 'BE': return 'french-belgian'
            #~ return 'french'
        #~ if lang == 'de':
            #~ if country == 'BE': return 'german-belgian'
            #~ return 'german'
        #~ raise NotImplementedError("lc2locale(%r,%r)" % (lang,country))
    #~ else:
        #~ return lang+'_'+country
        

def monthname(n):
    d = datetime.date(2013,n,1)
    return defaultfilters.date(d,'F')

def dtomy(d):
    """
    "date to month/year" :
    return the specified date as a localized string of type 'June 2011'."""
    if d is None: return ''
    return defaultfilters.date(d,'F Y')

#~ def dtos(d):
    #~ "Return the specified date as a localized short string of type '15.06.2011'."
    #~ if d is None: return ''
    #~ return defaultfilters.date(d,SHORT_DATE_FMT[get_language()])
    #~ return d.strftime(SHORT_DATE_FMT[LANG])

from djangosite.dbutils import dtos
from djangosite.dbutils import dtosl
#~ def dtosl(d):
    #~ "Return the specified date as a localized long string of type 'Wednesday, May 4, 2011'."
    #~ if d is None: return ''
    #~ return defaultfilters.date(d,LONG_DATE_FMT[get_language()])
    #~ return d.strftime(LONG_DATE_FMT[LANG])
    

    
def set_language(lang):
    """
    Thin wrapper around `django.utils.translation`.
    Activate the given language. Calls deactivate if 
    the given language is `None` or :attr:`DEFAULT_LANGUAGE`.
    """
    #~ print "20111111 babel.set_language()", lang
    #~ if lang is None or lang == DEFAULT_LANGUAGE:
    if lang is not None:
        if not lang in settings.SITE.AVAILABLE_LANGUAGES:
            raise Exception("Invalid language %r" % lang)
    if lang is None or lang == "en":
        #~ locale.setlocale(locale.LC_ALL,'')
        translation.deactivate()
    else:
        #~ if not lang in AVAILABLE_LANGUAGES:
            #~ raise Exception(
              #~ "Cannot set language to %s: available languages are %s." % (
              #~ lang,AVAILABLE_LANGUAGES))
        translation.activate(lang)
        #~ country = settings.LANGUAGE_CODE[3:]
        #~ locale.setlocale(locale.LC_ALL,lc2locale(lang,country))
    
        #~ save_ls = locale.getlocale()
        #~ ls = lc2locale(lang,country)
        #~ ls = 'de-DE' # de_DE
        #~ print ls
        #~ logger.debug("appy.pod render %s -> %s using locale=%r",tpl,target,ls)
        #~ locale.setlocale(locale.LC_ALL,'')
    
    
def babelattr(obj,attrname,*args):
    """
    Return the value of the specified attribute `attrname` of `obj`,
    but if `obj` also has a multi-language version of that 
    attribute for the current language, then prefer this attribute's 
    value if it is not blank.
    
    This is to be used in multilingual document templates.

    For example in a document template of a Contract you may now use the following expression::

      babelattr(self.type,'name')

    When generating a Contract in french (:attr:`isip.Contract.language` is ``fr``), 
    the expression will return :attr:`isip.ContractType.name_fr` if this field is not blank. 
    Otherwise (if the contract's language is not ``fr``, 
    of if this contract's type's `name_fr` field is blank) 
    it returns :attr:`isip.ContractType.name`.

    Not tested for other field types than CHAR.
    
    """
    LANG = translation.get_language()
    if LANG is not None and LANG != settings.SITE.default_language():
        v = getattr(obj,attrname+"_"+LANG,None)
        if v:
            return v
    return getattr(obj,attrname,*args)
        
getattr_lang = babelattr
    
def contribute_to_class(field,cls,fieldclass,**kw):
    if cls._meta.abstract:
        return
    kw.update(blank=True)
    for lang in settings.SITE.BABEL_LANGS:
        kw.update(verbose_name=string_concat(field.verbose_name,' ('+lang+')'))
        newfield = fieldclass(**kw)
        #~ newfield._lino_babel_field = True 
        newfield._lino_babel_field = field.name # used by dbtools.get_data_elems
        cls.add_to_class(field.name + '_' + lang,newfield)

class BabelCharField(models.CharField):
    """
    Define a variable number of clones of the "master" field, 
    one for each language of your :attr:`djangosite.Site.languages`.
    """
        
    def contribute_to_class(self, cls, name):
        super(BabelCharField,self).contribute_to_class(cls, name)
        contribute_to_class(self,cls,models.CharField,
            max_length=self.max_length)
        #~ kw = dict()
        #~ kw.update(max_length=self.max_length)
        #~ kw.update(blank=True)
        #~ for lang in BABEL_LANGS:
            #~ kw.update(verbose_name=self.verbose_name + ' ('+lang+')')
            #~ newfield = models.CharField(**kw)
            #~ newfield._lino_babel_field = True # used by dbtools.get_data_elems
            #~ cls.add_to_class(self.name + '_' + lang,newfield)
            

class BabelTextField(models.TextField):
    """
    Define a variable number of clones of the "master" field, 
    one for each babel language.
    """
    def contribute_to_class(self, cls, name):
        super(BabelTextField,self).contribute_to_class(cls, name)
        contribute_to_class(self,cls,models.TextField)
            
        #~ if cls._meta.abstract:
            #~ return
        #~ kw = dict()
        #~ kw.update(format=self.textfield_format)
        #~ kw.update(blank=True)
        #~ for lang in BABEL_LANGS:
            #~ kw.update(verbose_name=self.verbose_name + ' ('+lang+')')
            #~ # newfield = models.TextField(**kw)
            #~ newfield = fields.RichTextField(**kw)
            #~ newfield._lino_babel_field = True # used by dbtools.get_data_elems
            #~ cls.add_to_class(self.name + '_' + lang,newfield)


def babel_values(name,**kw):
    """
    kw2names
    """
    #~ d = { name : kw.get(default_language())}
    d = dict()
    v = kw.get(settings.SITE.default_language())
    if v is not None:
        d[name] = v
    for lang in settings.SITE.BABEL_LANGS:
        v = kw.get(lang,None)
        if v is not None:
            d[name+'_'+lang] = v
    return d
    
#~ kw2field = babel_values

def args2kw(name,*args):
    """
    Takes the basename of a BabelField and the values for each language.
    Returns a `dict` mapping the actual fieldnames to their values.
    """
    assert len(args) == len(settings.SITE.AVAILABLE_LANGUAGES)
    kw = {name:args[0]}
    for i,lang in enumerate(settings.SITE.BABEL_LANGS):
        kw[name+'_'+lang] = args[i+1]
    return kw
        

def field2kw(obj,name):
    d = { settings.SITE.default_language() : getattr(obj,name) }
    for lang in settings.SITE.BABEL_LANGS:
        v = getattr(obj,name+'_'+lang)
        if v:
            d[lang] = v
    return d
  
def field2args(obj,name):
    l = [ getattr(obj,name) ]
    for lang in settings.SITE.BABEL_LANGS:
        l.append(getattr(obj,name+'_'+lang))
    return l
  

def babelitem(**v):
    lng = translation.get_language()
    #~ print lng,v
    #~ lng = LANG or DEFAULT_LANGUAGE
    if lng == settings.SITE.DEFAULT_LANGUAGE:
        return v.get(lng)
    x = v.get(lng,None)
    if x is None:
        x = v.get(settings.SITE.DEFAULT_LANGUAGE)
    return x
    
# babel_get(v) = babelitem(**v)
        
def babeldict_getitem(d,k):
    v = d.get(k,None)
    if v is not None:
        assert type(v) is dict
        return babelitem(**v)
        
        
#~ def unused_discover():
    #~ """
    #~ This would somehow have to be called 
    #~ *during* and not after model setup.
    #~ """
    #~ logger.debug("Discovering babel fields...")
    #~ for model in models.get_models():
        #~ babel_fields = getattr(model,'babel_fields',None)
        #~ if babel_fields:
            #~ for name in babel_fields:
                #~ add_babel_field(model,name)
                
                
                
class BabelNamed(models.Model):
    """
    Mixin for models that have a babel field `name` 
    (labelled "Description") for each language.
    """
    
    class Meta:
        abstract = True
        
    name = BabelCharField(max_length=200,verbose_name=_("Designation"))
    
    def __unicode__(self):
        return babelattr(self,'name')
    
            
                
class LanguageField(models.CharField):
    """
    A field that lets the user select 
    a language from the available babel languages.
    """
    def __init__(self, *args, **kw):
        defaults = dict(
            verbose_name=_("Language"),
            choices=settings.SITE.LANGUAGE_CHOICES,
            default=settings.SITE.DEFAULT_LANGUAGE,
            #~ default=get_language,
            max_length=LANGUAGE_CODE_MAX_LENGTH,
            )
        defaults.update(kw)
        models.CharField.__init__(self,*args, **defaults)

                
def run_with_language(lang,func):                
    """
    Selects the specified language `lang`, 
    calls the specified functon `func`,
    restores the previously selected language.
    """
    current_lang = get_language()
    set_language(lang)
    try:
        rv = func()
    except Exception:
        set_language(current_lang)
        raise
    set_language(current_lang)
    return rv
                
                
LOOKUP_OP = '__iexact'

def lookup_filter(fieldname,value,**kw):
    """
    Return a `models.Q` to be used if you want to search for a given 
    string in any of the languages for the given babel field.
    """
    kw[fieldname+LOOKUP_OP] = value
    #~ kw[fieldname] = value
    flt = models.Q(**kw)
    del kw[fieldname+LOOKUP_OP]
    for lng in settings.SITE.BABEL_LANGS:
        kw[fieldname+'_'+lng+LOOKUP_OP] = value
        #~ flt = flt | models.Q(**{self.lookup_field.name+'_'+lng+'__iexact': value})
        #~ flt = flt | models.Q(**{self.lookup_field.name+'_'+lng: value})
        flt = flt | models.Q(**kw)
        del kw[fieldname+'_'+lng+LOOKUP_OP]
    return flt
        