class Operator(object):

    def __init__(self, name, *args, **kwargs):
        self.name = name
        self.args = args
        self.index = kwargs.get('index', None)
        self.out = kwargs.get('out', None)

    def __str__(self):
        values = []
        if len(self.args) == 0:
            value = ''
        else:
            for arg in self.args:
                if isinstance(arg, str) or isinstance(arg, unicode):
                    values.append('"%s"' % arg)
                elif (isinstance(arg, int) or isinstance(arg, float)
                      or isinstance(arg, long)):
                    values.append('%s' % str(arg))
                else:
                    raise Exception('Unexpected type')
            value = ','.join(values)
        return '.%s(%s)' % (self.name, value)

    def __repr__(self):
        return '<gremlin script: %s>' % str(self)


class SplitOperator(Operator):

    def __init__(self, *pipes):
        from omak import Omak
        super(SplitOperator, self).__init__(
            'copySplit',
            *pipes,
            out=Omak
        )

    def __str__(self):
        values = []
        for arg in self.args:
            value = '_()%s' % str(arg)
            values.append(value)
        value = ' ,'.join(values)
        return '.%s(%s)' % (self.name, value)


class Property(Operator):

    def __str__(self):
        return '.%s' % self.name


class GetItem(Operator):

    def __init__(self, idx, out):
        super(GetItem, self).__init__('getItem', out=out)
        self.idx = idx

    def __str__(self):
        return '[%s]' % self.idx


class GetRange(Operator):

    def __init__(self, start, end, out):
        super(GetRange, self).__init__('getRange', out=out)
        self.start = start
        self.end = end

    def __str__(self):
        return '[%s..%s]' % (self.start, self.end)


class ClosureOperator(Operator):

    def __init__(self, name, *closures, **kwargs):
        out = kwargs.pop('out', None)
        super(ClosureOperator, self).__init__(name, out=out)
        self.closures = closures

    def __str__(self):
        closures = ''.join(['{%s}' % closure for closure in self.closures])
        return ".%s%s" % (self.name, closures)

class SortOperator(ClosureOperator):

    def __str__(self):
        closures = ''.join(['{%s}' % closure for closure in self.closures])
        return ".%s%s.toList()" % (self.name, closures)


class LoopOperator(Operator):

    def __init__(self, index, closure, emit, out):
        super(LoopOperator, self).__init__('loop %s' % index, out=out)
        self.index = index
        self.closure = closure
        self.emit = emit

    def __str__(self):
        if isinstance(self.index, str):
            index = '"%s"' % self.index
        else:
            index = self.index
        if self.emit:
            emit = 'true'
        else:
            emit = 'false'
        return '.loop(%s){%s}{%s}' % (index, self.closure, emit)


class BaseProperty(Operator):

    def __init__(self, name, out):
        super(BaseProperty, self).__init__(name, out=out)

    def __str__(self):
        return self.name
