"""
<name>Optical aperture</name>
<description>An aperture</description>
<icon>icons/aperture.svg</icon>
<priority>10</priority>
"""
import sys
import Orange
from PyQt4 import *
from PyQt4.Qt import *

from Orange.widgets import widget, settings, gui
import Orange.data

from orangecontrib.srw.util.OpticalElementAperture import OpticalElementAperture, Disc, Rectangle
from orangecontrib.srw.util.OpticalBeam import OpticalBeam

from orangecontrib.srw.drivers.srw.SRWDriver import SRWDriver
from orangecontrib.srw.widgets.drivers.DriverSettingsWidget import DriverSettingsWidget

class OpticalElementApertureWidget(widget.OWWidget):
    name = "Optical aperture"
    description = "Optical aperture"
    icon = "icons/aperture.svg"
    
    inputs  = [("Optical beam", OpticalBeam, "onOpticalBeam", widget.Multiple)]
    outputs = [("Optical beam", OpticalBeam)]
    
    value_cbb_aperture_type = settings.Setting(0)

    value_le_diameter = settings.Setting(1.0)
    value_le_width = settings.Setting(1.0)
    value_le_height = settings.Setting(1.0)
    
    value_le_driver_settings = settings.Setting("")
         
    def __init__(self, parent=None, signalManager=None):
        widget.OWWidget.__init__(self, parent, signalManager)

        self.__optical_aperture = OpticalElementAperture("aperture")
       

        geometries = [Disc().description(), Rectangle().description()]
        self.geometries_mapping = {0 : Disc(),
                                   1 : Rectangle()}
            
        self.cbb_geometry_type = gui.comboBox(self,
                                              self,
                                              "value_cbb_aperture_type",
                                              box=None,
                                              label = "Aperture type",
                                              items = geometries,
                                              callback=self.cbb_geometry_change)
       
        self.le_diameter = gui.lineEdit(self,
                                        self,
                                        "value_le_diameter",
                                        label="Diameter [m]",
                                        validator=QDoubleValidator(bottom=0.0))
        
        self.le_width = gui.lineEdit(self,
                                     self,
                                     "value_le_width",
                                     label="Width [m]",
                                     validator=QDoubleValidator(bottom=0.0))

        self.le_height = gui.lineEdit(self,
                                      self,
                                      "value_le_height",
                                      label="Height [m]",
                                      validator=QDoubleValidator(bottom=0.0))

        self.__driver_settings_widget = DriverSettingsWidget(self.__optical_aperture,
                                                             self,
                                                             "value_le_driver_settings")

        self.__optical_aperture.setOnSynchronize(self.synchronizeToOpticalElement)      
        self.__optical_aperture.setOnCalculationStart(self.calculationStart)      
        self.__optical_aperture.setOnCalculationFinished(self.calculationFinished)
        
        self._updateEnableStates()      

    def calculationStart(self):
        self.progressBarInit()
        self.progressBarSet(0)
        QApplication.processEvents()

    def calculationFinished(self):
        self.progressBarSet(100)
        QApplication.processEvents()

    def synchronizeToOpticalElement(self):
        aperture_type_index = int(self.value_cbb_aperture_type)
        aperture = self.geometries_mapping[aperture_type_index]
        self.__optical_aperture.setApertureType(aperture)

        if aperture == Disc():
            diameter = float(self.value_le_diameter)
            self.__optical_aperture.setDiameter(diameter)
        elif aperture == Rectangle():
            width = float(self.value_le_width)
            self.__optical_aperture.setWidth(width)
            height = float(self.value_le_height)
            self.__optical_aperture.setHeight(height)

        
    def onOpticalBeam(self, optical_beam, sender):   
        print(sender)    
        optical_beam.sender().addOutput(self.__optical_aperture)
        
        sender = OpticalBeam(self.__optical_aperture)
        self.send("Optical beam", sender)
    
    def _updateEnableStates(self):
        aperture_type_index = int(self.value_cbb_aperture_type)
        aperture = self.geometries_mapping[aperture_type_index]
        self.__optical_aperture.setApertureType(aperture)

        if aperture == Disc():
            self.le_diameter.setEnabled(True)            
            self.le_width.setEnabled(False)
            self.le_height.setEnabled(False)
        elif aperture == Rectangle():
            self.le_diameter.setEnabled(False)            
            self.le_width.setEnabled(True)
            self.le_height.setEnabled(True)
        
    
    def cbb_geometry_change(self):
        self._updateEnableStates()
    
if __name__=="__main__":
    appl = QApplication(sys.argv)
    ow = OpticalElementApertureWidget()
    ow.show()
    appl.exec_()
