"""
Abstract driver for all optical simulation software backend.
"""
from orangecontrib.srw.util.OpticalElementScreen import OpticalElementScreen
from orangecontrib.srw.util.OpticalElementSpace import OpticalElementSpace
from orangecontrib.srw.util.OpticalElementAperture import OpticalElementAperture
from orangecontrib.srw.util.OpticalElementLens import OpticalElementLens
from orangecontrib.srw.util.OpticalElementSource import OpticalElementSource


class AbstractDriver(object):
    def isDataUpToDate(self, optical_element, in_data, stored_data):
        if stored_data.inputHash() is None:
            return False
        
        is_up_to_date = (self.hashInputData(optical_element, in_data) == stored_data.inputHash())
        return is_up_to_date
    
    def hashInputData(self, optical_element, in_data):
        """
        Hashes the input data.
        """
        #raise Exception("Needs reimplementation")
        return None        

    def createData(self):
        """
        Factory method for driver data.
        """
        raise Exception("Needs reimplementation")
  
    def calculateData(self, optical_element, in_data):
        """
        Calculates output data from input data.
        """
        output_data = self._calculateData(optical_element, in_data)
        
        if in_data is not None:
            input_hash = self.hashInputData(optical_element, in_data)
        else:
            input_hash = None
            
        output_data.setInputHash(input_hash)
        return output_data

    def _calculateData(self, optical_element, in_data):
        """
        Calculates output data from input data.
        """
        print("Calculating: %s:%s" % (optical_element.elementTypename(),
                                      optical_element.name()))
        
        if isinstance(optical_element, OpticalElementSource):
            result = self._calculateDataSource(optical_element, in_data)
        elif isinstance(optical_element, OpticalElementSpace):
            result = self._calculateDataSpace(optical_element, in_data)
        elif isinstance(optical_element, OpticalElementAperture):
            result = self._calculateDataAperture(optical_element, in_data)
        elif isinstance(optical_element, OpticalElementLens):
            result = self._calculateDataLens(optical_element, in_data)
        elif isinstance(optical_element, OpticalElementScreen):
            result = self._calculateDataScreen(optical_element, in_data)
        else:
            raise Exception("Optical element %s not implemented" % str(in_data))
    
        return result
    
    def _calculateDataSource(self, optical_element, in_data):
        """
        Calculates output data from input data for a source.
        """
        raise Exception("Needs reimplementation")        

    def _calculateDataSpace(self, optical_element, in_data):
        """
        Calculates output data from input data for a drift space.
        """
        raise Exception("Needs reimplementation")        
    
    def _calculateDataAperture(self, optical_element, in_data):
        """
        Calculates output data from input data for a aperture.
        """
        raise Exception("Needs reimplementation")        
    
    def _calculateDataLens(self, optical_element, in_data):
        """
        Calculates output data from input data for a lens.
        """
        raise Exception("Needs reimplementation")            
    
    def _calculateDataScreen(self, optical_element, in_data):
        """
        Calculates output data from input data for a screen.
        """
        raise Exception("Needs reimplementation")   
    
    def driverSettings(self, optical_element):             
        """
        Returns the driver settings for the given optical element.
        """
        # For the moment like this. Rethink if  more strict "splitting", i.e. the type like calculateData, is necessary. 
        raise Exception("Needs reimplementation")   
    
    def calculateIntensity3D(self, in_data):
        """
        Calculates 3D intensity distribution.
        """
        raise Exception("Needs reimplementation")                

    def calculatePhase3D(self, in_data):
        """
        Calculates 3D phase distribution.
        """
        raise Exception("Needs reimplementation")                
