__author__ = 'labx'

import numpy, math, random, os, copy
import sys
from PyQt4.QtGui import QLabel, QMessageBox, QWidget, QGraphicsScene, QGraphicsView, QFrame, QFont, QPalette, QColor, QGridLayout
from PyQt4.QtCore import Qt

try:
    from Orange.widgets import gui

    import PyMca5.PyMcaGraph.PlotBackend as PlotBackend
    import PyMca5.PyMcaGraph.backends.MatplotlibBackend as MatplotlibBackend
    import PyMca5.PyMcaGui.plotting.MaskImageWidget as MaskImageWidget
    import PyMca5.PyMcaGui.plotting.PlotWindow as PlotWindow
    import PyMca5.PyMcaGui.plotting.PlotWidget as PlotWidget
    import PyMca5.PyMcaGui.plotting.RGBCorrelatorGraph as RGBCorrelatorGraph
    import PyMca5.PyMcaGui.PyMcaQt as PyMcaQt

    import matplotlib
    import matplotlib.pyplot as plt
    from matplotlib import cm
    from matplotlib import figure as matfig
    import pylab
except ImportError:
    print(sys.exc_info()[1])
    pass

import Shadow.ShadowTools as ST
import Shadow.ShadowToolsPrivate as stp
from Shadow.ShadowToolsPrivate import plotxy_Ticket as plotxy_Ticket



class ConfirmDialog(QMessageBox):
    def __init__(self, parent = None, message=""):
        super(ConfirmDialog, self).__init__(parent)

        self.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)
        self.setIcon(QMessageBox.Question)
        self.setText(message)

    @classmethod
    def confirmed(cls, parent=None, message="Confirm Action?"):
        return ConfirmDialog(parent, message).exec_() == QMessageBox.Ok


class ShadowGui():

    @classmethod
    def lineEdit(cls, widget, master, value, label=None, labelWidth=None,
             orientation='vertical', box=None, callback=None,
             valueType=str, validator=None, controlWidth=None,
             callbackOnType=False, focusInCallback=None,
             enterPlaceholder=False, **misc):

        lEdit = gui.lineEdit(widget, master, value, label, labelWidth, orientation, box, callback, valueType, validator, controlWidth, callbackOnType, focusInCallback, enterPlaceholder, **misc)

        if value:
            if (valueType != str):
                lEdit.setAlignment(Qt.AlignRight)

        return lEdit

    @classmethod
    def widgetBox(cls, widget, box=None, orientation='vertical', margin=None, spacing=4, height=None, width=None, **misc):

        box = gui.widgetBox(widget, box, orientation, margin, spacing, **misc)
        box.layout().setAlignment(Qt.AlignTop)

        if not height is None:
            box.setFixedHeight(height)
        if not width is None:
            box.setFixedWidth(width)

        return box

    @classmethod
    def tabWidget(cls, widget, height=None, width=None):
        tabWidget = gui.tabWidget(widget)

        if not height is None:
            tabWidget.setFixedHeight(height)
        if not width is None:
            tabWidget.setFixedWidth(width)

        return tabWidget

    @classmethod
    def createTabPage(cls, tabWidget, name, widgetToAdd=None, canScroll=False, height=None, width=None):

        tab = gui.createTabPage(tabWidget, name, widgetToAdd, canScroll)
        tab.layout().setAlignment(Qt.AlignTop)

        if not height is None:
            tab.setFixedHeight(height)
        if not width is None:
            tab.setFixedWidth(width)

        return tab

    @classmethod
    def checkNumber(cls, value, field_name):
        try:
            float(value)
        except ValueError:
            raise Exception(str(field_name) + " is not a number")

        return value

    @classmethod
    def checkPositiveNumber(cls, value, field_name):
        value = ShadowGui.checkNumber(value, field_name)
        if (value < 0): raise Exception(field_name + " should be >= 0")

        return value

    @classmethod
    def checkPositiveAngle(cls, value, field_name):
        value = ShadowGui.checkNumber(value, field_name)
        if value < 0 or value > 360: raise Exception(field_name + " should be between 0 and 360 deg")

        return value

    @classmethod
    def checkFile(cls, fileName):
        filePath = os.getcwd() + '/' + fileName

        if not os.path.exists(filePath):
            raise Exception("File " + fileName + " not existing")


class ShadowPlotData:
    intensity = 0.0
    total_number_of_rays = 0
    total_good_rays = 0
    total_lost_rays = 0
    fwhm_h = 0.0
    fwhm_v = 0.0

    def __init__(self, intensity = 0.0,
                 total_number_of_rays = 0,
                 total_good_rays = 0,
                 total_lost_rays = 0,
                 fwhm_h = 0.0,
                 fwhm_v = 0.0):
        self.intensity = intensity
        self.total_number_of_rays = total_number_of_rays
        self.total_good_rays = total_good_rays
        self.total_lost_rays = total_lost_rays
        self.fwhm_h = fwhm_h
        self.fwhm_v = fwhm_v



class ShadowPlot:

    #########################################################################################
    #
    # THIS SECTION CONTAINS CLASSES THAT INHERIT FROM PYMCA5 PLOTTING STRUCTURE.
    # THE MODIFY THE BEHAVIOUR ADDING THE MANAGEMENT OF THE ZOOM EVENTS,
    # WITHOUT MODIFING THE ORIGINAL CLASSES
    #
    # THIS IS A MAINTANANCE-RISKY IMPLEMENTATION: COMPLIANCY MUST BE CHECKED WHEN PYMCA IS
    # UPDATED
    #########################################################################################

    class ShadowMatplotlibGraph(MatplotlibBackend.MatplotlibGraph):
        def __init__(self, parent=None, **kw):
            super(ShadowPlot.ShadowMatplotlibGraph, self).__init__(parent, **kw)

            self.fig.canvas.mpl_connect('button_release_event', self.onMouseReleasedWithEvent)

        def onMouseReleasedWithEvent(self, event):

            event.inaxes = None # prevent "How can it be here???" to be written in stdout

            self.onMouseReleased(event)

            xmin, xmax = self.ax.get_xlim()
            ymin, ymax = self.ax.get_ylim()

            ddict = {}
            ddict['event'] = "zoomCompleted"
            ddict['xmin'] = xmin
            ddict['xmax'] = xmax
            ddict['ymin'] = ymin
            ddict['ymax'] = ymax
            self._callback(ddict)

    class ShadowMatplotlibBackend(MatplotlibBackend.MatplotlibBackend):

        def __init__(self, parent=None, **kw):
            self.graph = ShadowPlot.ShadowMatplotlibGraph(parent, **kw)
            self.ax2 = self.graph.ax2
            self.ax = self.graph.ax
            PlotBackend.PlotBackend.__init__(self, parent)
            self._parent = parent
            self._logX = False
            self._logY = False
            self.setZoomModeEnabled = self.graph.setZoomModeEnabled
            self.setDrawModeEnabled = self.graph.setDrawModeEnabled
            self.isZoomModeEnabled = self.graph.isZoomModeEnabled
            self.isDrawModeEnabled = self.graph.isDrawModeEnabled
            self.getDrawMode = self.graph.getDrawMode
            self._oldActiveCurve = None
            self._oldActiveCurveLegend = None
            # should one have two methods, for enable and for show
            self._rightAxisEnabled = False
            self.enableAxis('right', False)

    class ShadowRGBCorrelatorGraph(RGBCorrelatorGraph.RGBCorrelatorGraph):

        def __init__(self, parent = None, selection=False, aspect=True,
                     colormap=False,
                     imageicons=False, standalonesave=True, standalonezoom=True,
                     profileselection=False, polygon=False):
            PyMcaQt.QWidget.__init__(self, parent)
            self.mainLayout = PyMcaQt.QVBoxLayout(self)
            self.mainLayout.setContentsMargins(0, 0, 0, 0)
            self.mainLayout.setSpacing(0)
            self._keepDataAspectRatioFlag = False
            self._buildToolBar(selection, colormap, imageicons,
                               standalonesave,
                               standalonezoom=standalonezoom,
                               profileselection=profileselection,
                               aspect=aspect,
                               polygon=polygon)
            self.graph = PlotWidget.PlotWidget(self, backend=ShadowPlot.ShadowMatplotlibBackend, aspect=aspect)
            self.graph.setGraphXLabel("Column")
            self.graph.setGraphYLabel("Row")
            self.graph.setYAxisAutoScale(True)
            self.graph.setXAxisAutoScale(True)
            if profileselection:
                if len(self._pickerSelectionButtons):
                    self.graph.sigPlotSignal.connect(\
                        self._graphPolygonSignalReceived)
                    self._pickerSelectionWidthValue.valueChanged[int].connect( \
                                 self.setPickerSelectionWith)

            self.saveDirectory = os.getcwd()
            self.mainLayout.addWidget(self.graph)
            self.printPreview = RGBCorrelatorGraph.PyMcaPrintPreview.PyMcaPrintPreview(modal = 0)

    class ShadowMaskImageWidget(MaskImageWidget.MaskImageWidget):

        def _build(self, standalonesave, profileselection=False, polygon=False):
            self.mainLayout = RGBCorrelatorGraph.qt.QVBoxLayout(self)
            self.mainLayout.setContentsMargins(0, 0, 0, 0)
            self.mainLayout.setSpacing(0)
            if self._MaskImageWidget__useTab:
                self.mainTab = RGBCorrelatorGraph.qt.QTabWidget(self)
                self.graphWidget = ShadowPlot.ShadowRGBCorrelatorGraph(self,
                                                       selection = self._MaskImageWidget__selectionFlag,
                                                       colormap=True,
                                                       imageicons=self._MaskImageWidget__imageIconsFlag,
                                                       standalonesave=False,
                                                       standalonezoom=False,
                                                       aspect=self._MaskImageWidget__aspect,
                                                       profileselection=profileselection,
                                                       polygon=polygon)
                self.mainTab.addTab(self.graphWidget, 'IMAGES')
            else:
                self.graphWidget = ShadowPlot.ShadowRGBCorrelatorGraph(self,
                                                   selection =self._MaskImageWidget__selectionFlag,
                                                   colormap=True,
                                                   imageicons=self._MaskImageWidget__imageIconsFlag,
                                                   standalonesave=False,
                                                   standalonezoom=False,
                                                   profileselection=profileselection,
                                                   aspect=self._MaskImageWidget__aspect,
                                                   polygon=polygon)

            if self._maxNRois > 1:
                # multiple ROI control
                self._buildMultipleRois()
            else:
                self._roiTags=[1]

            #for easy compatibility with RGBCorrelatorGraph
            self.graph = self.graphWidget.graph
            if profileselection:
                self.graphWidget.sigProfileSignal.connect(self._profileSignalSlot)

            if standalonesave:
                self.buildStandaloneSaveMenu()

            self.graphWidget.zoomResetToolButton.clicked.connect(self._zoomResetSignal)
            self.graphWidget.graph.setDrawModeEnabled(False)
            self.graphWidget.graph.setZoomModeEnabled(True)
            if self._MaskImageWidget__selectionFlag:
                if self._MaskImageWidget__imageIconsFlag:
                    self.setSelectionMode(False)
                    self._toggleSelectionMode()
                    self.graphWidget.graph.setDrawModeEnabled(True,
                                                              shape="rectangle",
                                                              label="mask")
                else:
                    self.setSelectionMode(True)
                    self._toggleSelectionMode()
            if self._MaskImageWidget__useTab:
                self.mainLayout.addWidget(self.mainTab)
            else:
                self.mainLayout.addWidget(self.graphWidget)

        def _zoomResetSignal(self):
            super(ShadowPlot.ShadowMaskImageWidget, self)._zoomResetSignal()

            ddict = {}
            ddict['event'] = "zoomReset"
            self.emitMaskImageSignal(ddict)

    #########################################################################################
    #
    # WIDGET FOR DETAILED PLOT
    #
    #########################################################################################

    class DetailedPlotWidget(QWidget):
        
        intensity_field = ""
        total_rays_field = ""
        total_good_rays_field = ""
        total_lost_rays_field = ""
        fwhm_h_field = ""
        fwhm_v_field = ""

        def __init__(self, x_scale_factor = 1.0, y_scale_factor = 1.0):
            super(ShadowPlot.DetailedPlotWidget, self).__init__()

            self.plot_canvas = ShadowPlot.ShadowMaskImageWidget(colormap=False, selection=False, imageicons=False, aspect=False)
            self.plot_canvas.setDefaultColormap(6, False)
            self.plot_canvas.setMinimumHeight(350*y_scale_factor)
            self.plot_canvas.setMaximumHeight(350*y_scale_factor)
            self.plot_canvas.setMaximumWidth(550*x_scale_factor)
            self.plot_canvas.setMaximumWidth(550*x_scale_factor)

            self.plot_canvas.sigMaskImageWidgetSignal.connect(self.resetZoom)
            self.plot_canvas.graphWidget.graph.sigPlotSignal.connect(self.zoom)

            self.plot_upper_canvas = PlotWindow.PlotWindow(roi=False, control=False, position=False, plugins=False)
            self.plot_upper_canvas.setDefaultPlotLines(True)
            self.plot_upper_canvas.setActiveCurveColor(color='darkblue')
            self.plot_upper_canvas.setFixedHeight(180*y_scale_factor)
            self.plot_upper_canvas.toolBar.hide()

            self.plot_right_canvas = PlotWindow.PlotWindow(roi=False, control=False, position=False, plugins=False)
            self.plot_right_canvas.setDefaultPlotLines(True)
            self.plot_right_canvas.setActiveCurveColor(color='darkblue')
            self.plot_right_canvas.setFixedHeight(180*x_scale_factor)
            self.plot_right_canvas.setFixedWidth(330*y_scale_factor)
            self.plot_right_canvas.toolBar.hide()

            scene = QGraphicsScene()
            item = scene.addWidget(self.plot_right_canvas)
            item.rotate(90)

            view = QGraphicsView()
            view.setScene(scene)
            view.setFixedWidth(180*x_scale_factor)
            view.setFixedHeight(350*y_scale_factor)
            view.setFrameStyle(QFrame.NoFrame)
            view.setAlignment(Qt.AlignLeft | Qt.AlignVCenter)

            info_box = QWidget()
            info_box_inner=ShadowGui.widgetBox(info_box, "Info")
            info_box_inner.setFixedHeight(180*y_scale_factor)

            self.intensity = ShadowGui.lineEdit(info_box_inner, self, "intensity_field", "Intensity", tooltip="Intensity", labelWidth=110, valueType=str, orientation="horizontal")
            self.total_rays = ShadowGui.lineEdit(info_box_inner, self, "total_rays_field", "Total Rays", tooltip="Total Rays", labelWidth=110, valueType=str, orientation="horizontal")
            self.total_good_rays = ShadowGui.lineEdit(info_box_inner, self, "total_good_rays_field", "Total Good Rays", tooltip="Total Good Rays", labelWidth=110, valueType=str, orientation="horizontal")
            self.total_lost_rays = ShadowGui.lineEdit(info_box_inner, self, "total_lost_rays_field", "Total Lost Rays", tooltip="Total Lost Rays", labelWidth=110, valueType=str, orientation="horizontal")
            self.fwhm_h = ShadowGui.lineEdit(info_box_inner, self, "fwhm_h_field", "FWHM H", tooltip="FWHM", labelWidth=110, valueType=str, orientation="horizontal")
            self.fwhm_v = ShadowGui.lineEdit(info_box_inner, self, "fwhm_v_field", "FWHM V", tooltip="FWHM", labelWidth=110, valueType=str, orientation="horizontal")

            self.intensity.setReadOnly(True)
            font = QFont(self.intensity.font())
            font.setBold(True)
            self.intensity.setFont(font)
            palette = QPalette(self.intensity.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.intensity.setPalette(palette)

            self.total_rays.setReadOnly(True)
            font = QFont(self.total_rays.font())
            font.setBold(True)
            self.total_rays.setFont(font)
            palette = QPalette(self.intensity.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.total_rays.setPalette(palette)

            self.total_good_rays.setReadOnly(True)
            font = QFont(self.total_good_rays.font())
            font.setBold(True)
            self.total_good_rays.setFont(font)
            palette = QPalette(self.total_good_rays.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.total_good_rays.setPalette(palette)

            self.total_lost_rays.setReadOnly(True)
            font = QFont(self.total_lost_rays.font())
            font.setBold(True)
            self.total_lost_rays.setFont(font)
            palette = QPalette(self.total_lost_rays.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.total_lost_rays.setPalette(palette)

            self.fwhm_h.setReadOnly(True)
            font = QFont(self.intensity.font())
            font.setBold(True)
            self.fwhm_h.setFont(font)
            palette = QPalette(self.fwhm_h.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.fwhm_h.setPalette(palette)

            self.fwhm_v.setReadOnly(True)
            font = QFont(self.fwhm_v.font())
            font.setBold(True)
            self.fwhm_v.setFont(font)
            palette = QPalette(self.fwhm_v.palette())
            palette.setColor(QPalette.Text, QColor('dark blue'))
            palette.setColor(QPalette.Base, QColor(243, 240, 160))
            self.fwhm_v.setPalette(palette)

            layout = QGridLayout()

            layout.addWidget(self.plot_upper_canvas, 0, 0, 1, 1)
            layout.addWidget(info_box, 0, 1, 1, 1)
            layout.addWidget(self.plot_canvas, 1, 0, 1, 1)
            layout.addWidget(view, 1, 1, 1, 1, Qt.AlignBottom | Qt.AlignLeft)

            layout.setColumnMinimumWidth(0, 550*x_scale_factor)
            layout.setColumnMinimumWidth(1, 350*x_scale_factor)

            self.setLayout(layout)

        def manageResetZoomSignal(self):
            self.plot_upper_canvas.resetZoom()
            self.plot_right_canvas.resetZoom()
            self.plot_right_canvas._plot.ax.invert_xaxis()
            self.plot_right_canvas.replot()
    
        def manageZoomSignal(self, xmin, xmax, ymin, ymax):
            self.plot_upper_canvas.setGraphXLimits(xmin=xmin, xmax=xmax, replot=True)
            self.plot_right_canvas.setGraphXLimits(xmin=ymin, xmax=ymax, replot=True)
            self.plot_right_canvas._plot.ax.invert_xaxis()


            self.plot_right_canvas.replot()
    
        def zoom(self, ddict):
            if ddict['event'] == 'zoomCompleted':
                self.manageZoomSignal(ddict['xmin'], ddict['xmax'], ddict['ymin'], ddict['ymax'])
    
        def resetZoom(self, ddict):
            if ddict['event'] == 'zoomReset':
                self.manageResetZoomSignal()
                
                
        def plotxy(self, beam, var_x, var_y, title, xtitle, ytitle, xrange=None, yrange=None, nolost=1, nbins=100):
            plot_data = ShadowPlot.plotxy(self.plot_canvas,
                                          self.plot_upper_canvas,
                                          self.plot_right_canvas,
                                          beam, var_x, var_y, xrange, yrange, nolost=nolost, title=title, xtitle=xtitle, ytitle=ytitle, nbins=nbins)

            self.plot_upper_canvas._plot.ax.xaxis.get_label().set_color('white')
            self.plot_upper_canvas._plot.ax.xaxis.get_label().set_fontsize(1)
            for label in self.plot_upper_canvas._plot.ax.xaxis.get_ticklabels():
                label.set_color('white')
                label.set_fontsize(1)

            self.plot_right_canvas._plot.ax.xaxis.get_label().set_color('white')
            self.plot_right_canvas._plot.ax.xaxis.get_label().set_fontsize(1)
            for label in self.plot_right_canvas._plot.ax.xaxis.get_ticklabels():
                label.set_color('white')
                label.set_fontsize(1)

            self.plot_right_canvas._plot.ax.invert_xaxis()
            self.plot_right_canvas.replot()
    
            self.intensity.setText("{:5.4f}".format(plot_data.intensity))
            self.total_rays.setText(str(plot_data.total_number_of_rays))
            self.total_good_rays.setText(str(plot_data.total_good_rays))
            self.total_lost_rays.setText(str(plot_data.total_lost_rays))
            self.fwhm_h.setText("{:9.8f}".format(plot_data.fwhm_h))
            self.fwhm_v.setText("{:9.8f}".format(plot_data.fwhm_v))

        def clear(self):
            self.plot_canvas.setImageData(None)
            self.plot_upper_canvas.clear()
            self.plot_upper_canvas.clear()

    #########################################################################################


    @classmethod
    def plot_histo(cls, plot_window, beam, col, nolost, xrange, yrange,  ref, title, xtitle, ytitle, nbins = 100):

        try:
            stp.Histo1_CheckArg(beam,col,xrange,yrange,nbins,nolost,ref,0,title,xtitle,ytitle,1,1)
        except stp.ArgsError as e:
            raise e

        col = col-1

        if ref==1: ref = 23

        if ref==0:
            x, good_only = ST.getshcol(beam,(col+1,10))
            weight = numpy.ones(len(x))
        else:
            x, good_only, weight = ST.getshcol(beam,(col+1, 10, ref))

        if nolost==0:
            t = numpy.where(good_only!=-3299)
        if nolost==1:
            t = numpy.where(good_only==1.0)
        if nolost==2:
            t = numpy.where(good_only!=1.0)

        if len(t[0])==0:
            print ("no rays match the selection, the histogram will not be plotted")
            return

        if ref==0:
            histogram, bins = numpy.histogram(x[t], bins=nbins, range=xrange)
        else:
            if not ytitle is None:  ytitle = ytitle + ' % ' + (stp.getLabel(ref-1))[0]
            histogram, bins = numpy.histogram(x[t], bins=nbins, range=xrange, weights=weight[t])

        fwhm, tf, ti = stp.calcFWHM(histogram,bins[1]-bins[0])

        bins = bins - ((numpy.max(x)/nbins)*0.5)

        if yrange==None: yrange = [0.0, numpy.max(histogram)*1.1]

        if not xtitle is None: plot_window.setGraphXLabel(xtitle)
        if not ytitle is None: plot_window.setGraphYLabel(ytitle)

        plot_window.setDrawModeEnabled(True, 'rectangle')
        plot_window.setZoomModeEnabled(True)

        plot_window.addCurve(bins[1:], histogram, title, symbol=',', color='blue', replace=True) #'+', '^', ','

        return fwhm

    @classmethod
    def plotxy(cls, mask_image_widget, upper_plot, right_plot, beam, cols1, cols2, xrange, yrange, nolost, title, xtitle, ytitle, nbins=100):

        nbins=nbins+1
        nbins_h=100

        try:
            stp.plotxy_CheckArg(beam, cols1, cols2, nbins, nbins_h, 5, xrange, yrange, 1, title, xtitle, ytitle,1,1,6)
        except stp.ArgsError as e:
            raise e

        col1,col2,col3,col4 = ST.getshcol(beam,(cols1,cols2,10,23,))

        if xtitle==None: xtitle=(stp.getLabel(cols1-1))[0]
        if ytitle==None: ytitle=(stp.getLabel(cols2-1))[0]

        if nolost==0: t = numpy.where(col3!=-3299)
        if nolost==1: t = numpy.where(col3==1.0)
        if nolost==2: t = numpy.where(col3!=1.0)

        if xrange==None: xrange = stp.setGoodRange(col1[t])
        if yrange==None: yrange = stp.setGoodRange(col2[t])

        tx = numpy.where((col1>xrange[0])&(col1<xrange[1]))
        ty = numpy.where((col2>yrange[0])&(col2<yrange[1]))

        tf = set(list(t[0])) & set(list(tx[0])) & set(list(ty[0]))
        t = (numpy.array(sorted(list(tf))),)

        if len(t[0])==0:
            print ("no point selected")
            return None

        weight = col4

        grid = numpy.zeros(nbins*nbins).reshape(nbins, nbins)

        intensity = 0.0

        for i in t[0]:
          indY = stp.findIndex(col1[i], nbins, xrange[0], xrange[1])
          indX = stp.findIndex(col2[i], nbins, yrange[0], yrange[1])
          try:
            grid[indX][indY] = grid[indX][indY] + weight[i]
            intensity = intensity + weight[i]
          except IndexError:
            pass

        mask_image_widget.setWindowTitle(title)
        mask_image_widget.setXLabel(xtitle)
        mask_image_widget.setYLabel(ytitle)
        mask_image_widget.setImageData(grid, xScale=(xrange[0], (xrange[1]-xrange[0])/nbins), yScale=(yrange[0], (yrange[1]-yrange[0])/nbins))

        fwhm_h = ShadowPlot.plot_histo(upper_plot, beam, cols1, nolost, xrange, None,  1, " ", xtitle, "Rays", nbins=nbins)
        fwhm_v = ShadowPlot.plot_histo(right_plot, beam, cols2, nolost, yrange, None,  1, " ", ytitle, "Rays", nbins=nbins)

        total_number_of_rays = len(beam.rays)
        total_good_rays = len(beam.rays[numpy.where(beam.rays[:,9] == 1)])

        return ShadowPlotData(intensity = intensity,
                              total_number_of_rays = total_number_of_rays,
                              total_good_rays = total_good_rays,
                              total_lost_rays = total_number_of_rays-total_good_rays,
                              fwhm_h = fwhm_h,
                              fwhm_v = fwhm_v)

    @classmethod
    def plotxy_preview(cls, plot_window, beam, cols1, cols2, nolost=0, title='PLOTXY', xtitle=None, ytitle=None):
        try:
            stp.plotxy_CheckArg(beam, cols1, cols2, 25, 25, 5, None, None, nolost, title, None, None, 0, 0, 0)
        except stp.ArgsError as e:
            raise e
        col1,col2,col3 = ST.getshcol(beam,(cols1,cols2,10,))

        if xtitle==None: xtitle=(stp.getLabel(cols1-1))[0]
        if ytitle==None: ytitle=(stp.getLabel(cols2-1))[0]

        if nolost==0: t = numpy.where(col3!=-3299)
        if nolost==1: t = numpy.where(col3==1.0)
        if nolost==2: t = numpy.where(col3!=1.0)

        plot_window.addCurve(col1[t], col2[t], title, symbol='o', color='blue', replace=True) #'+', '^', ','
        plot_window.setGraphXLabel(xtitle)
        plot_window.setGraphYLabel(ytitle)
        plot_window.setDrawModeEnabled(True, 'rectangle')
        plot_window.setZoomModeEnabled(True)

    @classmethod
    def plotxy_preview_image(cls, beam, cols1, cols2, nbins=25, nbins_h=None, level=5, xrange=None, yrange=None, nolost=0, title='PLOTXY', xtitle=None, ytitle=None, noplot=0, calfwhm=0, contour=0):
          if nbins_h==None: nbins_h=nbins+1
          try:
            stp.plotxy_CheckArg(beam,cols1,cols2,nbins,nbins_h,level,xrange,yrange,nolost,title,xtitle,ytitle,noplot,calfwhm,contour)
          except stp.ArgsError as e:
            raise e
          plt.ioff()
          col1,col2,col3,col4 = ST.getshcol(beam,(cols1,cols2,10,23,))

          nbins=nbins+1
          if xtitle==None: xtitle=(stp.getLabel(cols1-1))[0]
          if ytitle==None: ytitle=(stp.getLabel(cols2-1))[0]

          if nolost==0: t = numpy.where(col3!=-3299)
          if nolost==1: t = numpy.where(col3==1.0)
          if nolost==2: t = numpy.where(col3!=1.0)

          if xrange==None: xrange = stp.setGoodRange(col1[t])
          if yrange==None: yrange = stp.setGoodRange(col2[t])

          tx = numpy.where((col1>xrange[0])&(col1<xrange[1]))
          ty = numpy.where((col2>yrange[0])&(col2<yrange[1]))

          tf = set(list(t[0])) & set(list(tx[0])) & set(list(ty[0]))
          t = (numpy.array(sorted(list(tf))),)
          if len(t[0])==0:
            print ("no point selected")
            return None

          figure = pylab.plt.figure()

          axScatter = figure.add_axes([0.1,0.1,0.8,0.8])
          axScatter.set_xlabel(xtitle)
          axScatter.set_ylabel(ytitle)

          if contour==0:
            axScatter.scatter(col1[t],col2[t],s=0.5)
          if contour>0 and contour<7:
            if contour==1 or contour==3 or contour==5: w = numpy.ones( len(col1) )
            if contour==2 or contour==4 or contour==6: w = col4
            grid = numpy.zeros(nbins*nbins).reshape(nbins,nbins)
            for i in t[0]:
              indX = stp.findIndex(col1[i],nbins,xrange[0],xrange[1])
              indY = stp.findIndex(col2[i],nbins,yrange[0],yrange[1])
              try:
                grid[indX][indY] = grid[indX][indY] + w[i]
              except IndexError:
                pass
            X, Y = numpy.mgrid[xrange[0]:xrange[1]:nbins*1.0j,yrange[0]:yrange[1]:nbins*1.0j]
            L = numpy.linspace(numpy.amin(grid),numpy.amax(grid),level)
            if contour==1 or contour==2: axScatter.contour(X, Y, grid, colors='k', levels=L)
            if contour==3 or contour==4: axScatter.contour(X, Y, grid, levels=L)
            if contour==5 or contour==6: axScatter.pcolor(X, Y, grid)

          for tt in axScatter.get_xticklabels():
            tt.set_size('x-small')
          for tt in axScatter.get_yticklabels():
            tt.set_size('x-small')

          pylab.plt.draw()

          ticket = plotxy_Ticket()
          ticket.figure = figure
          ticket.xrange = xrange
          ticket.yrange = yrange
          ticket.xtitle = xtitle
          ticket.ytitle = ytitle
          ticket.title = title
          ticket.intensity = col4[t].sum()
          ticket.averagex = numpy.average( col1[t] )
          ticket.averagey = numpy.average( col2[t] )
          ticket.intensityinslit = 0
          return ticket

class ShadowMath:

    @classmethod
    def vectorial_product(cls, vector1, vector2):
        result = [0, 0, 0]

        result[0] = vector1[1]*vector2[2] - vector1[2]*vector2[1]
        result[1] = -(vector1[0]*vector2[2] - vector1[2]*vector2[0])
        result[2] = vector1[0]*vector2[1] - vector1[1]*vector2[0]

        return result

    @classmethod
    def scalar_product(cls, vector1, vector2):
        return vector1[0]*vector2[0] + vector1[1]*vector2[1] + vector1[2]*vector2[2]

    @classmethod
    def vector_modulus(cls, vector):
        return math.sqrt(cls.scalar_product(vector, vector))

    @classmethod
    def vector_multiply(cls, vector, constant):
        result = [0, 0, 0]

        result[0] = vector[0] * constant
        result[1] = vector[1] * constant
        result[2] = vector[2] * constant

        return result

    @classmethod
    def vector_divide(cls, vector, constant):
        result = [0, 0, 0]

        result[0] = vector[0] / constant
        result[1] = vector[1] / constant
        result[2] = vector[2] / constant

        return result

    @classmethod
    def vector_normalize(cls, vector):
        return cls.vector_divide(vector, cls.vector_modulus(vector))

    @classmethod
    def vector_sum(cls, vector1, vector2):
        result = [0, 0, 0]

        result[0] = vector1[0] + vector2[0]
        result[1] = vector1[1] + vector2[1]
        result[2] = vector1[2] + vector2[2]

        return result

    @classmethod
    def vector_difference(cls, vector1, vector2):
        result = [0, 0, 0]

        result[0] = vector1[0] - vector2[0]
        result[1] = vector1[1] - vector2[1]
        result[2] = vector1[2] - vector2[2]

        return result

    @classmethod
    def point_distance(cls, point1, point2):
        return cls.vector_modulus(cls.vector_difference(point1, point2))


class ShadowPhysics:

    @classmethod
    def getWavelengthfromShadowK(cls, k_mod): # in cm
        return (2*math.pi/k_mod)*1e+8 # in Angstrom

    def getShadowKFromWavelength(cls, wavelength): # in A
        return (2*math.pi/wavelength)*1e+8 # in cm

    @classmethod
    def getWavelengthFromEnergy(cls, energy): #in eV
        return 12397.639/energy # in Angstrom

    @classmethod
    def getEnergyFromWavelength(cls, wavelength): # in Angstrom
        return 12397.639/wavelength # in eV

    @classmethod
    def calculateBraggAngle(cls, wavelength, h, k, l, a):

        #
        # k = 2 pi / lambda
        #
        # lambda = 2 pi / k = 2 d sen(th)
        #
        # sen(th) = lambda / (2  d)
        #
        # d = a/sqrt(h^2 + k^2 + l^2)
        #
        # sen(th) = (sqrt(h^2 + k^2 + l^2) * lambda)/(2 a)

        theta_bragg = -1

        argument = (wavelength*math.sqrt(h**2+k**2+l**2))/(2*a)

        if argument <=1:
            result = math.asin(argument)

            if result > 0:
                theta_bragg = result

        return theta_bragg

    @classmethod
    def Chebyshev(cls, n, x):
        if n==0: return 1
        elif n==1: return x
        else: return 2*x*cls.Chebyshev(n-1, x)-cls.Chebyshev(n-2, x)

    @classmethod
    def ChebyshevBackground(cls, coefficients=[0,0,0,0,0,0], twotheta=0):
        coefficients_set = range(0, len(coefficients))
        background = 0

        for index in coefficients_set:
            background += coefficients[index]*cls.Chebyshev(index, twotheta)

        return background

    @classmethod
    def ChebyshevBackgroundNoised(cls, coefficients=[0,0,0,0,0,0], twotheta=0.0, n_sigma=1.0, random_generator=random.Random()):
        background = cls.ChebyshevBackground(coefficients, twotheta)
        sigma = math.sqrt(background) # poisson statistic

        noise = (n_sigma*sigma)*random_generator.random()
        sign_marker = random_generator.random()

        if sign_marker > 0.5:
            return int(round(background+noise, 0))
        else:
            return int(round(background-noise, 0))

    @classmethod
    def ExpDecay(cls, h, x):
      return math.exp(-h*x)

    @classmethod
    def ExpDecayBackground(cls, coefficients=[0,0,0,0,0,0], decayparams=[0,0,0,0,0,0], twotheta=0):
        coefficients_set = range(0, len(coefficients))
        background = 0

        for index in coefficients_set:
            background += coefficients[index]*cls.ExpDecay(decayparams[index], twotheta)

        return background

    @classmethod
    def ExpDecayBackgroundNoised(cls, coefficients=[0,0,0,0,0,0], decayparams=[0,0,0,0,0,0], twotheta=0, n_sigma=1, random_generator=random.Random()):
        background = cls.ExpDecayBackground(coefficients, decayparams, twotheta)
        sigma = math.sqrt(background) # poisson statistic

        noise = (n_sigma*sigma)*random_generator.random()
        sign_marker = random_generator.random()

        if sign_marker > 0.5:
            return int(round(background+noise, 0))
        else:
            return int(round(background-noise, 0))

if __name__ == "__main__":

    print(ShadowPhysics.Chebyshev(4, 21))
    print(ShadowPhysics.Chebyshev(0, 35))

    coefficients = [5.530814e+002, 2.487256e+000, -2.004860e-001, 2.246427e-003, -1.044517e-005, 1.721576e-008]
    random_generator=random.Random()

    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 10, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 11, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 12, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 13, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 14, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 15, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 16, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 17, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 18, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 19, random_generator=random_generator))
    print(ShadowPhysics.ChebyshevBackgroundNoised(coefficients, 20, random_generator=random_generator))
