import sys

from Orange.widgets import gui
from Orange.widgets.settings import Setting
from PyQt4 import QtGui
from PyQt4.QtGui import QApplication, qApp, QPalette, QColor, QFont

from orangecontrib.shadow.widgets.gui import ow_generic_element
from orangecontrib.shadow.util import srfunc
from orangecontrib.shadow.util.shadow_objects import EmittingStream, TTYGrabber, ShadowTriggerOut, ShadowBeam, \
    ShadowSource
from orangecontrib.shadow.util.shadow_util import ShadowGui, ConfirmDialog

class Wiggler(ow_generic_element.GenericElement):

    NONE_SPECIFIED = "NONE SPECIFIED"

    name = "Wiggler"
    description = "Shadow Source: Wiggler"
    icon = "icons/wiggler.png"
    maintainer = "Luca Rebuffi"
    maintainer_email = "luca.rebuffi(@at@)elettra.eu"
    priority = 3
    category = "Sources"
    keywords = ["data", "file", "load", "read"]

    inputs = [("Trigger", ShadowTriggerOut, "sendNewBeam")]

    outputs = [{"name":"Beam",
                "type":ShadowBeam,
                "doc":"Shadow Beam",
                "id":"beam"}]

    number_of_rays=Setting(5000)
    seed=Setting(5676561)
    user_unit = 1
    e_min=Setting(5000)
    e_max=Setting(100000)
    optimize_source_combo=Setting(0)

    max_number_of_rejected_rays = Setting(10000000)

    slit_distance = Setting(1000.0)
    min_x = Setting(-1.0)
    max_x = Setting(1.0)
    min_z = Setting(-1.0)
    max_z = Setting(1.0)

    file_with_phase_space_volume = Setting(NONE_SPECIFIED)

    energy=Setting(6.04)
    use_emittances_combo=Setting(1)
    sigma_x=Setting(0.0078)
    sigma_z=Setting(0.0036)
    emittance_x=Setting(3.8E-7)
    emittance_z=Setting(3.8E-9)
    distance_from_waist_x=Setting(0.0)
    distance_from_waist_z=Setting(0.0)

    type_combo=Setting(0)

    number_of_periods=Setting(50)
    k_value=Setting(7.85)
    id_period=Setting(0.04)

    file_with_b_vs_y = Setting("wiggler.b")
    file_with_harmonics = Setting("wiggler.h")

    CONTROL_AREA_WIDTH = 505

    want_main_area=1

    def __init__(self):
        super().__init__(show_automatic_box=False)

        #self.controlArea.setFixedWidth(self.CONTROL_AREA_WIDTH)

        left_box_1 = ShadowGui.widgetBox(self.controlArea, "Monte Carlo and Energy Spectrum", addSpace=True, orientation="vertical", height=320, width=self.CONTROL_AREA_WIDTH)

        ShadowGui.lineEdit(left_box_1, self, "number_of_rays", "Number of Rays", tooltip="Number of Rays", labelWidth=300, valueType=int, orientation="horizontal")

        ShadowGui.lineEdit(left_box_1, self, "seed", "Seed", tooltip="Seed", labelWidth=300, valueType=int, orientation="horizontal")
        ShadowGui.lineEdit(left_box_1, self, "e_min", "Minimum Photon Energy (eV)", tooltip="Minimum Energy (eV)", labelWidth=300, valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(left_box_1, self, "e_max", "Maximum Photon Energy (eV)", tooltip="Maximum Energy (eV)", labelWidth=300, valueType=float, orientation="horizontal")

        gui.comboBox(left_box_1, self, "optimize_source_combo", label="Optimize Source? (reject rays)", items=["No", "Using file with phase space volume", "Using slit/acceptance"], callback=self.set_OptimizeSource, labelWidth=280, orientation="horizontal")

        self.box_using_file_with_phase_space_volume = ShadowGui.widgetBox(left_box_1, "", addSpace=False, orientation="vertical")

        ShadowGui.lineEdit(self.box_using_file_with_phase_space_volume, self, "max_number_of_rejected_rays", "Max number of rejected rays (set 0 for infinity)", labelWidth=300, tooltip="Max number of rejected rays", valueType=int, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_file_with_phase_space_volume, self, "file_with_phase_space_volume", "File with phase space volume", labelWidth=190, tooltip="File with phase space volume", valueType=str, orientation="horizontal")

        self.box_using_slit_acceptance = ShadowGui.widgetBox(left_box_1, "", addSpace=False, orientation="vertical")

        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "max_number_of_rejected_rays", "Max number of rejected rays (set 0 for infinity)", labelWidth=300, tooltip="Max number of rejected rays", valueType=int, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "slit_distance", "Slit Distance [cm] (set 0 for angular acceptance)", labelWidth=300, tooltip="Slit Distance [cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "min_x", "Min X [cm]/Min Xp [rad]", labelWidth=300, tooltip="Min X/Min Xp", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "max_x", "Max X [cm]/Max Xp [rad]", labelWidth=300, tooltip="Max X/Max Xp", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "min_z", "Min Z [cm]/Min Zp [rad]", labelWidth=300, tooltip="Min Z/Min Zp", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_using_slit_acceptance, self, "max_z", "Max Z [cm]/Max Zp [rad]", labelWidth=300, tooltip="Max Z/Max Zp", valueType=float, orientation="horizontal")

        self.set_OptimizeSource()

        left_box_2 = ShadowGui.widgetBox(self.controlArea, "Machine Parameters", addSpace=True, orientation="vertical", height=240)

        ShadowGui.lineEdit(left_box_2, self, "energy", "Electron Energy [GeV]", tooltip="Energy [GeV]", labelWidth=300, valueType=float, orientation="horizontal")

        gui.comboBox(left_box_2, self, "use_emittances_combo", label="Use Emittances?", items=["No", "Yes"], callback=self.set_UseEmittances, labelWidth=300, orientation="horizontal")

        self.box_use_emittances = ShadowGui.widgetBox(left_box_2, "", addSpace=True, orientation="vertical")

        ShadowGui.lineEdit(self.box_use_emittances, self, "sigma_x", "Sigma X [cm]", labelWidth=300, tooltip="Sigma X [cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_use_emittances, self, "sigma_z", "Sigma Z [cm]", labelWidth=300, tooltip="Sigma Z [cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_use_emittances, self, "emittance_x", "Emittance X [rad.cm]", labelWidth=300, tooltip="Emittance X [rad.cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_use_emittances, self, "emittance_z", "Emittance Z [rad.cm]", labelWidth=300, tooltip="Emittance Z [rad.cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_use_emittances, self, "distance_from_waist_x", "Distance from Waist X [cm]", labelWidth=300, tooltip="Distance from Waist X [cm]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.box_use_emittances, self, "distance_from_waist_z", "Distance from Waist Z [cm]", labelWidth=300, tooltip="Distance from Waist Z [cm]", valueType=float, orientation="horizontal")

        self.set_UseEmittances()

        left_box_3 = ShadowGui.widgetBox(self.controlArea, "Wiggler Parameters", addSpace=True, orientation="vertical", height=140)

        gui.comboBox(left_box_3, self, "type_combo", label="Type", items=["conventional/sinusoidal", "B from file", "B from harmonics"], callback=self.set_Type, labelWidth=300, orientation="horizontal")

        ShadowGui.lineEdit(left_box_3, self, "number_of_periods", "Number of Periods", labelWidth=300, tooltip="Number of Periods", valueType=int, orientation="horizontal")

        self.conventional_sinusoidal_box = ShadowGui.widgetBox(left_box_3, "", addSpace=False, orientation="vertical")

        ShadowGui.lineEdit(self.conventional_sinusoidal_box, self, "k_value", "K value", labelWidth=300, tooltip="K value", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.conventional_sinusoidal_box, self, "id_period", "ID period [m]", labelWidth=300, tooltip="ID period [m]", valueType=float, orientation="horizontal")

        self.b_from_file_box = ShadowGui.widgetBox(left_box_3, "", addSpace=False, orientation="vertical")

        ShadowGui.lineEdit(self.b_from_file_box, self, "file_with_b_vs_y", "File with B vs Y", labelWidth=150, tooltip="File with B vs Y", valueType=str, orientation="horizontal")

        self.b_from_harmonics_box = ShadowGui.widgetBox(left_box_3, "", addSpace=False, orientation="vertical")

        ShadowGui.lineEdit(self.b_from_harmonics_box, self, "id_period", "ID period [m]", labelWidth=300, tooltip="ID period [m]", valueType=float, orientation="horizontal")
        ShadowGui.lineEdit(self.b_from_harmonics_box, self, "file_with_harmonics", "File with harmonics", labelWidth=150, tooltip="File with harmonics", valueType=str, orientation="horizontal")

        self.set_Type()

        gui.separator(self.controlArea, height=10)

        button_box = ShadowGui.widgetBox(self.controlArea, "", addSpace=False, orientation="horizontal")

        button = gui.button(button_box, self, "Run Shadow/trace", callback=self.runShadowSource)
        font = QFont(button.font())
        font.setBold(True)
        button.setFont(font)
        palette = QPalette(button.palette()) # make a copy of the palette
        palette.setColor(QPalette.ButtonText, QColor('Dark Blue'))
        button.setPalette(palette) # assign new palette
        button.setFixedHeight(45)

        button = gui.button(button_box, self, "Reset Fields", callback=self.callResetSettings)
        font = QFont(button.font())
        font.setItalic(True)
        button.setFont(font)
        palette = QPalette(button.palette()) # make a copy of the palette
        palette.setColor(QPalette.ButtonText, QColor('Dark Red'))
        button.setPalette(palette) # assign new palette
        button.setFixedHeight(45)
        button.setFixedWidth(100)

        gui.rubber(self.controlArea)

        gui.rubber(self.mainArea)

    def set_OptimizeSource(self):
        self.box_using_file_with_phase_space_volume.setVisible(self.optimize_source_combo == 1)
        self.box_using_slit_acceptance.setVisible(self.optimize_source_combo == 2)

    def set_UseEmittances(self):
        self.box_use_emittances.setVisible(self.use_emittances_combo == 1)

    def set_Type(self):
        self.conventional_sinusoidal_box.setVisible(self.type_combo == 0)
        self.b_from_file_box.setVisible(self.type_combo == 1)
        self.b_from_harmonics_box.setVisible(self.type_combo == 2)

    def runShadowSource(self):
        try:
            self.error()
            self.progressBarInit()

            self.number_of_rays = ShadowGui.checkPositiveNumber(self.number_of_rays, "Number of rays")
            self.seed = ShadowGui.checkPositiveNumber(self.seed, "Seed")
            self.e_min = ShadowGui.checkPositiveNumber(self.e_min, "Minimum energy")
            self.e_max = ShadowGui.checkPositiveNumber(self.e_max, "Maximum energy")
            self.max_number_of_rejected_rays = ShadowGui.checkPositiveNumber(self.max_number_of_rejected_rays, "Max Number of Rejected Rays")

            self.slit_distance = ShadowGui.checkPositiveNumber(self.slit_distance, "Horizontal half-divergence from [+]")
            self.min_x = ShadowGui.checkNumber(self.min_x, "Min X/Min Xp")
            self.max_x = ShadowGui.checkNumber(self.max_x, "Max X/Max Xp")
            self.min_z = ShadowGui.checkNumber(self.min_z, "Min X/Min Xp")
            self.max_z = ShadowGui.checkNumber(self.max_z, "Max X/Max Xp")

            self.energy = ShadowGui.checkPositiveNumber(self.energy, "Energy")
            self.sigma_x = ShadowGui.checkPositiveNumber(self.sigma_x, "Sigma x")
            self.sigma_z = ShadowGui.checkPositiveNumber(self.sigma_z, "Sigma z")
            self.emittance_x = ShadowGui.checkPositiveNumber(self.emittance_x, "Emittance x")
            self.emittance_z = ShadowGui.checkPositiveNumber(self.emittance_z, "Emittance z")
            self.distance_from_waist_x = ShadowGui.checkPositiveNumber(self.distance_from_waist_x, "Distance from waist x")
            self.distance_from_waist_z = ShadowGui.checkPositiveNumber(self.distance_from_waist_z, "Distance from waist z")

            self.number_of_periods = ShadowGui.checkPositiveNumber(self.number_of_periods, "Number of periods")
            self.k_value = ShadowGui.checkPositiveNumber(self.k_value, "K value")
            self.id_period = ShadowGui.checkPositiveNumber(self.id_period, "ID period")

            wigFile = bytes("xshwig.sha", 'utf-8')

            if self.type_combo == 0:
                inData = bytes("", 'utf-8')
            elif self.type_combo == 1:
                inData = bytes(self.file_with_b_vs_y, 'utf-8')
            elif self.type_combo == 2:
                inData = bytes(self.file_with_harmonics, 'utf-8')

            m_to_user_unit = 1.0
            if self.user_unit == 0:
                m_to_userunit = 1000.0
            elif self.user_unit == 1:
                m_to_userunit = 100.0


            self.progressBarSet(10)
            self.information(0, "Calculate electron trajectory")
            self.setStatusMessage("Calculate electron trajectory")
            qApp.processEvents()


            (traj,pars) = srfunc.wiggler_trajectory(b_from=self.type_combo, \
                          inData=inData, nPer=self.number_of_periods, nTrajPoints=501, \
                          ener_gev=self.energy, per=self.id_period, kValue=self.k_value, trajFile="tmp.traj")

            #
            # calculate cdf and write file for Shadow/Source
            #

            self.progressBarSet(20)
            self.information(0, "Calculate cdf and write file for Shadow/Source")
            self.setStatusMessage("Calculate cdf and write file for Shadow/Source")
            qApp.processEvents()

            srfunc.wiggler_cdf(traj, enerMin=self.e_min,  enerMax=self.e_max, enerPoints=1001, outFile=wigFile, elliptical=False)

            self.information(0, "CDF written to file %s \n"%(wigFile))
            self.setStatusMessage("CDF written to file %s \n"%(wigFile))
            qApp.processEvents()

            self.progressBarSet(40)

            self.information(0, "Set the wiggler parameters in the wiggler container")
            self.setStatusMessage("Set the wiggler parameters in the wiggler container")
            qApp.processEvents()

            m_to_user_unit = 1.0

            if self.user_unit == 0:
                m_to_userunit = 1000.0
            if self.user_unit == 1:
                m_to_userunit = 100.0


            shadow_src = ShadowSource.create_wiggler_src()

            shadow_src.src.NPOINT=self.number_of_rays
            shadow_src.src.ISTAR1=self.seed

            shadow_src.src.CONV_FACT = m_to_user_unit # from m to cm (or user unit)

            shadow_src.src.HDIV1 = 1.00000000000000
            shadow_src.src.HDIV2 = 1.00000000000000

            shadow_src.src.PH1=self.e_min
            shadow_src.src.PH2=self.e_max

            shadow_src.src.F_BOUND_SOUR = self.optimize_source_combo

            shadow_src.src.NTOTALPOINT = self.max_number_of_rejected_rays

            if self.optimize_source_combo == 1:
                shadow_src.src.FILE_BOUND = bytes(self.file_with_phase_space_volume, 'utf-8')
            elif self.optimize_source_combo == 2:

                shadow_src.src.FILE_BOUND = bytes("myslit.dat", 'utf-8')

                f = open(str(shadow_src.src.FILE_BOUND),"w")
                f.write("%e %e %e %e %e "%(self.slit_distance, self.min_x, self.max_x, self.min_z, self.max_z))
                f.close()

                self.information(0, "File written to disk: " + str(shadow_src.src.FILE_BOUND))
                qApp.processEvents()

            shadow_src.src.BENER=self.energy

            if self.use_emittances_combo == 0:
                shadow_src.src.SIGMAX=0
                shadow_src.src.SIGMAY=0
                shadow_src.src.SIGMAZ=0
                shadow_src.src.EPSI_X=0
                shadow_src.src.EPSI_Z=0
                shadow_src.src.EPSI_DX=0
                shadow_src.src.EPSI_DZ=0
            else:
                shadow_src.src.SIGMAX=self.sigma_x
                shadow_src.src.SIGMAY=0
                shadow_src.src.SIGMAZ=self.sigma_z
                shadow_src.src.EPSI_X=self.emittance_x
                shadow_src.src.EPSI_Z=self.emittance_z
                shadow_src.src.EPSI_DX=self.distance_from_waist_x
                shadow_src.src.EPSI_DZ=self.distance_from_waist_z

            shadow_src.src.FILE_TRAJ = wigFile

            sys.stdout = EmittingStream(textWritten=self.writeStdOut)
            if self.trace_shadow:
                grabber = TTYGrabber()
                grabber.start()

            self.progressBarSet(50)
            self.setStatusMessage("Running Shadow/Source")

            beam_out = ShadowBeam.traceFromSource(shadow_src)

            if self.trace_shadow:
                grabber.stop()

                for row in grabber.ttyData:
                   self.writeStdOut(row)

            self.information(0, "Plotting Results")
            qApp.processEvents()

            self.progressBarSet(80)
            self.plot_results(beam_out)

            self.information()
            qApp.processEvents()

            self.send("Beam", beam_out)
        except Exception as exception:
            self.error(0, exception.args[0])
            QtGui.QMessageBox.critical(self, "QMessageBox.critical()",
                exception.args[0],
                QtGui.QMessageBox.Ok)

        self.progressBarFinished()

    def sendNewBeam(self, trigger):
        if trigger and trigger.new_beam == True:
            self.runShadowSource()

if __name__ == "__main__":
    a = QApplication(sys.argv)
    ow = Wiggler()
    ow.show()
    a.exec_()
    ow.saveSettings()
