#
# pdis.xpath.atoms
#
# Copyright 2004 Helsinki Institute for Information Technology (HIIT)
# and the authors.  All rights reserved.
#
# Authors: Ken Rimey <rimey@hiit.fi>
#

# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

"""
XPath syntax nodes for atoms
"""

from pdis.xpath.xpath_exceptions import \
     XPathNotImplementedError, XPathEvaluationError

#
# Base classes
#
# These are generally abstract, except that the lexer instantiates
# QNames for its internal use.
#

class Name:
    def __init__(self, s):
        self.name = s

    def __str__(self):
        return self.name

class QName:
    def __init__(self, s1, s2=None):
        if s2 is None:
            self.prefix = None
            self.local_part = s1
        else:
            self.prefix = s1
            self.local_part = s2

    def __str__(self):
        if self.prefix is None:
            return self.local_part
        else:
            return "%s:%s" % (self.prefix, self.local_part)

#
# Atomic nodes
#
# These appear in the token stream generated by the lexer.
#

class AxisName(Name):
    """
    Location step axis name

    Note that these are not used in the final syntax tree, where the
    axis name for a location step is simply stored as a string.
    """
    pass

class NodeType(Name):
    """
    Node type test for location step

    The "name" attribute will be one of "comment", "text",
    "processing-instruction", or "node".
    """
    pass

class NameTest(QName):
    """
    Name test for location step

    Here local_part is permitted to have the value "*".
    """
    def expand(self, context):
        if self.prefix is None:
            uri = None
        else:
            uri = context.namespace_mapping.get(self.prefix)
            if uri is None:
                raise XPathEvaluationError, \
                      'Unbound namespace prefix "%s".' % self.prefix

        if self.local_part == '*':
            name = None
        else:
            name = self.local_part

        return (uri, name)

class FunctionName(QName):
    """
    Function name in function call expression
    """
    pass

class Literal:
    """
    Node representing a string literal
    """
    def __init__(self, s):
        self.value = s

    def __str__(self):
        if '"' in self.value:
            return "'%s'" % self.value
        else:
            return '"%s"' % self.value

    def evaluate(self, context):
        return self.value

class Number:
    """
    Node representing a double-precision floating-point number
    """
    def __init__(self, value):
        self.value = float(value)

    def __str__(self):
        v = self.value
        if v == int(v):
            v = int(v)
        return str(v)

    def evaluate(self, context):
        return self.value

class VariableReference(QName):
    """
    Variable reference node
    """
    def __str__(self):
        return "$%s" % QName.__str__(self)

    def evaluate(self, context):
        raise XPathNotImplementedError, "Variable references not implemented."
