from panopticon.base import PanopticonAttribute, get_metaclass
from panopticon.connection import Connection
from panopticon.base import PanopticonManager

NETWORKING_PATH = 'inmates'

class ComputerGroupManager(PanopticonManager):
    managed_obj_name = "panopticon"
    manager_attribute_name = "computer_groups"

    def _update_after_get(self):
        items = self.panopticon._meta.computer_groups
        return items

class ComputerGroup(PanopticonAttribute):
    def __init__(self, *computers, **kwords):
        super(PanopticonAttribute).__init__(self, **kwords)
        self.computers_names = computers

    def _get_args_kwords(self):
        return self.computer_names, {}

    def __iter__(self):
        for computer in self.panopticon.computers:
            if computer.name in self.computers_names:
                yield computer

    def __getitem__(self, name):
        for computer in self:
            if computer.name == name:
                return computer
        raise KeyError

class ComputerManager(PanopticonManager):
    def _update_after_get(self):
        items = dict(self.panopticon._meta.computers)
        for network in self.panopticon.networks:
            for computer in network:
                items[computer.name] = computer
        return items

class Computer(PanopticonAttribute):
    def __init__(self, ip=None, dns_name=None, key_name=None, network=None,
            mac=None, panopticon=None, name=None):
        super(Computer, self).__init__(name=name, panopticon=panopticon)
        self.ip = ip
        self.dns_name = dns_name
        self.key_name = key_name
        self.network = network
        self.mac = mac
        self.key = None

    def get_conn(self):
        return Connection(self)

    def _get_args_kwords(self):
        kwords = dict(ip=self.ip, dns_name=self.dns_name, key_name=self.key_name,
                network=self.network, mac=self.mac, panopticon=self.panopticon,
                name=self.name)
        return (), kwords

    @property
    def stand_alone(self):
        return self.network is None

    def get_ip(self):
        return self.ip if self.stand_alone else self.network.get_ip(self.ip)

    def get_dns_name(self):
        return self.dns_name if self.stand_alone else self.network.get_dns_name(self.dns_name)

class NetworkManager(PanopticonManager):
    def _update_after_get(self):
        return dict(self.panopticon._meta.networks)

class NetworkMetaclass(type):
    def __new__(cls, classname, bases, attrs):
        computers = {}
        todelete = []
        for name, attr in attrs.iteritems():
            if isinstance(attr, Computer):
                attr.name = name
                computers[name] = attr
                todelete.append(name)
        for name in todelete:
            del attrs[name]
        meta_attrs = {'computers' : computers}
        attrs['_meta'] = get_metaclass(meta_attrs)
        return type.__new__(cls, classname, bases, attrs)

class Network(PanopticonAttribute):
    __metaclass__ = NetworkMetaclass

    def __init__(self, panopticon=None, name=None):
        super(Network).__init__(self, name=name, panopticon=panopticon)
        self._computers = self._update_computers()

    def get_ip(self, ip):
        return "%s.%s" % (self.ip, ip)

    def get_dns_name(self, dns):
        return "%s.%s" % (self.dns, dns)

    def __iter__(self):
        for name, computer in self._meta.computers.iteritems():
            computer = computer.reinstantiate(self.panopticon)
            computer.network = self
            yield computer.name, computer

    def __getitem__(self, name):
        for cname, computer in self:
            if cname == name:
                return computer
        raise KeyError


