import stat
import datetime
import paramiko
import gevent
# from panopticon.informer.filetracker import filetracker, file_id

local_data = gevent.local.local()

class RunCommandError(Exception):
    pass

class SetConnectionToComputer(object):
    def __init__(self, computer):
        self.computer = computer

    def __enter__(self):
        local_data.conn = self.computer.get_conn()

    def __exit__(self, exc_type, exc_val, exc_tb):
        local_data.conn = None

class Connection(object):
    def __init__(self, computer):
        self.computer = computer
        self.hostname = computer.get_dns_name()
        self.username = "root" 
        self.conn = paramiko.SSHClient()
        self.conn.set_log_channel('panopticon.ssh')
        self.conn.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self.conn.connect(self.hostname, username=self.username)

    def run(self, command, grab_error=False, bufsize=-1):
        chan = self.conn._transport.open_session()
        chan.exec_command(command)
        stdout_file = chan.makefile('rb', bufsize)
        stderr_file = chan.makefile('rb', bufsize)
        stdout = stdout_file.read()
        stderr = stderr_file.read()
        exit_status = chan.recv_exit_status()
        if grab_error:
            raise RunCommandError('Command "%s" ran in host %s returned exit code %d' % (command, self.hostname, exit_status))
        return exit_status, stdout, stderr 

    def get_remote_file(self, filename):
        return RemoteFile(self.conn, filename)

    def get_file_id(self, filename):
        return file_id(self.hostname, filename)

    def get(self, local_filename, remote_filename, callback=None):
        sftp = self.conn.open_sftp()
        sftp.get(local_filename, remote_filename, callback)
        sftp.close()

    def put(self, local_filename, remote_filename, callback=None, confirm=True):
        sftp = self.conn.open_sftp()
        sftp.put(local_filename, remote_filename, callback, confirm=confirm)
        self.get_remote_file(remote_filename).update_filetracker()
        sftp.close()

    def close(self):
        self.conn.close()

    def __enter__(self):
        return self

    def __exit__(self):
        self.close()

class RemoteFile(object):
    def __init__(self, conn, filename):
        self.conn = conn
        self.sftp = conn.conn.open_sftp()
        self.filename = filename

    def file_id(self):
        return self.conn.get_file_id(self.filename)

    def open(self, mode='r', bufsize=-1):
        if mode == 'w':
            filetracker.update_file(self.file_id, datetime.utcnow())
        return self.sftp.file(self.filename, mode=mode, bufsize=bufsize)

    def update(self, update_function):
        data = update_function(self.open().read())
        with self.open(mode='w') as f:
            f.write(data)
        self.update_filetracker()

    def update_filetracker(self):
        filetracker.update(self.file_id, self.m_time())

    def check_filetracker(self):
        return filetracker.check(self.file_id, self.m_time())

    def stat(self):
        return self.sftp.stat(self.filename)

    def is_dir(self):
        return stat.S_ISDIR(self.stat())

    def is_chr(self):
        return stat.S_ISCHR(self.stat())

    def is_blk(self):
        return stat.S_ISBLK(self.stat())

    def is_reg(self):
        return stat.S_ISREG(self.stat())

    def is_fifo(self):
        return stat.S_ISFIFO(self.stat())

    def is_lnk(self):
        return stat.S_ISLNK(self.stat())

    def is_sock(self):
        return stat.S_ISSOCK(self.stat())

    def a_time(self):
        return stat.ST_ATIME(self.stat())

    def m_time(self):
        return stat.ST_MTIME(self.stat())

    def c_time(self):
        return stat.ST_CTIME(self.stat())

    def md5(self):
        exit, stdout, stderr = self.conn.run('md5sum %s' % self.filename)
        return stdout.split(' ')[0]

    def close(self):
        self.sftp.close()

    def __enter__(self):
        return self

    def __exit__(self):
        self.close()

