import gevent
from panopticon.base import RoleAttribute
from panopticon.util.managers import AttributableManager
from panopticon.connection import SetConnectionToComputer
from panopticon.util.listlogic import check_all_true, check_at_least_one_true

A_PENDING = 0
A_STARTED = 1
A_COMPLETED = 2
A_FAILED = 3

def check_greenlets_and_raises(greenlets):
    for greenlet in greenlets:
        if not greenlet.successful():
            raise greenlet.exception

class ActionRunner(object):
    def __init__(self):
        self.status = A_PENDING
        self.value = None

    @property
    def finished(self):
        return self.status in A_COMPLETED, A_FAILED

    @property
    def pending(self):
        return self.status == A_PENDING

    @property
    def started(self):
        return self.status == A_STARTED

    @property
    def completed(self):
        return self.status == A_COMPLETED

    @property
    def failed(self):
        return self.status == A_FAILED

    def check_required(self, action):
        return False

    def _get_status(self):
        return A_FAILED

    def _get_result(self):
        return {}

    def launch(self, *args, **kwargs):
        pass

    def __call__(self, *args, **kwargs):
        self.status = A_STARTED
        self.launch(*args, **kwargs)
        self.status = self._get_status()
        self.value = self._get_result()

class role_action(ActionRunner, RoleAttribute):
    def __init__(self, function, required_roles_names=[], role=None, name=None):
        ActionRunner.__init__(self)
        RoleAttribute.__init__(self, role=role, name=name)
        self.required_roles_names = required_roles_names
        self.function = function
        self.service = role.service if role is not None else None
        self.sensitive = role.sensitive if role is not None else True 
        self.greenlets = None

    def _get_args_kwords(self):
        return (self.function,), {'required_roles_names':self.required_roles_names} 

    def check_required(self, action):
        return action.role in self.required_roles

    def _get_status(self):
        successful = map(lambda x:x.successful(), self.greenlets)
        if self.sensitive:
            worked = check_all_true(successful)
        else:
            worked = check_at_least_one_true(successful)
        return A_COMPLETED if worked else A_FAILED

    def _get_result(self):
        result = {}
        computers = self.role.computers
        for computer, greenlet in zip(computers, self.greenlets):
            result[computer] = greenlet.value if greenlet.successful else greenlet.exception
        return result

    def launch(self):
        self._load_greenlets()
        gevent.joinall(self.greenlets)

    @property
    def required_roles(self):
        required_roles = list(self.role.required_roles)
        for r_name in self.required_roles_names:
            role = self.service.roles[r_name]
            required_roles.append(role)
        return list(set(required_roles))

    def _get_runner(self, computer):
        def function(*args, **kwargs):
            with SetConnectionToComputer(computer):
                return self.function(computer, self.service, *args, **kwargs)
        return function

    def _load_greenlets(self, *args, **kwargs):
        self.greenlets = []
        for computer in self.role.computers:
            greenlet = gevent.spawn(self._get_runner(computer), *args, **kwargs)
            self.greenlets.append(greenlet)

    def __repr__(self):
        class_name = self.__class__.__name__
        role_name = self.role.name
        service_name = self.service.name
        action_name = self.name
        service = "service: '%s'" % service_name if service_name is not None else ''
        return "<%s: action: '%s' role: '%s' %s>" % (class_name,
                action_name, role_name, service)

class RoleActionManager(AttributableManager):
    managed_obj_name = "role"
    manager_attribute_name = "actions"

    def _update_after_get(self):
        return self.role._meta.actions 

class LauncherActionRunner(ActionRunner):
    def __init__(self, name, manager):
        super(LauncherActionRunner, self).__init__()
        self.name = name
        self.manager = manager
        self.actions = manager[name]
        self.required = {}
        self.depends = {}
        self._update_requirements()

    def check_required(self, action):
        pass

    def _get_status(self):
         completed = reduce(lambda x,y:x and y.completed, self.actions, True)
         return A_COMPLETED if completed else A_FAILED

    def _get_result(self):
        return [ x.value for x in self.actions ]

    def launch(self, *args, **kwargs):
        self.launch_pending_actions(*args, **kwargs)

    def _update_requirements(self):
        for action in self.actions:
            self.depends[action] = []
            self.required[action] = []
        for required_action in self.actions:
            for depend_action in self.actions:
                if required_action.check_required(depend_action):
                    self.depends[depend_action].append(required_action)
                    self.required[required_action].append(depend_action)

    def _check_actions_completed(self):
        return reduce(lambda x,y:x and y.is_finished, self.actions, True)

    def _get_ready_actions(self):
        return filter(lambda x:x.completed, self.actions)

    def _update_depends_status(self, action, status):
        for dependant in self.required[action]:
            dependant.status = status
            self._update_dependants_status(dependant, status)

    def _get_action_launcher(self, action):
        def action_launcher(*args, **kwargs):
            action(*args, **kwargs)
            if action.completed:
                self.launch_pending_actions(*args, **kwargs)
            elif action.failed:
                self._update_depends_status(action, A_FAILED)
        return action_launcher

    def _check_all_required_ready(self, action):
        required_actions = self.required[action] 
        result = set(self._get_ready_actions()).issuperset(required_actions)
        return result

    def launch_pending_actions(self, *args, **kwargs):
        runable_actions = []
        for action in self.actions:
            if action.pending:
                if self._check_all_required_ready(action):
                    runable_actions.append(self._get_action_launcher(action))
        geventlets = map(lambda x:gevent.spawn(x,*args,**kwargs), runable_actions)
        gevent.joinall(geventlets)

    def __repr__(self):
        class_name = self.__class__.__name__
        action_name = self.name
        manager_instance = str(self.manager._get_obj())
        return "<%s: launching actions: '%s' on '%s'>" % (class_name, action_name,
                manager_instance)

class ServiceActionLauncher(LauncherActionRunner):
    def check_required(self, action):
        return action.manager.service in self.manager.service.required_services

class PanopticonActionLauncher(LauncherActionRunner):
    pass

class ActionManager(AttributableManager):
    action_launcher_class = LauncherActionRunner
    manager_attribute_name = "actions"

    def _to_attribute(self, name, actions):
        return self.action_launcher_class(manager=self, name=name)

    def _update_after_get(self):
        pass


class ServiceActionManager(ActionManager):
    action_launcher_class = ServiceActionLauncher
    managed_obj_name = "service"

    def _update_after_get(self):
        actions = {}
        for rname, role in self.service.roles:
            for name, action in role.actions:
                if actions.has_key(name):
                    actions[name].append(action)
                else:
                    actions[name] = [action]
        return actions

class PanopticonActionManager(ActionManager):
    action_launcher_class = PanopticonActionLauncher
    managed_obj_name = "panopticon"

    def _update_after_get(self):
        actions = {} 
        for sname, service in self.panopticon.services:
            for name, panopticon_actions in service.actions:
                action = getattr(service.actions, name)
                if actions.has_key(name):
                    actions[name].append(action)
                else:
                    actions[name] = [action]
        return actions



