from paramiko import RSAKey as pRSAKey, DSSKey
from sqlalchemy import create_engine, Column, DateTime, String, Integer, Text
from sqlalchemy.orm import sessionmaker, relationship, backref
from sqlalchemy.schema import ForeignKey
from sqlalchemy.ext.declarative import declarative_base
from panopticon.util.database import key_value_property

Base = declarative_base()

class Service(Base):
    __tablename__ = "services"
    name = Column(String(50), primary_key=True)


class Computer(Base):
    __tablename__ = "computers"
    name = Column(String(255), primary_key=True)
    status = Column(String(50))
    key_name = Column(String(100), ForeignKey('keys.name', onupdate="CASCADE"))
    key = relationship("Key", backref=backref('computers'))

    def __init__(self, name, key_name="", status="Unknown"):
        self.name = name
        self.status = status
        self.key_name = key_name

class Log(Base):
    __tablename__ = "logs"
    time = Column(DateTime(), primary_key=True)
    level = Column(String(10))
    message = Column(Text())
    computer_name = Column(String(255), ForeignKey('computers.name',ondelete="CASCADE", onupdate="CASCADE"))
    computer = relationship("Computer",backref=backref('logs', order_by=time))
    service_name = Column(String(50), ForeignKey('services.name',ondelete="CASCADE", onupdate="CASCADE"))
    computer = relationship("Service",backref=backref('services', order_by=time))

    def __init__(self,computer_name, time, level, message):
        self.time = time
        self.level = level
        self.message = message
        self.computer_name = computer_name

class FileTrack(Base):
    __tablename__ = "filetracks"
    uid = Column("uid", String(32), primary_key=True)
    _computer_name = Column("computer_name", String(255),ForeignKey('computers.name'))
    _path = Column("path", Text())
    last_mod_time = Column("last_mod_time", DateTime())
    md5 = Column("md5", String(32))

    def __init__(self, computer_name, path, last_mod_time, md5=""):
        self.computer_name = computer_name
        self.path = path
        self.last_mod_time = last_mod_time
        self.md5 = md5
        self.update_uid()

    @property
    def computer_name(self):
        return self._computer_name

    @computer_name.setter
    def computer_name(self, value):
        self._computer_name = value
        self.update_uid()

    @property
    def path(self):
        return self._path

    @path.setter
    def path(self, value):
        self._path = value
        self.update_uid()

    def update_uid(self):
        if self.computer_name and self.path:
            self.uid = "%s:%s" % (self.computer_name, self.path)
        else:
            self.uid = ""

class Key(Base):
    __tablename__ = "keys"
    name = Column(String(100), primary_key=True)
    algorithm = Column(String(20))
    v1 = Column(String(2048))
    v2 = Column(String(2048))
    v3 = Column(String(2048))
    v4 = Column(String(2048))
    key_class = None
    key_vals = []
    __mapper_args__ = {'polymorphic_on' : algorithm}

    @classmethod
    def build_from_paramiko_key(cls, name, p_key):
        if isinstance(p_key, pRSAKey):
            return RSAKey(name, p_key.e, p_key.n)
        elif isinstance(p_key, DSSKey):
            return DSAKey(name, p_key.p, p_key.q, p_key.g, p_key.y)
        else:
            raise Exception("Not valid key")

    def __init__(self, name, algorithm, v1, v2, v3, v4):
        self.name = name
        self.algorithm = algorithm
        self.v1 = v1
        self.v2 = v2
        self.v3 = v3
        self.v4 = v4

    def get_paramiko_key(self):
        vals = [ getattr(self, x) for x in self.key_vals ]
        return self.key_class(vals=vals)

class RSAKey(Key):
    __mapper_args__ = {'polymorphic_identity':'rsa'}
    key_class = pRSAKey
    key_vals = [ 'e', 'n' ]

    def __init__(self, name, e, n):
        self.name = name
        self.algorithm = "rsa"
        self.e = e
        self.n = n

    e = key_value_property("v1")
    n = key_value_property("v2")


class DSAKey(Key):
    __mapper_args__ = {'polymorphic_identity':'dsa'}
    key_class = DSSKey
    key_vals = [ 'p', 'q', 'g', 'y' ]

    def __init__(self, namevp, q, g, y):
        self.name = name
        self.algorithm = "dsa"
        self.p = p
        self.q = q
        self.g = g
        self.y = y

    p = key_value_property("v1")
    q = key_value_property("v2")
    g = key_value_property("v3")
    y = key_value_property("v4")

class PanopticonDB(object):
    def __init__(self, panopticon, engine=None):
        self.panopticon = panopticon
        self.engine = engine if engine is not None else create_engine(panopticon.db_url)
        Base.metadata.create_all(self.engine)
        self.Session = sessionmaker(bind=self.engine)
        self.session = self.Session()

