import re
from datetime import datetime
import logging
from panopticon.database import Log
from panopticon.util.coloredformatter import ColoredFormatter

service_message_splitter_regex = re.compile(r"^(\w*):(\w*) : (.*)$")
computer_message_splitter_regex = re.compile(r"^(\w*) : (.*)$")

def get_screen_handler():
    screen_handler = logging.StreamHandler()
    screen_handler.setLevel(logging.ERROR)
    screen_handler.setFormatter(ColoredFormatter("%(asctime)s - %(levelname)s - %(message)s", "%H:%M:%S"))
    return screen_handler

class PanopticonHandler(logging.Handler):
    def __init__(self, panopticon, level=logging.NOTSET):
        logging.Handler.__init__(self, level=level)
        self.panopticon = panopticon
        self.session = panopticon.db.session

    def emit(self, record):
        time = datetime.fromtimestamp(record.created)
        match = service_message_splitter_regex.match(record.msg)
        if match:
            computer_name,service_name, msg = match.groups()
        else:
            service_name = ""
            match = computer_message_splitter_regex.match(record.msg)
            if match:
                computer_name, msg = match.groups()
            else:
                msg = record.msg
                computer_name = ""
        log = Log(computer_name, time, record.levelname.lower(), msg)
        self.session.add(log)
        self.session.commit()

    def handleError(self,record):
        screen_handler = get_screen_handler()
        screen_handler.handle(record)
        
def get_panopticon_logger(panopticon, level="notset", screen_dump=False):
    level = getattr(logging, level.upper())
    panopticon_name = panopticon.__class__.__name__
    logger = logging.getLogger(panopticon_name)
    panopticon_handler = PanopticonHandler(panopticon, level=level)
    logger.addHandler(panopticon_handler)
    if screen_dump:
        logger.addHandler(get_screen_handler())
    return logger


        




