from panopticon.base import get_metaclass
from panopticon.database import PanopticonDB
from panopticon.keys import KeyManager
from panopticon.util.modules import get_class_path
from panopticon.logger import get_panopticon_logger
from panopticon.services import Service, ServiceManager
from panopticon.actions import PanopticonActionManager
from panopticon.computers import Computer, ComputerManager
from panopticon.computers import ComputerGroup, ComputerGroupManager
from panopticon.computers import Network, NetworkManager
from panopticon.util.inspect import get_attribute_instances, update_dict_of_dicts
from panopticon.util.managers import ManagerMetaclass


class PanopticonMetaclass(ManagerMetaclass):
    nameable_attrs = Service, Computer, Network, ComputerGroup
    mateable_categories = (("services", Service),
                           ("computers", Computer),
                           ("networks", Network),
                           ("computer_groups", ComputerGroup))

    def __new__(cls, clsname, bases, attrs):
        super(PanopticonMetaclass, cls).__new__(cls, clsname, bases, attrs)
        attrs['_meta'] = cls.get_meta(bases, attrs)
        return type.__new__(cls, clsname, bases, attrs)

    @classmethod
    def get_meta(cls, bases, attrs):
        meta_attrs = cls.create_meta_from_attrs(attrs)
        meta_attrs = cls.update_meta_from_bases(meta_attrs, bases)
        return get_metaclass(meta_attrs)

    @classmethod
    def create_meta_from_attrs(cls, attrs):
        todelete = []
        meta_attrs = {}
        for cat_name, cat_class in cls.mateable_categories:
            meta_attrs[cat_name] = {} 
        for name, attr in attrs.iteritems():
            for cat_name, cat_class in cls.mateable_categories:
                if isinstance(attr, cat_class):
                    attr.name = name
                    meta_attrs[cat_name][name] = attr
                    todelete.append(name)
                    continue
        for name in todelete:
            del attrs[name]
        return meta_attrs

    @classmethod
    def update_meta_from_bases(cls, meta, bases):
        bases = list(bases)
        bases.reverse()
        bases_dicts = [ x._meta.getdict() for x in bases if hasattr(x,"_meta") ]
        bases_dicts.append(meta)
        return update_dict_of_dicts(*bases_dicts)

class Panopticon(object):
    db_url = ""
    key_path = ""
    __metaclass__ = PanopticonMetaclass
    actions = PanopticonActionManager()
    services = ServiceManager()
    computers = ComputerManager()
    networks = NetworkManager()
    computer_groups = ComputerGroupManager()
    keys = KeyManager()

    def __init__(self, logging_level="notset", active_computers=None, active_services=None):
        self._update_meta()
        self.active_computers = active_computers
        self.active_services = active_services
        self.path = get_class_path(self)
        self.db = PanopticonDB(self)
        self.logger = get_panopticon_logger(self, level=logging_level)

    def _update_meta(self):
        result = {}
        for mname, mvalue in self._meta.itermeta():
            workon = {}
            result[mname] = workon
            for name, value in mvalue.iteritems():
                workon[name] = value.reinstantiate(self)
        meta = get_metaclass(result)
        self._meta = meta

    def get_network(self, network_name):
        for network in self.networks:
            if network.name == network_name:
                return network

    def get_computers(self, names=None):
        if names is None:
            computers = get_attribute_instances(self, Computer)
            for network in self.networks:
                computers.extend(network.get_computers())
        else:
            computers = []
            for attr_name in names:
                attr = getattr(self, attr_name)
                if isinstance(attr, Computer):
                    computers.append(attr)
                elif isinstance(attr, Network):
                    computers.extend(network.get_computers)
                elif isinstance(attr, ComputerGroup):
                    computers.extend(attr.computers)
        return list(set(computers))

    def __str__(self):
        return self.__class__.__name__

