# -*- coding: utf-8 -*-
import os

import csv
import xml.etree.cElementTree as et
from collections import defaultdict

import numpy as np

import pathomx.ui as ui
import pathomx.db as db
import pathomx.threads as threads
import pathomx.utils as utils

from pathomx.plugins import ImportPlugin
from pathomx.data import DataSet


class ChenomxApp(ui.ImportDataApp):

    import_filename_filter = "All compatible files (*.csv *.txt *.tsv);;All files (*.*)"
    import_description = "Open a mapped metabolite file from Chenomx"

    def load_datafile(self, file):

        reader = csv.reader(open(file, 'rU'), delimiter='\t', dialect='excel')
        hrow = next(reader)  # Get top row

        slabels = []
        data = []

        if hrow[0] == 'Profiled Data Type':  # Is a Chenomx output file; use the other columns to map data scale/etc. once implemented
            next(reader)  # Skip date row
            hrow = next(reader)
            labels = hrow[2:]  # We strip off the pH here; might be nice to keep it
            entities = [self.m.db.synrev[l] if l in self.m.db.synrev else None for l in labels]  # Map to entities if they exist

            next(reader)  # Skip compound ID
            next(reader)  # Skip InChI
            next(reader)  # Skip SMILES

            for hrow in reader:  # Now read the data rows
                slabels.append(hrow[0])
                td = []
                for x in hrow[2:]:
                    try:
                        td.append(float(x))
                    except:
                        td.append(0)
                data.append(td)

        data = np.array(data)
        dso = DataSet(size=data.shape)
        print(data.shape)
        dso.labels[1] = labels
        dso.entities[1] = entities
        dso.labels[0] = slabels
        dso.data = data

        return {'output': dso}

    #Profiled Data Type    Concentrations ( mM )
    #Export Date    Tue Nov 12 14:13:57 GMT 2013
    #    pH    1,3-Dihydroxyacetone    1,3-Dimethylurate    1,6-Anhydro-β-D-glucose    1,7-Dimethylxanthine    1-Methylnicotinamide    2'-Deoxyadenosine    2'-Deoxyguanosine    2'-Deoxyinosine    2-Aminoadipate    2-Aminobutyrate    2-Ethylacrylate    2-Furoate    2-Hydroxy-3-methylvalerate    2-Hydroxybutyrate    2-Hydroxyglutarate    2-Hydroxyisobutyrate    2-Hydroxyisocaproate    2-Hydroxyisovalerate    2-Hydroxyphenylacetate    2-Hydroxyvalerate    2-Methylglutarate    2-Octenoate    2-Oxobutyrate    2-Oxocaproate    2-Oxoglutarate    2-Oxoisocaproate    2-Oxovalerate    2-Phenylpropionate    2-Phosphoglycerate    3,4-Dihydroxymandelate    3,5-Dibromotyrosine    3-Aminoisobutyrate    3-Chlorotyrosine    3-Hydromuconate    3-Hydroxy-3-methylglutarate    3-Hydroxybutyrate    3-Hydroxyisovalerate    3-Hydroxykynurenine    3-Hydroxymandelate    3-Hydroxyphenylacetate    3-Indoxylsulfate    3-Methyl-2-oxovalerate    3-Methyladipate    3-Methylglutarate    3-Methylxanthine    3-Phenyllactate    3-Phenylpropionate    4-Aminobutyrate    4-Aminohippurate    4-Carboxyglutamate    4-Hydroxy-3-methoxymandelate    4-Hydroxybenzoate    4-Hydroxybutyrate    4-Hydroxyphenylacetate    4-Hydroxyphenyllactate    4-Pyridoxate    5,6-Dihydrothymine    5,6-Dihydrouracil    5-Aminolevulinate    5-Hydroxyindole-3-acetate    5-Hydroxylysine    5-Hydroxytryptophan    5-Methoxysalicylate    6-Hydroxynicotinate    ADP    AMP    ATP    Acetamide    Acetaminophen    Acetate    Acetoacetate    Acetone    Acetylsalicylate    Adenine    Adenosine    Adipate    Alanine    Allantoin    Alloisoleucine    Anserine    Anthranilate    Arginine    Ascorbate    Asparagine    Aspartate    Azelate    Benzoate    Betaine    Biotin    Butanone    Butyrate    Cadaverine    Caffeine    Caprate    Caprylate    Carnitine    Carnosine    Catechol    Cholate    Choline    Cinnamate    Citrate    Citrulline    Creatine    Creatine phosphate    Creatinine    Cystathionine    Cysteine    Cystine    Cytidine    Cytosine    DSS (Chemical Shape Indicator)    DSS-d6 (Chemical Shape Indicator)    Dimethylamine    Epicatechin    Ethanol    Ethanolamine    Ethylene glycol    Ethylmalonate    Ferulate    Formate    Fructose    Fucose    Fumarate    GTP    Galactarate    Galactitol    Galactonate    Galactose    Gentisate    Glucarate    Glucitol    Gluconate    Glucose    Glutamate    Glutamine    Glutarate    Glutaric acid monomethyl ester    Glutathione    Glycerate    Glycerol    Glycine    Glycocholate    Glycolate    Glycylproline    Guanidoacetate    Guanosine    Hippurate    Histamine    Histidine    Homocitrulline    Homocysteine    Homocystine    Homogentisate    Homoserine    Homovanillate    Hypoxanthine    Ibuprofen    Imidazole    Indole-3-acetate    Indole-3-lactate    Inosine    Isobutyrate    Isocaproate    Isocitrate    Isoeugenol    Isoleucine    Isopropanol    Isovalerate    Kynurenate    Kynurenine    Lactate    Lactose    Leucine    Levulinate    Lysine    Malate    Maleate    Malonate    Maltose    Mandelate    Mannitol    Mannose    Methanol    Methionine    Methylamine    Methylguanidine    Methylmalonate    Methylsuccinate    N,N-Dimethylformamide    N,N-Dimethylglycine    N-Acetylaspartate    N-Acetylcysteine    N-Acetylglutamate    N-Acetylglutamine    N-Acetylglycine    N-Acetylornithine    N-Acetylserotonin    N-Acetyltyrosine    N-Carbamoyl-β-alanine    N-Carbamoylaspartate    N-Isovaleroylglycine    N-Methylhydantoin    N-Nitrosodimethylamine    N-Phenylacetylglycine    N-Phenylacetylphenylalanine    N6-Acetyllysine    NAD+    NADH    NADP+    NADPH    Niacinamide    Nicotinamide N-oxide    Nicotinate    Nicotinic acid adenine dinucleotide    Nicotinurate    O-Acetylcarnitine    O-Acetylcholine    O-Phosphocholine    O-Phosphoethanolamine    O-Phosphoserine    Ornithine    Oxypurinol    Pantothenate    Phenol    Phenylacetate    Phenylalanine    Phthalate    Pimelate    Pipecolate    Proline    Propionate    Propylene glycol    Protocatechuate    Putrescine    Pyridoxine    Pyrimidine    Pyroglutamate    Pyruvate    Quinolinate    Riboflavin    Ribose    S-Adenosylhomocysteine    S-Sulfocysteine    Saccharopine    Salicylate    Salicylurate    Sarcosine    Sebacate    Serine    Serotonin    Suberate    Succinate    Succinylacetone    Sucrose    Tartrate    Taurine    Theophylline    Threonate    Threonine    Thymidine    Thymine    Thymol    Tiglylglycine    Trigonelline    Trimethylamine    Trimethylamine N-oxide    Tropate    Tryptophan    Tyramine    Tyrosine    UDP-galactose    UDP-glucose    Uracil    Urea    Uridine    Urocanate    Valerate    Valine    Valproate    Vanillate    Xanthine    Xanthosine    Xylose    cis-Aconitate    dCTP    dTTP    myo-Inositol    o-Cresol    p-Cresol    sn-Glycero-3-phosphocholine    trans-4-Hydroxy-L-proline    trans-Aconitate    β-Alanine    γ-Glutamylphenylalanine    π-Methylhistidine    τ-Methylhistidine
    #Compound ID        140    346    261    347    338    277    158    168    112    111    208    68    185    179    174    69    134    221    175    254    152    304    305    307    181    153    312    471    322    353    478    91    481    359    5    4    284    1158    363    364    6    182    184    237    348    147    71    101    349    4304    149    372    102    7    96    378    171    160    282    78    90    1491    381    1566    1713    1714    1716    8    315    9    79    11    306    272    276    12    232    13    110    142    1829    240    113    231    234    81    82    15    250    350    128    1978    84    384    85    21    114    1999    176    302    386    22    29    23    2101    24    141    228    218    161    150    25    10    27    2    205    97    99    394    371    32    86    107    100    2410    396    286    263    258    34    398    259    399    183    229    226    35    198    493    87    36    242    402    103    200    332    270    37    2418    217    204    93    186    3    314    148    164    324    40    106    2468    269    409    297    313    411    225    72    41    311    188    42    83    235    115    291    94    416    121    122    108    201    260    178    213    44    354    288    66    8587    28    194    1679    424    738    425    2729    2733    127    143    429    287    2795    1288    342    929    125    499    500    8688    2734    267    837    253    2147    2781    195    2830    321    129    224    117    271    256    191    50    227    197    132    118    230    53    262    443    3032    252    3044    119    133    444    251    170    498    154    189    55    56    57    135    216    3148    136    60    77    193    450    61    159    120    219    255    137    460    220    334    62    63    464    241    138    243    1255    1287    139    64    162    151    299    215    73    370    156    167    89    351    867    883    223    432    435    333    214    275    67    274    76    1
    #InChI        InChI=1S/C3H6O3/c4-1-3(6)2-5/h4-5H,1-2H2    InChI=1S/C7H8N4O3/c1-10-4-3(8-6(13)9-4)5(12)11(2)7(10)14/h1-2H3,(H2,8,9,13)    InChI=1S/C6H10O5/c7-3-2-1-10-6(11-2)5(9)4(3)8/h2-9H,1H2/t2-,3-,4+,5-,6-/m1/s1    InChI=1S/C7H8N4O2/c1-10-3-8-5-4(10)6(12)11(2)7(13)9-5/h3H,1-2H3,(H,9,13)    InChI=1S/C7H8N2O/c1-9-4-2-3-6(5-9)7(8)10/h2-5H,1H3,(H-,8,10)/p+1    InChI=1S/C10H13N5O3/c11-9-8-10(13-3-12-9)15(4-14-8)7-1-5(17)6(2-16)18-7/h3-7,16-17H,1-2H2,(H2,11,12,13)/t5-,6+,7+/m0/s1    InChI=1S/C10H13N5O4/c11-10-13-8-7(9(18)14-10)12-3-15(8)6-1-4(17)5(2-16)19-6/h3-6,16-17H,1-2H2,(H3,11,13,14,18)/t4-,5+,6+/m0/s1    InChI=1S/C10H12N4O4/c15-2-6-5(16)1-7(18-6)14-4-13-8-9(14)11-3-12-10(8)17/h3-7,15-16H,1-2H2,(H,11,12,17)/t5-,6+,7+/m0/s1    InChI=1S/C6H11NO4/c7-4(6(10)11)2-1-3-5(8)9/h4H,1-3,7H2,(H,8,9)(H,10,11)/t4-/m0/s1    InChI=1S/C4H9NO2/c1-2-3(5)4(6)7/h3H,2,5H2,1H3,(H,6,7)/t3-/m0/s1    InChI=1S/C5H8O2/c1-3-4(2)5(6)7/h2-3H2,1H3,(H,6,7)    InChI=1S/C5H4O3/c6-5(7)4-2-1-3-8-4/h1-3H,(H,6,7)    InChI=1S/C6H12O3/c1-3-4(2)5(7)6(8)9/h4-5,7H,3H2,1-2H3,(H,8,9)/t4-,5-/m1/s1    InChI=1S/C4H8O3/c1-2-3(5)4(6)7/h3,5H,2H2,1H3,(H,6,7)/t3-/m0/s1    InChI=1S/C5H8O5/c6-3(5(9)10)1-2-4(7)8/h3,6H,1-2H2,(H,7,8)(H,9,10)/t3-/m0/s1    InChI=1S/C4H8O3/c1-4(2,7)3(5)6/h7H,1-2H3,(H,5,6)    InChI=1S/C6H12O3/c1-4(2)3-5(7)6(8)9/h4-5,7H,3H2,1-2H3,(H,8,9)/t5-/m0/s1    InChI=1S/C5H10O3/c1-3(2)4(6)5(7)8/h3-4,6H,1-2H3,(H,7,8)/t4-/m0/s1    InChI=1S/C8H8O3/c9-7-4-2-1-3-6(7)5-8(10)11/h1-4,9H,5H2,(H,10,11)    InChI=1S/C5H10O3/c1-2-3-4(6)5(7)8/h4,6H,2-3H2,1H3,(H,7,8)/t4-/m0/s1    InChI=1S/C6H10O4/c1-4(6(9)10)2-3-5(7)8/h4H,2-3H2,1H3,(H,7,8)(H,9,10)    InChI=1S/C8H14O2/c1-2-3-4-5-6-7-8(9)10/h6-7H,2-5H2,1H3,(H,9,10)/b7-6+    InChI=1S/C4H6O3/c1-2-3(5)4(6)7/h2H2,1H3,(H,6,7)    InChI=1S/C6H10O3/c1-2-3-4-5(7)6(8)9/h2-4H2,1H3,(H,8,9)    InChI=1S/C5H6O5/c6-3(5(9)10)1-2-4(7)8/h1-2H2,(H,7,8)(H,9,10)    InChI=1S/C6H10O3/c1-4(2)3-5(7)6(8)9/h4H,3H2,1-2H3,(H,8,9)    InChI=1S/C5H8O3/c1-2-3-4(6)5(7)8/h2-3H2,1H3,(H,7,8)    InChI=1S/C9H10O2/c1-7(9(10)11)8-5-3-2-4-6-8/h2-7H,1H3,(H,10,11)    InChI=1S/C3H7O7P/c4-1-2(3(5)6)10-11(7,8)9/h2,4H,1H2,(H,5,6)(H2,7,8,9)/t2-/m1/s1    InChI=1S/C8H8O5/c9-5-2-1-4(3-6(5)10)7(11)8(12)13/h1-3,7,9-11H,(H,12,13)/t7-/m0/s1    InChI=1S/C9H9Br2NO3/c10-5-1-4(2-6(11)8(5)13)3-7(12)9(14)15/h1-2,7,13H,3,12H2,(H,14,15)/t7-/m0/s1    InChI=1S/C4H9NO2/c1-3(2-5)4(6)7/h3H,2,5H2,1H3,(H,6,7)/t3-/m0/s1    InChI=1S/C9H10ClNO3/c10-6-3-5(1-2-8(6)12)4-7(11)9(13)14/h1-3,7,12H,4,11H2,(H,13,14)/t7-/m0/s1    InChI=1S/C6H8O4/c7-5(8)3-1-2-4-6(9)10/h1-2H,3-4H2,(H,7,8)(H,9,10)/b2-1+    InChI=1S/C6H10O5/c1-6(11,2-4(7)8)3-5(9)10/h11H,2-3H2,1H3,(H,7,8)(H,9,10)    InChI=1S/C4H8O3/c1-3(5)2-4(6)7/h3,5H,2H2,1H3,(H,6,7)/t3-/m1/s1    InChI=1S/C5H10O3/c1-5(2,8)3-4(6)7/h8H,3H2,1-2H3,(H,6,7)    InChI=1S/C10H12N2O4/c11-6(10(15)16)4-8(14)5-2-1-3-7(13)9(5)12/h1-3,6,13H,4,11-12H2,(H,15,16)/t6-/m0/s1    InChI=1S/C8H8O4/c9-6-3-1-2-5(4-6)7(10)8(11)12/h1-4,7,9-10H,(H,11,12)/t7-/m0/s1    InChI=1S/C8H8O3/c9-7-3-1-2-6(4-7)5-8(10)11/h1-4,9H,5H2,(H,10,11)    InChI=1S/C8H7NO4S/c10-14(11,12)13-8-5-9-7-4-2-1-3-6(7)8/h1-5,9H,(H,10,11,12)    InChI=1S/C6H10O3/c1-3-4(2)5(7)6(8)9/h4H,3H2,1-2H3,(H,8,9)    InChI=1S/C7H12O4/c1-5(4-7(10)11)2-3-6(8)9/h5H,2-4H2,1H3,(H,8,9)(H,10,11)/t5-/m1/s1    InChI=1S/C6H10O4/c1-4(2-5(7)8)3-6(9)10/h4H,2-3H2,1H3,(H,7,8)(H,9,10)    InChI=1S/C6H6N4O2/c1-10-4-3(7-2-8-4)5(11)9-6(10)12/h2H,1H3,(H,7,8)(H,9,11,12)    InChI=1S/C9H10O3/c10-8(9(11)12)6-7-4-2-1-3-5-7/h1-5,8,10H,6H2,(H,11,12)/t8-/m1/s1    InChI=1S/C9H10O2/c10-9(11)7-6-8-4-2-1-3-5-8/h1-5H,6-7H2,(H,10,11)    InChI=1S/C4H9NO2/c5-3-1-2-4(6)7/h1-3,5H2,(H,6,7)    InChI=1S/C9H10N2O3/c10-7-3-1-6(2-4-7)9(14)11-5-8(12)13/h1-4H,5,10H2,(H,11,14)(H,12,13)    InChI=1S/C6H9NO6/c7-3(6(12)13)1-2(4(8)9)5(10)11/h2-3H,1,7H2,(H,8,9)(H,10,11)(H,12,13)/t3-/m0/s1    InChI=1S/C9H10O5/c1-14-7-4-5(2-3-6(7)10)8(11)9(12)13/h2-4,8,10-11H,1H3,(H,12,13)/t8-/m0/s1    InChI=1S/C7H6O3/c8-6-3-1-5(2-4-6)7(9)10/h1-4,8H,(H,9,10)    InChI=1S/C4H8O3/c5-3-1-2-4(6)7/h5H,1-3H2,(H,6,7)    InChI=1S/C8H8O3/c9-7-3-1-6(2-4-7)5-8(10)11/h1-4,9H,5H2,(H,10,11)    InChI=1S/C9H10O4/c10-7-3-1-6(2-4-7)5-8(11)9(12)13/h1-4,8,10-11H,5H2,(H,12,13)/t8-/m1/s1    InChI=1S/C8H9NO4/c1-4-7(11)6(8(12)13)5(3-10)2-9-4/h2,10-11H,3H2,1H3,(H,12,13)    InChI=1S/C5H8N2O2/c1-3-2-6-5(9)7-4(3)8/h3H,2H2,1H3,(H2,6,7,8,9)    InChI=1S/C4H6N2O2/c7-3-1-2-5-4(8)6-3/h1-2H2,(H2,5,6,7,8)    InChI=1S/C5H9NO3/c6-3-4(7)1-2-5(8)9/h1-3,6H2,(H,8,9)    InChI=1S/C10H9NO3/c12-7-1-2-9-8(4-7)6(5-11-9)3-10(13)14/h1-2,4-5,11-12H,3H2,(H,13,14)    InChI=1S/C6H14N2O3/c7-3-4(9)1-2-5(8)6(10)11/h4-5,9H,1-3,7-8H2,(H,10,11)/t4-,5+/m1/s1    InChI=1S/C11H12N2O3/c12-9(11(15)16)3-6-5-13-10-2-1-7(14)4-8(6)10/h1-2,4-5,9,13-14H,3,12H2,(H,15,16)/t9-/m0/s1    InChI=1S/C8H8O4/c1-12-5-2-3-7(9)6(4-5)8(10)11/h2-4,9H,1H3,(H,10,11)    InChI=1S/C6H5NO3/c8-5-2-1-4(3-7-5)6(9)10/h1-3H,(H,7,8)(H,9,10)    InChI=1S/C10H15N5O10P2/c11-8-5-9(13-2-12-8)15(3-14-5)10-7(17)6(16)4(24-10)1-23-27(21,22)25-26(18,19)20/h2-4,6-7,10,16-17H,1H2,(H,21,22)(H2,11,12,13)(H2,18,19,20)/t4-,6-,7-,10-/m1/s1    InChI=1S/C10H14N5O7P/c11-8-5-9(13-2-12-8)15(3-14-5)10-7(17)6(16)4(22-10)1-21-23(18,19)20/h2-4,6-7,10,16-17H,1H2,(H2,11,12,13)(H2,18,19,20)/t4-,6-,7-,10-/m1/s1    InChI=1S/C10H16N5O13P3/c11-8-5-9(13-2-12-8)15(3-14-5)10-7(17)6(16)4(26-10)1-25-30(21,22)28-31(23,24)27-29(18,19)20/h2-4,6-7,10,16-17H,1H2,(H,21,22)(H,23,24)(H2,11,12,13)(H2,18,19,20)/t4-,6-,7-,10-/m1/s1    InChI=1S/C2H5NO/c1-2(3)4/h1H3,(H2,3,4)    InChI=1S/C8H9NO2/c1-6(10)9-7-2-4-8(11)5-3-7/h2-5,11H,1H3,(H,9,10)    InChI=1S/C2H4O2/c1-2(3)4/h1H3,(H,3,4)    InChI=1S/C4H6O3/c1-3(5)2-4(6)7/h2H2,1H3,(H,6,7)    InChI=1S/C3H6O/c1-3(2)4/h1-2H3    InChI=1S/C9H8O4/c1-6(10)13-8-5-3-2-4-7(8)9(11)12/h2-5H,1H3,(H,11,12)    InChI=1S/C5H5N5/c6-4-3-5(9-1-7-3)10-2-8-4/h1-2H,(H3,6,7,8,9,10)    InChI=1S/C10H13N5O4/c11-8-5-9(13-2-12-8)15(3-14-5)10-7(18)6(17)4(1-16)19-10/h2-4,6-7,10,16-18H,1H2,(H2,11,12,13)/t4-,6-,7-,10-/m1/s1    InChI=1S/C6H10O4/c7-5(8)3-1-2-4-6(9)10/h1-4H2,(H,7,8)(H,9,10)    InChI=1S/C3H7NO2/c1-2(4)3(5)6/h2H,4H2,1H3,(H,5,6)/t2-/m0/s1    InChI=1S/C4H6N4O3/c5-3(10)6-1-2(9)8-4(11)7-1/h1H,(H3,5,6,10)(H2,7,8,9,11)    InChI=1S/C6H13NO2/c1-3-4(2)5(7)6(8)9/h4-5H,3,7H2,1-2H3,(H,8,9)/t4-,5+/m1/s1    InChI=1S/C10H16N4O3/c1-14-6-12-5-7(14)4-8(10(16)17)13-9(15)2-3-11/h5-6,8H,2-4,11H2,1H3,(H,13,15)(H,16,17)/t8-/m0/s1    InChI=1S/C7H7NO2/c8-6-4-2-1-3-5(6)7(9)10/h1-4H,8H2,(H,9,10)    InChI=1S/C6H14N4O2/c7-4(5(11)12)2-1-3-10-6(8)9/h4H,1-3,7H2,(H,11,12)(H4,8,9,10)/t4-/m0/s1    InChI=1S/C6H8O6/c7-1-2(8)5-3(9)4(10)6(11)12-5/h2,5,7-10H,1H2/t2-,5+/m0/s1    InChI=1S/C4H8N2O3/c5-2(4(8)9)1-3(6)7/h2H,1,5H2,(H2,6,7)(H,8,9)/t2-/m0/s1    InChI=1S/C4H7NO4/c5-2(4(8)9)1-3(6)7/h2H,1,5H2,(H,6,7)(H,8,9)/t2-/m0/s1    InChI=1S/C9H16O4/c10-8(11)6-4-2-1-3-5-7-9(12)13/h1-7H2,(H,10,11)(H,12,13)    InChI=1S/C7H6O2/c8-7(9)6-4-2-1-3-5-6/h1-5H,(H,8,9)    InChI=1S/C5H11NO2/c1-6(2,3)4-5(7)8/h4H2,1-3H3    InChI=1S/C10H16N2O3S/c13-8(14)4-2-1-3-7-9-6(5-16-7)11-10(15)12-9/h6-7,9H,1-5H2,(H,13,14)(H2,11,12,15)/t6-,7-,9-/m0/s1    InChI=1S/C4H8O/c1-3-4(2)5/h3H2,1-2H3    InChI=1S/C4H8O2/c1-2-3-4(5)6/h2-3H2,1H3,(H,5,6)    InChI=1S/C5H14N2/c6-4-2-1-3-5-7/h1-7H2    InChI=1S/C8H10N4O2/c1-10-4-9-6-5(10)7(13)12(3)8(14)11(6)2/h4H,1-3H3    InChI=1S/C10H20O2/c1-2-3-4-5-6-7-8-9-10(11)12/h2-9H2,1H3,(H,11,12)    InChI=1S/C8H16O2/c1-2-3-4-5-6-7-8(9)10/h2-7H2,1H3,(H,9,10)    InChI=1S/C7H15NO3/c1-8(2,3)5-6(9)4-7(10)11/h6,9H,4-5H2,1-3H3/t6-/m1/s1    InChI=1S/C9H14N4O3/c10-2-1-8(14)13-7(9(15)16)3-6-4-11-5-12-6/h4-5,7H,1-3,10H2,(H,11,12)(H,13,14)(H,15,16)/t7-/m0/s1    InChI=1S/C6H6O2/c7-5-3-1-2-4-6(5)8/h1-4,7-8H    InChI=1S/C24H40O5/c1-13(4-7-21(28)29)16-5-6-17-22-18(12-20(27)24(16,17)3)23(2)9-8-15(25)10-14(23)11-19(22)26/h13-20,22,25-27H,4-12H2,1-3H3,(H,28,29)/t13-,14+,15-,16-,17+,18+,19-,20+,22+,23+,24-/m1/s1    InChI=1S/C5H14NO/c1-6(2,3)4-5-7/h7H,4-5H2,1-3H3/q+1    InChI=1S/C9H8O2/c10-9(11)7-6-8-4-2-1-3-5-8/h1-7H,(H,10,11)/b7-6+    InChI=1S/C6H8O7/c7-3(8)1-6(13,5(11)12)2-4(9)10/h13H,1-2H2,(H,7,8)(H,9,10)(H,11,12)    InChI=1S/C6H13N3O3/c7-4(5(10)11)2-1-3-9-6(8)12/h4H,1-3,7H2,(H,10,11)(H3,8,9,12)/t4-/m0/s1    InChI=1S/C4H9N3O2/c1-7(4(5)6)2-3(8)9/h2H2,1H3,(H3,5,6)(H,8,9)    InChI=1S/C4H10N3O5P/c1-7(2-3(8)9)4(5)6-13(10,11)12/h2H2,1H3,(H,8,9)(H4,5,6,10,11,12)    InChI=1S/C4H7N3O/c1-7-2-3(8)6-4(7)5/h2H2,1H3,(H2,5,6,8)    InChI=1S/C7H14N2O4S/c8-4(6(10)11)1-2-14-3-5(9)7(12)13/h4-5H,1-3,8-9H2,(H,10,11)(H,12,13)/t4-,5-/m0/s1    InChI=1S/C3H7NO2S/c4-2(1-7)3(5)6/h2,7H,1,4H2,(H,5,6)/t2-/m0/s1    InChI=1S/C6H12N2O4S2/c7-3(5(9)10)1-13-14-2-4(8)6(11)12/h3-4H,1-2,7-8H2,(H,9,10)(H,11,12)/t3-,4-/m0/s1    InChI=1S/C9H13N3O5/c10-5-1-2-12(9(16)11-5)8-7(15)6(14)4(3-13)17-8/h1-2,4,6-8,13-15H,3H2,(H2,10,11,16)/t4-,6-,7-,8-/m1/s1    InChI=1S/C4H5N3O/c5-3-1-2-6-4(8)7-3/h1-2H,(H3,5,6,7,8)    InChI=1S/C6H16O3SSi/c1-11(2,3)6-4-5-10(7,8)9/h4-6H2,1-3H3,(H,7,8,9)    InChI=1S/C6H16O3SSi/c1-11(2,3)6-4-5-10(7,8)9/h4-6H2,1-3H3,(H,7,8,9)/i4D2,5D2,6D2    InChI=1S/C2H7N/c1-3-2/h3H,1-2H3    InChI=1S/C15H14O6/c16-8-4-11(18)9-6-13(20)15(21-14(9)5-8)7-1-2-10(17)12(19)3-7/h1-5,13,15-20H,6H2/t13-,15-/m1/s1    InChI=1S/C2H6O/c1-2-3/h3H,2H2,1H3    InChI=1S/C2H7NO/c3-1-2-4/h4H,1-3H2    InChI=1S/C2H6O2/c3-1-2-4/h3-4H,1-2H2    InChI=1S/C5H8O4/c1-2-3(4(6)7)5(8)9/h3H,2H2,1H3,(H,6,7)(H,8,9)    InChI=1S/C10H10O4/c1-14-9-6-7(2-4-8(9)11)3-5-10(12)13/h2-6,11H,1H3,(H,12,13)/b5-3+    InChI=1S/CH2O2/c2-1-3/h1H,(H,2,3)    InChI=1S/C6H12O6/c7-1-3-4(9)5(10)6(11,2-8)12-3/h3-5,7-11H,1-2H2/t3-,4-,5+,6?/m1/s1    InChI=1S/C6H12O5/c1-2-3(7)4(8)5(9)6(10)11-2/h2-10H,1H3/t2-,3+,4+,5-,6?/m0/s1    InChI=1S/C4H4O4/c5-3(6)1-2-4(7)8/h1-2H,(H,5,6)(H,7,8)/b2-1+    InChI=1S/C10H16N5O14P3/c11-10-13-7-4(8(18)14-10)12-2-15(7)9-6(17)5(16)3(27-9)1-26-31(22,23)29-32(24,25)28-30(19,20)21/h2-3,5-6,9,16-17H,1H2,(H,22,23)(H,24,25)(H2,19,20,21)(H3,11,13,14,18)/t3-,5-,6-,9-/m1/s1    InChI=1S/C6H10O8/c7-1(3(9)5(11)12)2(8)4(10)6(13)14/h1-4,7-10H,(H,11,12)(H,13,14)/t1-,2+,3+,4-    InChI=1S/C6H14O6/c7-1-3(9)5(11)6(12)4(10)2-8/h3-12H,1-2H2/t3-,4+,5+,6-    InChI=1S/C6H12O7/c7-1-2(8)3(9)4(10)5(11)6(12)13/h2-5,7-11H,1H2,(H,12,13)/t2-,3+,4+,5-/m1/s1    InChI=1S/C6H12O6/c7-1-2-3(8)4(9)5(10)6(11)12-2/h2-11H,1H2/t2-,3+,4+,5-,6?/m1/s1    InChI=1S/C7H6O4/c8-4-1-2-6(9)5(3-4)7(10)11/h1-3,8-9H,(H,10,11)    InChI=1S/C6H10O8/c7-1(3(9)5(11)12)2(8)4(10)6(13)14/h1-4,7-10H,(H,11,12)(H,13,14)/t1-,2-,3-,4+/m0/s1    InChI=1S/C6H14O6/c7-1-3(9)5(11)6(12)4(10)2-8/h3-12H,1-2H2/t3-,4+,5-,6-/m1/s1    InChI=1S/C6H12O7/c7-1-2(8)3(9)4(10)5(11)6(12)13/h2-5,7-11H,1H2,(H,12,13)/t2-,3-,4+,5-/m1/s1    InChI=1S/C6H12O6/c7-1-2-3(8)4(9)5(10)6(11)12-2/h2-11H,1H2/t2-,3-,4+,5-,6-/m1/s1    InChI=1S/C5H9NO4/c6-3(5(9)10)1-2-4(7)8/h3H,1-2,6H2,(H,7,8)(H,9,10)/t3-/m0/s1    InChI=1S/C5H10N2O3/c6-3(5(9)10)1-2-4(7)8/h3H,1-2,6H2,(H2,7,8)(H,9,10)/t3-/m0/s1    InChI=1S/C5H8O4/c6-4(7)2-1-3-5(8)9/h1-3H2,(H,6,7)(H,8,9)    InChI=1S/C6H10O4/c1-10-6(9)4-2-3-5(7)8/h2-4H2,1H3,(H,7,8)    InChI=1S/C10H17N3O6S/c11-5(10(18)19)1-2-7(14)13-6(4-20)9(17)12-3-8(15)16/h5-6,20H,1-4,11H2,(H,12,17)(H,13,14)(H,15,16)(H,18,19)/t5-,6-/m0/s1    InChI=1S/C3H6O4/c4-1-2(5)3(6)7/h2,4-5H,1H2,(H,6,7)/t2-/m1/s1    InChI=1S/C3H8O3/c4-1-3(6)2-5/h3-6H,1-2H2    InChI=1S/C2H5NO2/c3-1-2(4)5/h1,3H2,(H,4,5)    InChI=1S/C26H43NO6/c1-14(4-7-22(31)27-13-23(32)33)17-5-6-18-24-19(12-21(30)26(17,18)3)25(2)9-8-16(28)10-15(25)11-20(24)29/h14-21,24,28-30H,4-13H2,1-3H3,(H,27,31)(H,32,33)/t14-,15+,16-,17-,18+,19+,20-,21+,24+,25+,26-/m1/s1    InChI=1S/C2H4O3/c3-1-2(4)5/h3H,1H2,(H,4,5)    InChI=1S/C7H12N2O3/c8-4-6(10)9-3-1-2-5(9)7(11)12/h5H,1-4,8H2,(H,11,12)/t5-/m0/s1    InChI=1S/C3H7N3O2/c4-3(5)6-1-2(7)8/h1H2,(H,7,8)(H4,4,5,6)    InChI=1S/C10H13N5O5/c11-10-13-7-4(8(19)14-10)12-2-15(7)9-6(18)5(17)3(1-16)20-9/h2-3,5-6,9,16-18H,1H2,(H3,11,13,14,19)/t3-,5-,6-,9-/m1/s1    InChI=1S/C9H9NO3/c11-8(12)6-10-9(13)7-4-2-1-3-5-7/h1-5H,6H2,(H,10,13)(H,11,12)    InChI=1S/C5H9N3/c6-2-1-5-3-7-4-8-5/h3-4H,1-2,6H2,(H,7,8)    InChI=1S/C6H9N3O2/c7-5(6(10)11)1-4-2-8-3-9-4/h2-3,5H,1,7H2,(H,8,9)(H,10,11)/t5-/m0/s1    InChI=1S/C7H15N3O3/c8-5(6(11)12)3-1-2-4-10-7(9)13/h5H,1-4,8H2,(H,11,12)(H3,9,10,13)/t5-/m0/s1    InChI=1S/C4H9NO2S/c5-3(1-2-8)4(6)7/h3,8H,1-2,5H2,(H,6,7)/t3-/m0/s1    InChI=1S/C8H16N2O4S2/c9-5(7(11)12)1-3-15-16-4-2-6(10)8(13)14/h5-6H,1-4,9-10H2,(H,11,12)(H,13,14)/t5-,6-/m0/s1    InChI=1S/C8H8O4/c9-6-1-2-7(10)5(3-6)4-8(11)12/h1-3,9-10H,4H2,(H,11,12)    InChI=1S/C4H9NO3/c5-3(1-2-6)4(7)8/h3,6H,1-2,5H2,(H,7,8)/t3-/m0/s1    InChI=1S/C9H10O4/c1-13-8-4-6(5-9(11)12)2-3-7(8)10/h2-4,10H,5H2,1H3,(H,11,12)    InChI=1S/C5H4N4O/c10-5-3-4(7-1-6-3)8-2-9-5/h1-2H,(H2,6,7,8,9,10)    InChI=1S/C13H18O2/c1-9(2)8-11-4-6-12(7-5-11)10(3)13(14)15/h4-7,9-10H,8H2,1-3H3,(H,14,15)    InChI=1S/C3H4N2/c1-2-5-3-4-1/h1-3H,(H,4,5)    InChI=1S/C10H9NO2/c12-10(13)5-7-6-11-9-4-2-1-3-8(7)9/h1-4,6,11H,5H2,(H,12,13)    InChI=1S/C11H11NO3/c13-10(11(14)15)5-7-6-12-9-4-2-1-3-8(7)9/h1-4,6,10,12-13H,5H2,(H,14,15)/t10-/m0/s1    InChI=1S/C10H12N4O5/c15-1-4-6(16)7(17)10(19-4)14-3-13-5-8(14)11-2-12-9(5)18/h2-4,6-7,10,15-17H,1H2,(H,11,12,18)/t4-,6-,7-,10-/m1/s1    InChI=1S/C4H8O2/c1-3(2)4(5)6/h3H,1-2H3,(H,5,6)    InChI=1S/C6H12O2/c1-5(2)3-4-6(7)8/h5H,3-4H2,1-2H3,(H,7,8)    InChI=1S/C6H8O7/c7-3(8)1-2(5(10)11)4(9)6(12)13/h2,4,9H,1H2,(H,7,8)(H,10,11)(H,12,13)/t2-,4+/m1/s1    InChI=1S/C10H12O2/c1-3-4-8-5-6-9(11)10(7-8)12-2/h3-7,11H,1-2H3/b4-3+    InChI=1S/C6H13NO2/c1-3-4(2)5(7)6(8)9/h4-5H,3,7H2,1-2H3,(H,8,9)/t4-,5-/m0/s1    InChI=1S/C3H8O/c1-3(2)4/h3-4H,1-2H3    InChI=1S/C5H10O2/c1-4(2)3-5(6)7/h4H,3H2,1-2H3,(H,6,7)    InChI=1S/C10H7NO3/c12-9-5-8(10(13)14)11-7-4-2-1-3-6(7)9/h1-5H,(H,11,12)(H,13,14)    InChI=1S/C10H12N2O3/c11-7-4-2-1-3-6(7)9(13)5-8(12)10(14)15/h1-4,8H,5,11-12H2,(H,14,15)/t8-/m0/s1    InChI=1S/C3H6O3/c1-2(4)3(5)6/h2,4H,1H3,(H,5,6)/t2-/m0/s1    InChI=1S/C12H22O11/c13-1-3-5(15)6(16)9(19)12(22-3)23-10-4(2-14)21-11(20)8(18)7(10)17/h3-20H,1-2H2/t3-,4-,5+,6+,7-,8-,9-,10-,11?,12+/m1/s1    InChI=1S/C6H13NO2/c1-4(2)3-5(7)6(8)9/h4-5H,3,7H2,1-2H3,(H,8,9)/t5-/m0/s1    InChI=1S/C5H8O3/c1-4(6)2-3-5(7)8/h2-3H2,1H3,(H,7,8)    InChI=1S/C6H14N2O2/c7-4-2-1-3-5(8)6(9)10/h5H,1-4,7-8H2,(H,9,10)/t5-/m0/s1    InChI=1S/C4H6O5/c5-2(4(8)9)1-3(6)7/h2,5H,1H2,(H,6,7)(H,8,9)    InChI=1S/C4H4O4/c5-3(6)1-2-4(7)8/h1-2H,(H,5,6)(H,7,8)/b2-1-    InChI=1S/C3H4O4/c4-2(5)1-3(6)7/h1H2,(H,4,5)(H,6,7)    InChI=1S/C12H22O11/c13-1-3-5(15)6(16)9(19)12(22-3)23-10-4(2-14)21-11(20)8(18)7(10)17/h3-20H,1-2H2/t3-,4-,5-,6+,7-,8-,9-,10-,11?,12-/m1/s1    InChI=1S/C8H8O3/c9-7(8(10)11)6-4-2-1-3-5-6/h1-5,7,9H,(H,10,11)/t7-/m0/s1    InChI=1S/C6H14O6/c7-1-3(9)5(11)6(12)4(10)2-8/h3-12H,1-2H2/t3-,4-,5-,6-/m1/s1    InChI=1S/C6H12O6/c7-1-2-3(8)4(9)5(10)6(11)12-2/h2-11H,1H2/t2-,3-,4+,5+,6?/m1/s1    InChI=1S/CH4O/c1-2/h2H,1H3    InChI=1S/C5H11NO2S/c1-9-3-2-4(6)5(7)8/h4H,2-3,6H2,1H3,(H,7,8)/t4-/m0/s1    InChI=1S/CH5N/c1-2/h2H2,1H3    InChI=1S/C2H7N3/c1-5-2(3)4/h1H3,(H4,3,4,5)    InChI=1S/C4H6O4/c1-2(3(5)6)4(7)8/h2H,1H3,(H,5,6)(H,7,8)    InChI=1S/C5H8O4/c1-3(5(8)9)2-4(6)7/h3H,2H2,1H3,(H,6,7)(H,8,9)/t3-/m1/s1    InChI=1S/C3H7NO/c1-4(2)3-5/h3H,1-2H3    InChI=1S/C4H9NO2/c1-5(2)3-4(6)7/h3H2,1-2H3,(H,6,7)    InChI=1S/C6H9NO5/c1-3(8)7-4(6(11)12)2-5(9)10/h4H,2H2,1H3,(H,7,8)(H,9,10)(H,11,12)/t4-/m0/s1    InChI=1S/C5H9NO3S/c1-3(7)6-4(2-10)5(8)9/h4,10H,2H2,1H3,(H,6,7)(H,8,9)/t4-/m0/s1    InChI=1S/C7H11NO5/c1-4(9)8-5(7(12)13)2-3-6(10)11/h5H,2-3H2,1H3,(H,8,9)(H,10,11)(H,12,13)/t5-/m0/s1    InChI=1S/C7H12N2O4/c1-4(10)9-5(7(12)13)2-3-6(8)11/h5H,2-3H2,1H3,(H2,8,11)(H,9,10)(H,12,13)/t5-/m0/s1    InChI=1S/C4H7NO3/c1-3(6)5-2-4(7)8/h2H2,1H3,(H,5,6)(H,7,8)    InChI=1S/C7H14N2O3/c1-5(10)9-6(7(11)12)3-2-4-8/h6H,2-4,8H2,1H3,(H,9,10)(H,11,12)/t6-/m0/s1    InChI=1S/C12H14N2O2/c1-8(15)13-5-4-9-7-14-12-3-2-10(16)6-11(9)12/h2-3,6-7,14,16H,4-5H2,1H3,(H,13,15)    InChI=1S/C11H13NO4/c1-7(13)12-10(11(15)16)6-8-2-4-9(14)5-3-8/h2-5,10,14H,6H2,1H3,(H,12,13)(H,15,16)/t10-/m0/s1    InChI=1S/C4H8N2O3/c5-4(9)6-2-1-3(7)8/h1-2H2,(H,7,8)(H3,5,6,9)    InChI=1S/C5H8N2O5/c6-5(12)7-2(4(10)11)1-3(8)9/h2H,1H2,(H,8,9)(H,10,11)(H3,6,7,12)/t2-/m0/s1    InChI=1S/C7H13NO3/c1-5(2)3-6(9)8-4-7(10)11/h5H,3-4H2,1-2H3,(H,8,9)(H,10,11)    InChI=1S/C4H6N2O2/c1-6-2-3(7)5-4(6)8/h2H2,1H3,(H,5,7,8)    InChI=1S/C2H6N2O/c1-4(2)3-5/h1-2H3    InChI=1S/C10H11NO3/c12-9(11-7-10(13)14)6-8-4-2-1-3-5-8/h1-5H,6-7H2,(H,11,12)(H,13,14)    InChI=1S/C17H17NO3/c19-16(12-14-9-5-2-6-10-14)18-15(17(20)21)11-13-7-3-1-4-8-13/h1-10,15H,11-12H2,(H,18,19)(H,20,21)/t15-/m0/s1    InChI=1S/C8H16N2O3/c1-6(11)10-5-3-2-4-7(9)8(12)13/h7H,2-5,9H2,1H3,(H,10,11)(H,12,13)/t7-/m0/s1    InChI=1S/C21H27N7O14P2/c22-17-12-19(25-7-24-17)28(8-26-12)21-16(32)14(30)11(41-21)6-39-44(36,37)42-43(34,35)38-5-10-13(29)15(31)20(40-10)27-3-1-2-9(4-27)18(23)33/h1-4,7-8,10-11,13-16,20-21,29-32H,5-6H2,(H5-,22,23,24,25,33,34,35,36,37)/t10-,11-,13-,14-,15-,16-,20-,21-/m1/s1    InChI=1S/C21H29N7O14P2/c22-17-12-19(25-7-24-17)28(8-26-12)21-16(32)14(30)11(41-21)6-39-44(36,37)42-43(34,35)38-5-10-13(29)15(31)20(40-10)27-3-1-2-9(4-27)18(23)33/h1,3-4,7-8,10-11,13-16,20-21,29-32H,2,5-6H2,(H2,23,33)(H,34,35)(H,36,37)(H2,22,24,25)/t10-,11-,13-,14-,15-,16-,20-,21-/m1/s1    InChI=1S/C21H28N7O17P3/c22-17-12-19(25-7-24-17)28(8-26-12)21-16(44-46(33,34)35)14(30)11(43-21)6-41-48(38,39)45-47(36,37)40-5-10-13(29)15(31)20(42-10)27-3-1-2-9(4-27)18(23)32/h1-4,7-8,10-11,13-16,20-21,29-31H,5-6H2,(H7-,22,23,24,25,32,33,34,35,36,37,38,39)/t10-,11-,13-,14-,15-,16-,20-,21-/m1/s1    InChI=1S/C21H30N7O17P3/c22-17-12-19(25-7-24-17)28(8-26-12)21-16(44-46(33,34)35)14(30)11(43-21)6-41-48(38,39)45-47(36,37)40-5-10-13(29)15(31)20(42-10)27-3-1-2-9(4-27)18(23)32/h1,3-4,7-8,10-11,13-16,20-21,29-31H,2,5-6H2,(H2,23,32)(H,36,37)(H,38,39)(H2,22,24,25)(H2,33,34,35)/t10-,11-,13-,14-,15-,16-,20-,21-/m1/s1    InChI=1S/C6H6N2O/c7-6(9)5-2-1-3-8-4-5/h1-4H,(H2,7,9)    InChI=1S/C6H6N2O2/c7-6(9)5-2-1-3-8(10)4-5/h1-4H,(H2,7,9)    InChI=1S/C6H5NO2/c8-6(9)5-2-1-3-7-4-5/h1-4H,(H,8,9)    InChI=1S/C21H26N6O15P2/c22-17-12-18(24-7-23-17)27(8-25-12)20-16(31)14(29)11(41-20)6-39-44(36,37)42-43(34,35)38-5-10-13(28)15(30)19(40-10)26-3-1-2-9(4-26)21(32)33/h1-4,7-8,10-11,13-16,19-20,28-31H,5-6H2,(H4-,22,23,24,32,33,34,35,36,37)/t10-,11-,13-,14-,15-,16-,19-,20-/m1/s1    InChI=1S/C8H8N2O3/c11-7(12)5-10-8(13)6-2-1-3-9-4-6/h1-4H,5H2,(H,10,13)(H,11,12)    InChI=1S/C9H17NO4/c1-7(11)14-8(5-9(12)13)6-10(2,3)4/h8H,5-6H2,1-4H3/t8-/m1/s1    InChI=1S/C7H16NO2/c1-7(9)10-6-5-8(2,3)4/h5-6H2,1-4H3/q+1    InChI=1S/C5H14NO4P/c1-6(2,3)4-5-10-11(7,8)9/h4-5H2,1-3H3,(H-,7,8,9)    InChI=1S/C2H8NO4P/c3-1-2-7-8(4,5)6/h1-3H2,(H2,4,5,6)    InChI=1S/C3H8NO6P/c4-2(3(5)6)1-10-11(7,8)9/h2H,1,4H2,(H,5,6)(H2,7,8,9)/t2-/m0/s1    InChI=1S/C5H12N2O2/c6-3-1-2-4(7)5(8)9/h4H,1-3,6-7H2,(H,8,9)/t4-/m0/s1    InChI=1S/C5H4N4O2/c10-4-2-1-6-9-3(2)7-5(11)8-4/h1H,(H3,6,7,8,9,10,11)    InChI=1S/C9H17NO5/c1-9(2,5-11)7(14)8(15)10-4-3-6(12)13/h7,11,14H,3-5H2,1-2H3,(H,10,15)(H,12,13)/t7-/m0/s1    InChI=1S/C6H6O/c7-6-4-2-1-3-5-6/h1-5,7H    InChI=1S/C8H8O2/c9-8(10)6-7-4-2-1-3-5-7/h1-5H,6H2,(H,9,10)    InChI=1S/C9H11NO2/c10-8(9(11)12)6-7-4-2-1-3-5-7/h1-5,8H,6,10H2,(H,11,12)/t8-/m0/s1    InChI=1S/C8H6O4/c9-7(10)5-3-1-2-4-6(5)8(11)12/h1-4H,(H,9,10)(H,11,12)    InChI=1S/C7H12O4/c8-6(9)4-2-1-3-5-7(10)11/h1-5H2,(H,8,9)(H,10,11)    InChI=1S/C6H11NO2/c8-6(9)5-3-1-2-4-7-5/h5,7H,1-4H2,(H,8,9)/t5-/m0/s1    InChI=1S/C5H9NO2/c7-5(8)4-2-1-3-6-4/h4,6H,1-3H2,(H,7,8)/t4-/m0/s1    InChI=1S/C3H6O2/c1-2-3(4)5/h2H2,1H3,(H,4,5)    InChI=1S/C3H8O2/c1-3(5)2-4/h3-5H,2H2,1H3    InChI=1S/C7H6O4/c8-5-2-1-4(7(10)11)3-6(5)9/h1-3,8-9H,(H,10,11)    InChI=1S/C4H12N2/c5-3-1-2-4-6/h1-6H2    InChI=1S/C8H11NO3/c1-5-8(12)7(4-11)6(3-10)2-9-5/h2,10-12H,3-4H2,1H3    InChI=1S/C4H4N2/c1-2-5-4-6-3-1/h1-4H    InChI=1S/C5H7NO3/c7-4-2-1-3(6-4)5(8)9/h3H,1-2H2,(H,6,7)(H,8,9)/t3-/m0/s1    InChI=1S/C3H4O3/c1-2(4)3(5)6/h1H3,(H,5,6)    InChI=1S/C7H5NO4/c9-6(10)4-2-1-3-8-5(4)7(11)12/h1-3H,(H,9,10)(H,11,12)    InChI=1S/C17H20N4O6/c1-7-3-9-10(4-8(7)2)21(5-11(23)14(25)12(24)6-22)15-13(18-9)16(26)20-17(27)19-15/h3-4,11-12,14,22-25H,5-6H2,1-2H3,(H,20,26,27)/t11-,12+,14-/m1/s1    InChI=1S/C5H10O5/c6-1-2-3(7)4(8)5(9)10-2/h2-9H,1H2/t2-,3-,4-,5?/m1/s1    InChI=1S/C14H20N6O5S/c15-6(14(23)24)1-2-26-3-7-9(21)10(22)13(25-7)20-5-19-8-11(16)17-4-18-12(8)20/h4-7,9-10,13,21-22H,1-3,15H2,(H,23,24)(H2,16,17,18)/t6-,7+,9+,10+,13+/m0/s1    InChI=1S/C3H7NO5S2/c4-2(3(5)6)1-10-11(7,8)9/h2H,1,4H2,(H,5,6)(H,7,8,9)/t2-/m0/s1    InChI=1S/C11H20N2O6/c12-7(10(16)17)3-1-2-6-13-8(11(18)19)4-5-9(14)15/h7-8,13H,1-6,12H2,(H,14,15)(H,16,17)(H,18,19)/t7-,8-/m0/s1    InChI=1S/C7H6O3/c8-6-4-2-1-3-5(6)7(9)10/h1-4,8H,(H,9,10)    InChI=1S/C9H9NO4/c11-7-4-2-1-3-6(7)9(14)10-5-8(12)13/h1-4,11H,5H2,(H,10,14)(H,12,13)    InChI=1S/C3H7NO2/c1-4-2-3(5)6/h4H,2H2,1H3,(H,5,6)    InChI=1S/C10H18O4/c11-9(12)7-5-3-1-2-4-6-8-10(13)14/h1-8H2,(H,11,12)(H,13,14)    InChI=1S/C3H7NO3/c4-2(1-5)3(6)7/h2,5H,1,4H2,(H,6,7)/t2-/m0/s1    InChI=1S/C10H12N2O/c11-4-3-7-6-12-10-2-1-8(13)5-9(7)10/h1-2,5-6,12-13H,3-4,11H2    InChI=1S/C8H14O4/c9-7(10)5-3-1-2-4-6-8(11)12/h1-6H2,(H,9,10)(H,11,12)    InChI=1S/C4H6O4/c5-3(6)1-2-4(7)8/h1-2H2,(H,5,6)(H,7,8)    InChI=1S/C7H10O4/c1-5(8)4-6(9)2-3-7(10)11/h2-4H2,1H3,(H,10,11)    InChI=1S/C12H22O11/c13-1-4-6(16)8(18)9(19)11(21-4)23-12(3-15)10(20)7(17)5(2-14)22-12/h4-11,13-20H,1-3H2/t4-,5-,6-,7-,8+,9-,10+,11-,12+/m1/s1    InChI=1S/C4H6O6/c5-1(3(7)8)2(6)4(9)10/h1-2,5-6H,(H,7,8)(H,9,10)/t1-,2-/m1/s1    InChI=1S/C2H7NO3S/c3-1-2-7(4,5)6/h1-3H2,(H,4,5,6)    InChI=1S/C7H8N4O2/c1-10-5-4(8-3-9-5)6(12)11(2)7(10)13/h3H,1-2H3,(H,8,9)    InChI=1S/C4H8O5/c5-1-2(6)3(7)4(8)9/h2-3,5-7H,1H2,(H,8,9)/t2-,3+/m1/s1    InChI=1S/C4H9NO3/c1-2(6)3(5)4(7)8/h2-3,6H,5H2,1H3,(H,7,8)/t2-,3+/m1/s1    InChI=1S/C10H14N2O5/c1-5-3-12(10(16)11-9(5)15)8-2-6(14)7(4-13)17-8/h3,6-8,13-14H,2,4H2,1H3,(H,11,15,16)/t6-,7+,8+/m0/s1    InChI=1S/C5H6N2O2/c1-3-2-6-5(9)7-4(3)8/h2H,1H3,(H2,6,7,8,9)    InChI=1S/C10H14O/c1-7(2)9-5-4-8(3)6-10(9)11/h4-7,11H,1-3H3    InChI=1S/C7H11NO3/c1-3-5(2)7(11)8-4-6(9)10/h3H,4H2,1-2H3,(H,8,11)(H,9,10)/b5-3+    InChI=1S/C7H7NO2/c1-8-4-2-3-6(5-8)7(9)10/h2-5H,1H3    InChI=1S/C3H9N/c1-4(2)3/h1-3H3    InChI=1S/C3H9NO/c1-4(2,3)5/h1-3H3    InChI=1S/C9H10O3/c10-6-8(9(11)12)7-4-2-1-3-5-7/h1-5,8,10H,6H2,(H,11,12)/t8-/m1/s1    InChI=1S/C11H12N2O2/c12-9(11(14)15)5-7-6-13-10-4-2-1-3-8(7)10/h1-4,6,9,13H,5,12H2,(H,14,15)/t9-/m0/s1    InChI=1S/C8H11NO/c9-6-5-7-1-3-8(10)4-2-7/h1-4,10H,5-6,9H2    InChI=1S/C9H11NO3/c10-8(9(12)13)5-6-1-3-7(11)4-2-6/h1-4,8,11H,5,10H2,(H,12,13)/t8-/m0/s1    InChI=1S/C15H24N2O17P2/c18-3-5-8(20)10(22)12(24)14(32-5)33-36(28,29)34-35(26,27)30-4-6-9(21)11(23)13(31-6)17-2-1-7(19)16-15(17)25/h1-2,5-6,8-14,18,20-24H,3-4H2,(H,26,27)(H,28,29)(H,16,19,25)/t5-,6-,8+,9-,10+,11-,12-,13-,14-/m1/s1    InChI=1S/C15H24N2O17P2/c18-3-5-8(20)10(22)12(24)14(32-5)33-36(28,29)34-35(26,27)30-4-6-9(21)11(23)13(31-6)17-2-1-7(19)16-15(17)25/h1-2,5-6,8-14,18,20-24H,3-4H2,(H,26,27)(H,28,29)(H,16,19,25)/t5-,6-,8-,9-,10+,11-,12-,13-,14-/m1/s1    InChI=1S/C4H4N2O2/c7-3-1-2-5-4(8)6-3/h1-2H,(H2,5,6,7,8)    InChI=1S/CH4N2O/c2-1(3)4/h(H4,2,3,4)    InChI=1S/C9H12N2O6/c12-3-4-6(14)7(15)8(17-4)11-2-1-5(13)10-9(11)16/h1-2,4,6-8,12,14-15H,3H2,(H,10,13,16)/t4-,6-,7-,8-/m1/s1    InChI=1S/C6H6N2O2/c9-6(10)2-1-5-3-7-4-8-5/h1-4H,(H,7,8)(H,9,10)/b2-1+    InChI=1S/C5H10O2/c1-2-3-4-5(6)7/h2-4H2,1H3,(H,6,7)    InChI=1S/C5H11NO2/c1-3(2)4(6)5(7)8/h3-4H,6H2,1-2H3,(H,7,8)/t4-/m0/s1    InChI=1S/C8H16O2/c1-3-5-7(6-4-2)8(9)10/h7H,3-6H2,1-2H3,(H,9,10)    InChI=1S/C8H8O4/c1-12-7-4-5(8(10)11)2-3-6(7)9/h2-4,9H,1H3,(H,10,11)    InChI=1S/C5H4N4O2/c10-4-2-3(7-1-6-2)8-5(11)9-4/h1H,(H3,6,7,8,9,10,11)    InChI=1S/C10H12N4O6/c15-1-3-5(16)6(17)9(20-3)14-2-11-4-7(14)12-10(19)13-8(4)18/h2-3,5-6,9,15-17H,1H2,(H2,12,13,18,19)/t3-,5-,6-,9-/m1/s1    InChI=1S/C5H10O5/c6-2-1-10-5(9)4(8)3(2)7/h2-9H,1H2/t2-,3+,4-,5?/m1/s1    InChI=1S/C6H6O6/c7-4(8)1-3(6(11)12)2-5(9)10/h1H,2H2,(H,7,8)(H,9,10)(H,11,12)/b3-1-    InChI=1S/C9H16N3O13P3/c10-7-1-2-12(9(14)11-7)8-3-5(13)6(23-8)4-22-27(18,19)25-28(20,21)24-26(15,16)17/h1-2,5-6,8,13H,3-4H2,(H,18,19)(H,20,21)(H2,10,11,14)(H2,15,16,17)/t5-,6+,8+/m0/s1    InChI=1S/C10H17N2O14P3/c1-5-3-12(10(15)11-9(5)14)8-2-6(13)7(24-8)4-23-28(19,20)26-29(21,22)25-27(16,17)18/h3,6-8,13H,2,4H2,1H3,(H,19,20)(H,21,22)(H,11,14,15)(H2,16,17,18)/t6-,7+,8+/m0/s1    InChI=1S/C6H12O6/c7-1-2(8)4(10)6(12)5(11)3(1)9/h1-12H/t1-,2-,3-,4+,5-,6-    InChI=1S/C7H8O/c1-6-4-2-3-5-7(6)8/h2-5,8H,1H3    InChI=1S/C7H8O/c1-6-2-4-7(8)5-3-6/h2-5,8H,1H3    InChI=1S/C8H20NO6P/c1-9(2,3)4-5-14-16(12,13)15-7-8(11)6-10/h8,10-11H,4-7H2,1-3H3/t8-/m1/s1    InChI=1S/C5H9NO3/c7-3-1-4(5(8)9)6-2-3/h3-4,6-7H,1-2H2,(H,8,9)/t3-,4+/m1/s1    InChI=1S/C6H6O6/c7-4(8)1-3(6(11)12)2-5(9)10/h1H,2H2,(H,7,8)(H,9,10)(H,11,12)/b3-1+    InChI=1S/C3H7NO2/c4-2-1-3(5)6/h1-2,4H2,(H,5,6)    InChI=1S/C14H18N2O5/c15-10(13(18)19)6-7-12(17)16-11(14(20)21)8-9-4-2-1-3-5-9/h1-5,10-11H,6-8,15H2,(H,16,17)(H,18,19)(H,20,21)/t10-,11-/m0/s1    InChI=1S/C7H11N3O2/c1-10-4-9-3-5(10)2-6(8)7(11)12/h3-4,6H,2,8H2,1H3,(H,11,12)/t6-/m0/s1    InChI=1S/C7H11N3O2/c1-10-3-5(9-4-10)2-6(8)7(11)12/h3-4,6H,2,8H2,1H3,(H,11,12)/t6-/m0/s1
    #SMILES        OCC(=O)CO    CN1C(=O)N(C)C2=C(NC(=O)N2)C1=O    O[C@H]1[C@H](O)[C@H]2CO[C@H](O2)[C@@H]1O    C[N]1C=NC2=C1C(=O)N(C)C(=O)N2    C[N+]1=CC=CC(=C1)C(N)=O    NC1=NC=NC2=C1N=C[N]2[C@H]3C[C@H](O)[C@@H](CO)O3    NC1=NC2=C(N=C[N]2[C@H]3C[C@H](O)[C@@H](CO)O3)C(=O)N1    OC[C@H]1O[C@H](C[C@@H]1O)[N]2C=NC3=C2N=CNC3=O    N[C@@H](CCCC(O)=O)C(O)=O    CC[C@H](N)C(O)=O    CCC(=C)C(O)=O    OC(=O)C1=CC=CO1    CC[C@@H](C)[C@@H](O)C(O)=O    CC[C@H](O)C(O)=O    O[C@@H](CCC(O)=O)C(O)=O    CC(C)(O)C(O)=O    CC(C)C[C@H](O)C(O)=O    CC(C)[C@H](O)C(O)=O    OC(=O)CC1=C(O)C=CC=C1    CCC[C@H](O)C(O)=O    CC(CCC(O)=O)C(O)=O    CCCCC/C=C/C(O)=O    CCC(=O)C(O)=O    CCCCC(=O)C(O)=O    OC(=O)CCC(=O)C(O)=O    CC(C)CC(=O)C(O)=O    CCCC(=O)C(O)=O    CC(C(O)=O)C1=CC=CC=C1    OC[C@@H](OP(O)(O)=O)C(O)=O    O[C@H](C(O)=O)C1=CC=C(O)C(=C1)O    N[C@@H](CC1=CC(=C(O)C(=C1)Br)Br)C(O)=O    C[C@@H](CN)C(O)=O    N[C@@H](CC1=CC=C(O)C(=C1)Cl)C(O)=O    OC(=O)C/C=C/CC(O)=O    CC(O)(CC(O)=O)CC(O)=O    C[C@@H](O)CC(O)=O    CC(C)(O)CC(O)=O    N[C@@H](CC(=O)C1=C(N)C(=CC=C1)O)C(O)=O    O[C@H](C(O)=O)C1=CC(=CC=C1)O    OC(=O)CC1=CC(=CC=C1)O    OS(=O)(=O)OC1=C[NH]C2=CC=CC=C12    CCC(C)C(=O)C(O)=O    C[C@H](CCC(O)=O)CC(O)=O    CC(CC(O)=O)CC(O)=O    CN1C(=O)NC(=O)C2=C1N=C[NH]2    O[C@H](CC1=CC=CC=C1)C(O)=O    OC(=O)CCC1=CC=CC=C1    NCCCC(O)=O    NC1=CC=C(C=C1)C(=O)NCC(O)=O    N[C@@H](CC(C(O)=O)C(O)=O)C(O)=O    COC1=C(O)C=CC(=C1)[C@H](O)C(O)=O    OC(=O)C1=CC=C(O)C=C1    OCCCC(O)=O    OC(=O)CC1=CC=C(O)C=C1    O[C@H](CC1=CC=C(O)C=C1)C(O)=O    CC1=NC=C(CO)C(=C1O)C(O)=O    CC1CNC(=O)NC1=O    O=C1CCNC(=O)N1    NCC(=O)CCC(O)=O    OC(=O)CC1=C[NH]C2=C1C=C(O)C=C2    NC[C@H](O)CC[C@H](N)C(O)=O    NC(CC1=C[NH]C2=C1C=C(O)C=C2)C(O)=O    COC1=CC=C(O)C(=C1)C(O)=O    OC(=O)C1=CC=C(O)N=C1    NC1=NC=NC2=C1N=C[N]2[C@@H]3O[C@H](COP(O)(=O)OP(O)(O)=O)[C@@H](O)[C@H]3O    NC1=NC=NC2=C1N=C[N]2[C@@H]3O[C@H](COP(O)(O)=O)[C@@H](O)[C@H]3O    NC1=NC=NC2=C1N=C[N]2[C@@H]3O[C@H](COP(O)(=O)OP(O)(=O)OP(O)(O)=O)[C@@H](O)[C@H]3O    CC(N)=O    CC(=O)NC1=CC=C(O)C=C1    CC(O)=O    CC(=O)CC(O)=O    CC(C)=O    CC(=O)OC1=CC=CC=C1C(O)=O    NC1=NC=NC2=C1N=C[NH]2    NC1=NC=NC2=C1N=C[N]2[C@@H]3O[C@H](CO)[C@@H](O)[C@H]3O    OC(=O)CCCCC(O)=O    C[C@H](N)C(O)=O    NC(=O)NC1NC(=O)NC1=O    CC[C@@H](C)[C@H](N)C(O)=O    C[N]1C=NC=C1C[C@H](NC(=O)CCN)C(O)=O    NC1=CC=CC=C1C(O)=O    N[C@@H](CCCNC(N)=N)C(O)=O    OC[C@H](O)[C@H]1OC(=O)C(=C1O)O    N[C@@H](CC(N)=O)C(O)=O    N[C@@H](CC(O)=O)C(O)=O    OC(=O)CCCCCCCC(O)=O    OC(=O)C1=CC=CC=C1    C[N+](C)(C)CC([O-])=O    OC(=O)CCCC[C@@H]1SC[C@@H]2NC(=O)N[C@H]12    CCC(C)=O    CCCC(O)=O    NCCCCCN    C[N]1C=NC2=C1C(=O)N(C)C(=O)N2C    CCCCCCCCCC(O)=O    CCCCCCCC(O)=O    C[N+](C)(C)C[C@H](O)CC([O-])=O    NCCC(=O)N[C@@H](CC1=C[NH]C=N1)C(O)=O    OC1=CC=CC=C1O    C[C@H](CCC(O)=O)[C@H]1CC[C@H]2[C@@H]3[C@H](O)C[C@@H]4C[C@H](O)CC[C@]4(C)[C@H]3C[C@H](O)[C@]12C    C[N+](C)(C)CCO    OC(=O)/C=C/C1=CC=CC=C1    OC(=O)CC(O)(CC(O)=O)C(O)=O    N[C@@H](CCCNC(N)=O)C(O)=O    CN(CC(O)=O)C(N)=N    CN(CC(O)=O)C(=N)NP(O)(O)=O    CN1CC(=O)NC1=N    N[C@@H](CCSC[C@H](N)C(O)=O)C(O)=O    N[C@@H](CS)C(O)=O    N[C@@H](CSSC[C@H](N)C(O)=O)C(O)=O    NC1=NC(=O)N(C=C1)[C@@H]2O[C@H](CO)[C@@H](O)[C@H]2O    NC1=NC(=O)NC=C1    C[Si](C)(C)CCCS(O)(=O)=O    C[Si](C)(C)C(C(C(S(O)(=O)=O)([2H])[2H])([2H])[2H])([2H])[2H]    CNC    O[C@@H]1CC2=C(O)C=C(O)C=C2O[C@@H]1C3=CC=C(O)C(=C3)O    CCO    NCCO    OCCO    CCC(C(O)=O)C(O)=O    COC1=C(O)C=CC(=C1)/C=C/C(O)=O    OC=O    OC[C@H]1OC(O)(CO)[C@@H](O)[C@@H]1O    C[C@@H]1OC(O)[C@@H](O)[C@H](O)[C@@H]1O    OC(=O)/C=C/C(O)=O    NC1=NC2=C(N=C[N]2[C@@H]3O[C@H](COP(O)(=O)OP(O)(=O)OP(O)(O)=O)[C@@H](O)[C@H]3O)C(=O)N1    O[C@@H]([C@@H](O)[C@H](O)C(O)=O)[C@@H](O)C(O)=O    OC[C@H](O)[C@@H](O)[C@@H](O)[C@H](O)CO    OC[C@@H](O)[C@H](O)[C@H](O)[C@@H](O)C(O)=O    OC[C@H]1OC(O)[C@H](O)[C@@H](O)[C@H]1O    OC(=O)C1=CC(=CC=C1O)O    O[C@@H]([C@H](O)[C@@H](O)C(O)=O)[C@H](O)C(O)=O    OC[C@H](O)[C@@H](O)[C@H](O)[C@H](O)CO    OC[C@@H](O)[C@@H](O)[C@H](O)[C@@H](O)C(O)=O    OC[C@H]1O[C@@H](O)[C@H](O)[C@@H](O)[C@@H]1O    N[C@@H](CCC(O)=O)C(O)=O    N[C@@H](CCC(N)=O)C(O)=O    OC(=O)CCCC(O)=O    COC(=O)CCCC(O)=O    NC(CCC(=O)NC(CS)C(=O)NCC(O)=O)C(O)=O    OC[C@@H](O)C(O)=O    OCC(O)CO    NCC(O)=O    C[C@H](CCC(=O)NCC(O)=O)[C@H]1CC[C@H]2[C@@H]3[C@H](O)C[C@@H]4C[C@H](O)CC[C@]4(C)[C@H]3C[C@H](O)[C@]12C    OCC(O)=O    NCC(=O)N1CCC[C@H]1C(O)=O    NC(=N)NCC(O)=O    NC1=NC2=C(N=C[N]2[C@@H]3O[C@H](CO)[C@@H](O)[C@H]3O)C(=O)N1    OC(=O)CNC(=O)C1=CC=CC=C1    NCCC1=CN=C[NH]1    N[C@@H](CC1=CN=C[NH]1)C(O)=O    N[C@@H](CCCCNC(N)=O)C(O)=O    N[C@@H](CCS)C(O)=O    N[C@@H](CCSSCC[C@H](N)C(O)=O)C(O)=O    OC(=O)CC1=C(O)C=CC(=C1)O    N[C@@H](CCO)C(O)=O    COC1=C(O)C=CC(=C1)CC(O)=O    O=C1NC=NC2=C1N=C[NH]2    CC(C)CC1=CC=C(C=C1)C(C)C(O)=O    [NH]1C=CN=C1    OC(=O)CC1=C[NH]C2=CC=CC=C12    OC(CC1=C[NH]C2=C1C=CC=C2)C(O)=O    OC[C@H]1O[C@H]([C@H](O)[C@@H]1O)[N]2C=NC3=C2N=CNC3=O    CC(C)C(O)=O    CC(C)CCC(O)=O    O[C@@H]([C@@H](CC(O)=O)C(O)=O)C(O)=O    COC1=C(O)C=CC(=C1)/C=C/C    CC[C@H](C)[C@H](N)C(O)=O    CC(C)O    CC(C)CC(O)=O    OC(=O)C1=NC2=CC=CC=C2C(=C1)O    N[C@@H](CC(=O)C1=C(N)C=CC=C1)C(O)=O    C[C@H](O)C(O)=O    OC[C@H]1O[C@@H](O[C@H]2[C@H](O)[C@@H](O)C(O)O[C@@H]2CO)[C@H](O)[C@@H](O)[C@H]1O    CC(C)C[C@H](N)C(O)=O    CC(=O)CCC(O)=O    NCCCC[C@H](N)C(O)=O    OC(CC(O)=O)C(O)=O    OC(=O)/C=C\C(O)=O    OC(=O)CC(O)=O    OC[C@H]1O[C@H](O[C@H]2[C@H](O)[C@@H](O)C(O)O[C@@H]2CO)[C@H](O)[C@@H](O)[C@@H]1O    O[C@H](C(O)=O)C1=CC=CC=C1    OC[C@@H](O)[C@@H](O)[C@H](O)[C@H](O)CO    OC[C@H]1OC(O)[C@@H](O)[C@@H](O)[C@@H]1O    CO    CSCC[C@H](N)C(O)=O    CN    CNC(N)=N    CC(C(O)=O)C(O)=O    C[C@H](CC(O)=O)C(O)=O    CN(C)C=O    CN(C)CC(O)=O    CC(=O)N[C@@H](CC(O)=O)C(O)=O    CC(=O)N[C@@H](CS)C(O)=O    CC(=O)N[C@@H](CCC(O)=O)C(O)=O    CC(=O)N[C@@H](CCC(N)=O)C(O)=O    CC(=O)NCC(O)=O    CC(=O)N[C@@H](CCCN)C(O)=O    CC(=O)NCCC1=C[NH]C2=C1C=C(O)C=C2    CC(=O)N[C@@H](CC1=CC=C(O)C=C1)C(O)=O    NC(=O)NCCC(O)=O    NC(=O)N[C@@H](CC(O)=O)C(O)=O    CC(C)CC(=O)NCC(O)=O    CN1CC(=O)NC1=O    CN(C)N=O    OC(=O)CNC(=O)CC1=CC=CC=C1    OC(=O)[C@H](CC1=CC=CC=C1)NC(=O)CC2=CC=CC=C2    CC(=O)NCCCC[C@H](N)C(O)=O    NC(=O)C1=C[N+](=CC=C1)[C@@H]2O[C@H](COP(O)(=O)OP([O-])(=O)OC[C@H]3O[C@H]([C@H](O)[C@@H]3O)[N]4C=NC5=C4N=CN=C5N)[C@@H](O)[C@H]2O    NC(=O)C1=CN(C=CC1)[C@@H]2O[C@H](COP(O)(=O)OP(O)(=O)OC[C@H]3O[C@H]([C@H](O)[C@@H]3O)[N]4C=NC5=C4N=CN=C5N)[C@@H](O)[C@H]2O    NC(=O)C1=C[N+](=CC=C1)[C@@H]2O[C@H](COP(O)(=O)OP([O-])(=O)OC[C@H]3O[C@H]([C@H](OP(O)(O)=O)[C@@H]3O)[N]4C=NC5=C4N=CN=C5N)[C@@H](O)[C@H]2O    NC(=O)C1=CN(C=CC1)[C@@H]2O[C@H](COP(O)(=O)OP(O)(=O)OC[C@H]3O[C@H]([C@H](OP(O)(O)=O)[C@@H]3O)[N]4C=NC5=C4N=CN=C5N)[C@@H](O)[C@H]2O    NC(=O)C1=CC=CN=C1    NC(=O)C1=CC=C[N+](=C1)[O-]    OC(=O)C1=CC=CN=C1    NC1=NC=NC2=C1N=C[N]2[C@@H]3O[C@H](COP(O)(=O)OP(O)(=O)OC[C@H]4O[C@H]([C@H](O)[C@@H]4O)[N+]5=CC=CC(=C5)C([O-])=O)[C@@H](O)[C@H]3O    OC(=O)CNC(=O)C1=CC=CN=C1    CC(=O)O[C@H](CC([O-])=O)C[N+](C)(C)C    CC(=O)OCC[N+](C)(C)C    C[N+](C)(C)CCOP(O)([O-])=O    NCCOP(O)(O)=O    N[C@@H](COP(O)(O)=O)C(O)=O    NCCC[C@H](N)C(O)=O    OC1=NC(=C2C=N[NH]C2=N1)O    CC(C)(CO)[C@@H](O)C(=O)NCCC(O)=O    OC1=CC=CC=C1    OC(=O)CC1=CC=CC=C1    N[C@@H](CC1=CC=CC=C1)C(O)=O    OC(=O)C1=CC=CC=C1C(O)=O    OC(=O)CCCCCC(O)=O    OC(=O)[C@@H]1CCCCN1    OC(=O)[C@@H]1CCCN1    CCC(O)=O    CC(O)CO    OC(=O)C1=CC=C(O)C(=C1)O    NCCCCN    CC1=NC=C(CO)C(=C1O)CO    C1=CN=CN=C1    OC(=O)[C@@H]1CCC(=O)N1    CC(=O)C(O)=O    OC(=O)C1=CC=CN=C1C(O)=O    CC1=C(C)C=C2N(C[C@@H](O)[C@@H](O)[C@@H](O)CO)C3=NC(=O)NC(=O)C3=NC2=C1    OC[C@H]1OC(O)[C@H](O)[C@@H]1O    N[C@@H](CCSC[C@H]1O[C@H]([C@H](O)[C@@H]1O)[N]2C=NC3=C2N=CN=C3N)C(O)=O    N[C@@H](CSS(O)(=O)=O)C(O)=O    N[C@@H](CCCCN[C@@H](CCC(O)=O)C(O)=O)C(O)=O    OC(=O)C1=CC=CC=C1O    OC(=O)CNC(=O)C1=CC=CC=C1O    CNCC(O)=O    OC(=O)CCCCCCCCC(O)=O    N[C@@H](CO)C(O)=O    NCCC1=C[NH]C2=CC=C(O)C=C12    OC(=O)CCCCCCC(O)=O    OC(=O)CCC(O)=O    CC(=O)CC(=O)CCC(O)=O    OC[C@H]1O[C@H](O[C@]2(CO)O[C@H](CO)[C@@H](O)[C@@H]2O)[C@H](O)[C@@H](O)[C@@H]1O    O[C@H]([C@@H](O)C(O)=O)C(O)=O    NCCS(O)(=O)=O    CN1C(=O)N(C)C2=C([NH]C=N2)C1=O    OC[C@@H](O)[C@H](O)C(O)=O    C[C@@H](O)[C@H](N)C(O)=O    CC1=CN([C@H]2C[C@H](O)[C@@H](CO)O2)C(=O)NC1=O    CC1=CNC(=O)NC1=O    CC(C)C1=C(O)C=C(C)C=C1    C/C=C(\C)C(=O)NCC(O)=O    C[N+]1=CC=CC(=C1)C([O-])=O    CN(C)C    C[N+](C)(C)[O-]    OC[C@@H](C(O)=O)C1=CC=CC=C1    N[C@@H](CC1=C[NH]C2=CC=CC=C12)C(O)=O    NCCC1=CC=C(O)C=C1    N[C@@H](CC1=CC=C(O)C=C1)C(O)=O    OC[C@H]1O[C@H](OP(O)(=O)OP(O)(=O)OC[C@H]2O[C@H]([C@H](O)[C@@H]2O)N3C=CC(=O)NC3=O)[C@H](O)[C@@H](O)[C@H]1O    OC[C@H]1O[C@H](OP(O)(=O)OP(O)(=O)OC[C@H]2O[C@H]([C@H](O)[C@@H]2O)N3C=CC(=O)NC3=O)[C@H](O)[C@@H](O)[C@@H]1O    O=C1NC=CC(=O)N1    NC(N)=O    OC[C@H]1O[C@H]([C@H](O)[C@@H]1O)N2C=CC(=O)NC2=O    OC(=O)/C=C/C1=C[NH]C=N1    CCCCC(O)=O    CC(C)[C@H](N)C(O)=O    CCCC(CCC)C(O)=O    COC1=C(O)C=CC(=C1)C(O)=O    O=C1NC(=O)C2=C(N1)N=C[NH]2    OC[C@H]1O[C@H]([C@H](O)[C@@H]1O)[N]2C=NC3=C2NC(=O)NC3=O    O[C@@H]1COC(O)[C@H](O)[C@H]1O    OC(=O)C/C(=C/C(O)=O)C(O)=O    NC1=NC(=O)N(C=C1)[C@H]2C[C@H](O)[C@@H](COP(O)(=O)OP(O)(=O)OP(O)(O)=O)O2    CC1=CN([C@H]2C[C@H](O)[C@@H](COP(O)(=O)OP(O)(=O)OP(O)(O)=O)O2)C(=O)NC1=O    O[C@H]1[C@H](O)[C@H](O)[C@H](O)[C@@H](O)[C@@H]1O    CC1=C(O)C=CC=C1    CC1=CC=C(O)C=C1    C[N+](C)(C)CCOP([O-])(=O)OC[C@H](O)CO    O[C@H]1CN[C@@H](C1)C(O)=O    OC(=O)C/C(=C\C(O)=O)C(O)=O    NCCC(O)=O    N[C@@H](CCC(=O)N[C@@H](CC1=CC=CC=C1)C(O)=O)C(O)=O    C[N]1C=NC=C1C[C@H](N)C(O)=O    C[N]1C=NC(=C1)C[C@H](N)C(O)=O
    #nmr_x185.cnx    7.00    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    0.3675    <-->    <-->    <-->    <-->    <-->    0.0625    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    0.0594    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    3.8157    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    0.1246    <-->    <-->    <-->    1.8834    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    <-->    0.0724

class Chenomx(ImportPlugin):

    def __init__(self, **kwargs):
        super(Chenomx, self).__init__(**kwargs)
        self.register_app_launcher(ChenomxApp)
