#
#   PROGRAM:     Power Console
#   MODULE:      base.py
#   DESCRIPTION: Base classes and functions
#
#  The contents of this file are subject to the Initial
#  Developer's Public License Version 1.0 (the "License");
#  you may not use this file except in compliance with the
#  License. You may obtain a copy of the License at
#  http://www.firebirdsql.org/index.php?op=doc&id=idpl
#
#  Software distributed under the License is distributed AS IS,
#  WITHOUT WARRANTY OF ANY KIND, either express or implied.
#  See the License for the specific language governing rights
#  and limitations under the License.
#
#  The Original Code was created by Pavel Cisar
#
#  Copyright (c) 2007 Pavel Cisar <pcisar@users.sourceforge.net>
#  and all contributors signed below.
#
#  All Rights Reserved.
#  Contributor(s): ______________________________________.

import pkg_resources
import pyparsing as pp
import types
import sys

#--- Constants

HELP_PREFIX     = 'help_'
GRAMMAR_FULL    = 'full'
GRAMMAR_CHKCMD  = 'chkcmd'

#--- Globals

displays = {}   # Global cache of display instances. key is class name+context
renderers = []  # Global cache of object renderer classes

#--- Exceptions

class pcException(Exception):
    """An exception generated directly by PowerConsole.

    All exceptions thrown by PowerConsole should be derived from this class.
    """

    def __init__(self, message):
        """Construct a new 'PWException'.

        'message' -- A string describing the cause of the message.
        """

        Exception.__init__(self, message)

class pcError(pcException):
    """General programming error generated by PowerConsole."""
    pass


class pcParseError(pcException):
    """Wrongly specified internal command."""
    pass

#--- Functions

def isiterable(obj):
    """Return True if 'obj' supports new iteration protocol."""
    return hasattr(obj, '__iter__') and callable(obj.__iter__)

def unquote(str):
    """Return 'str' unquoted."""
    if (str[-1:] == '"') and (str[:1] == '"'):
        return str[1:-1]
    else:
        return str

def iif(expr,valueA,valueB):
    if expr: return valueA
    else: return valueB

def columnize(list, displaywidth=80):
    """Generator func that returns a list of strings as a compact set of columns.

    Each column is only as wide as necessary.
    Columns are separated by three spaces (one was not legible enough).
    
    list    An iterable of items. If item is not a string, str(item) is used.
    """

    itemlist = [str(item) for item in list]
    size = len(itemlist)
    if size == 1:
        print '%s' % str(itemlist[0])
        return
    # Try every row count from 1 upwards
    for nrows in range(1, len(itemlist)):
        ncols = (size+nrows-1) // nrows
        colwidths = []
        totwidth = -3
        for col in range(ncols):
            colwidth = 0
            for row in range(nrows):
                i = row + nrows*col
                if i >= size:
                    break
                x = itemlist[i]
                colwidth = max(colwidth, len(x))
            colwidths.append(colwidth)
            totwidth += colwidth + 3
            if totwidth > displaywidth:
                break
        if totwidth <= displaywidth:
            break
    else:
        nrows = len(itemlist)
        ncols = 1
        colwidths = [0]
    lines = []
    for row in range(nrows):
        texts = []
        for col in range(ncols):
            i = row + nrows*col
            if i >= size:
                x = ""
            else:
                x = itemlist[i]
            texts.append(x)
        while texts and not texts[-1]:
            del texts[-1]
        for col in range(len(texts)):
            texts[col] = texts[col].ljust(colwidths[col])
        yield "%s" % str("   ".join(texts))

def update_meta (self, other):
    self.__name__ = other.__name__
    self.__doc__ = other.__doc__
    self.__dict__.update(other.__dict__)
    return self

#--- Classes

class LateBindingProperty (property):
    """Recipe from Tim Delaney, 2005/03/31
       http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/408713

class C(object):

    def getx(self):
        print 'C.getx'
        return self._x

    def setx(self, x):
        print 'C.setx'
        self._x = x

    def delx(self):
        print 'C.delx'
        del self._x

    x = LateBindingProperty(getx, setx, delx)

class D(C):

    def setx(self, x):
        print 'D.setx'
        super(D, self).setx(x)

    def delx(self):
        print 'D.delx'
        super(D, self).delx()

c = C()
c.x = 1
c.x
c.x
del c.x

print

d = D()
d.x = 1
d.x
d.x
del d.x

This has the advantages that:

a. You get back an actual property object (with attendant memory savings, performance increases, etc);

b. It's the same syntax as using property(fget, fset, fdel, doc) except for the name;

c. It will fail earlier (when you define the class as opposed to when you use it).

d. It's shorter ;)

e. If you inspect the property you will get back functions with the correct __name__, __doc__, etc.

    """
    def __new__(self, fget=None, fset=None, fdel=None, doc=None):
        if fget is not None:
            def __get__(obj, objtype=None, name=fget.__name__):
                fget = getattr(obj, name)
                return fget()
            fget = update_meta(__get__, fget)
        if fset is not None:
            def __set__(obj, value, name=fset.__name__):
                fset = getattr(obj, name)
                return fset(value)
            fset = update_meta(__set__, fset)
        if fdel is not None:
            def __delete__(obj, name=fdel.__name__):
                fdel = getattr(obj, name)
                return fdel()
            fdel = update_meta(__delete__, fdel)
        return property(fget, fset, fdel, doc)


class HelpProvider(object):
    """Base class for all PowerConsole built-in help topic providers."""

    def __init__(self, pythonhelp):
        self._help = pythonhelp
        self._topics = []
        for attr in dir(self):
            if attr.startswith(HELP_PREFIX):
                self._topics.append(attr[len(HELP_PREFIX):])

    def getTopics(self):
        """Return list of topics handled by provider.

        It's possible that provider can handle additional topics 
        not included in this list, but all listed topics must be handled.

        By default, it returns list of names for instance attributes with name 
        matching HELP_PREFIX+topic pattern stripped from HELP_PREFIX.
        """

        return self._topics

    def canHandle(self,topic):
        """Return True if this instance can provide help for 'topic'.
        
        By default, instance can provide help if it has attribute with
        name matching HELP_PREFIX+topic pattern.
        """

        return hasattr(self, HELP_PREFIX+topic)

    def getHelp(self, topic):
        """Return help for 'topic'.

        By default, it returns attribute with name matching HELP_PREFIX+topic.
        """

        return getattr(self, HELP_PREFIX+topic, None)


class Command(object):
    """Base class for all PowerConsole built-in commands."""
    
    usesTerminator = False
    
    # Handy basic grammar building blocks for use in descendants

    INTEGER      = pp.Word(pp.nums+"+-",pp.nums).setName('integer')
    INTEGER.setParseAction(lambda st, loc, toks:int(toks[0]))
    IDENT        = pp.Word(pp.alphas,pp.alphanums+'_.').setName('ident')
    IDENT.errmsg = 'identifier expected'
    FILENAME     = (pp.Word(pp.alphas + 
                    pp.alphas8bit + pp.nums + '_/\-:.').setName('filename') | 
                    pp.QuotedString("'")).setName('filename')
    CHUNK        = pp.CharsNotIn('\n')
    ARG          = pp.White().suppress() + pp.OneOrMore(CHUNK + (
                    (pp.Optional(pp.lineEnd.setParseAction(pp.replaceWith(' '))) +
                    pp.FollowedBy(CHUNK)) ^
                    pp.Optional(pp.lineEnd.suppress())
                    )
        ).setName('arg')
    EXPR         = pp.White().suppress() + pp.OneOrMore(CHUNK + (
                    (pp.Optional(pp.lineEnd.setParseAction(pp.replaceWith(' '))) +
                    pp.FollowedBy(CHUNK)) ^
                    pp.Optional(pp.lineEnd.suppress())
                    )
        ).setName('expr')

    def __init__(self,controller):
        self.controller = controller
        self._name = None

    def _getUserNamespace(self):
        """Return dictionary of names defined as locals in controller."""

        return self.controller.locals

    def _getContextLocals(self,depth=0):
        """Return locals from specified frame.
        
        'depth' is number of frames up in stack; 0 means one frame above caller.
        """

        return sys._getframe(depth+2).f_locals

    def _tokenToInt(self, st, loc, toks):
        return int(toks[0])

    def _upper(s, l, tokens):
        return tokens[0].upper()

    def _fail(self, s, loc, expr, err):
        """pyparsing fail action that raises ParseFatalException.

        It's used to stop parsing with error on required grammar elements.
        
        EXAMPLE:
        Next defined grammar element for: USER 'username' that will fail
        when 'username' is not specified.
        optUser = keyUser + \
            QuotedString("'").setResultsName('user').setFailAction(self._fail)
        """

        raise pp.ParseFatalException(s,loc,err,expr)

    def _compile(s, l, tokens):
        """pyparsing parse action that converts parsed command.
        
        Command is converted into string with python code to call it's 
        implementation function with parameters.

        The resulting code has format:
        _NAME_execute(PARAM_LIST)
        where:
        NAME is value of grammar token with ResultsName "cmd"
        PARAM_LIST is comma-separated list of named parameters (i.e. name=value)
        constructed from named grammar tokens.
        
        Because the result must be valid Python function call, parameter values
        are quoted using next logic:
        1. IntType is converted to arg=value
        2. String types are converted to arg='value' using correct quotation for 
           given value.
        3. Value of any other type is converted to arg='value'
        
        IMPORTANT:
        This parse action must be assigned to root (and only to root) token
        in defined grammar.
        
        EXAMPLE:
        Let's have next grammar:
        action [argument]
        So the definition in descendant's __init__ is:
            self.keyAction = CaselessKeyword('action')
            self.cmdAction = self.keyAction.setResultsName('cmd') + \
                Optional(self.arg.setResultsName('arg'))
            self.cmdAction.setParseAction(self._compile)
        Then next commands will be translated into:
        >>>parseString('action')
        _action_execute()
        >>>parseString('action 1')
        _action_execute(arg=1)
        >>>parseString('action doIt')
        _action_execute(arg='doIt')
        >>>parseString("action 'doIt'")
        _action_execute(arg="'doIt'")
        >>>parseString('action "doIt"')
        _action_execute(arg='"doIt"')
        >>>parseString('''action "It's smile time!"''')
        _action_execute(arg=""""It's smile time!"""")
        >>>parseString('action obj')
        _action_execute(arg='obj')
        """

        params = []
        for par,value in tokens.items():
            #print type(value)
            if par == 'expr':
                s = "%s=%s" % (par,value[0])
            elif par == 'cmd':
                cmd = value
                continue
            else:
                if type(value) == pp.ParseResults:
                    value = ''.join(value)
                if isinstance(value,basestring):
                    single = value.find("'")
                    double = value.find('"')
                    if (single != -1) and (double != -1):
                        s = '%s="""%s"""' % (par,value)
                    elif (single != -1) and (double == -1):
                        s = '%s="%s"' % (par,value)
                    else:
                        s = "%s='%s'" % (par,value)
                elif type(value) == types.IntType:
                    s = "%s=%i" % (par,value)
                else:
                    s = "%s='%s'" % (par,value)
            params.append(s)
        return '_%s_execute(%s)' % (cmd,','.join(params))

    def _getGrammar(self):
        """Return grammar for command.

        Descendant has to override it.
        """

        return None

    def _getCmdKeyword(self):
        """Return grammar element for command's keyword(s). 
        
        It's used by interpreter to identify input lines where internal command
        starts whithout fully parsing the command.
        
        Descendant has to override it.
        """

        return None

    def getName(self):
        """Return command name defined by grammar token with 'cmd' resultsName."""

        def _getName(grammar):
            if grammar is None:
                return None
            if hasattr(grammar,'exprs'):
                l = grammar.exprs
            else:
                l = [grammar,]
            for g in l:
                if g.resultsName == 'cmd':
                    return g.name
                elif hasattr(g,'exprs') and len(g.exprs) > 0:
                    name = _getName(g)
                    if name is not None:
                        return name
            return None

        if self._name is None:
            name = _getName(self._getGrammar())
            if name is not None:
                self._name = unquote(name)
            else:
                raise pcError("Command %s doesn't have a name." % 
                                self.__class__.__name__)
        return self._name

    def getGrammar(self, grammarType):
        """Return grammar definition of requested type.
        
        Return None if grammar of requested type is not available.
        
        There could be multiple 'types' of grammar for command used for
        different purposes.
        
        Now recognized grammar types are:
        GRAMMAR_FULL -- Grammar used to parse and convert command to callable.
        GRAMMAR_CHKCMD -- Grammar for check whether line might be this command.
        """

        if grammarType == GRAMMAR_FULL:
            return self._getGrammar()
        elif grammarType == GRAMMAR_CHKCMD:
            return self._getCmdKeyword()
        else:
            return None

    def execute(self,**kwargs):
        """Main method that implements the command.
        
        Defined grammar (GRAMMAR_FULL) must convert the command text into
        python code to call this function under name '_<cmdname>_execute'
        with specified arguments.

        Descendant has to override it.
        """

        print "Method %s.execute not implemented" % self.__class__.__name__


class ObjectVisitor(object):
    """Base/mixin class for object visitors."""

    def __getattr__(self,name):
        """Dispatch unhandled visitor call.
        
        When visitXXX is called by visitee but it's not defined by visitor, 
        this method ensures that visitDefaultAction is called instead requested 
        specialised method.
        
        This is defined just for convenience, to save unnecessary specialised 
        method existence checks in visitee.
        """

        if name.startswith('visit'):
            for extension in self.visitor_extension_list:
                if hasattr(extension,name):
                    return getattr(extension,name)
            return self.visitDefaultAction
        else:
            return super(ObjectVisitor,self).__getattr__(name)

    def visitDefaultAction(self,obj):
        pass


class Display(object):
    """Base class for all PowerConsole Displays."""

    def open(self):
        pass
    def close(self):
        pass


class LineDisplay(Display):
    """Default Line Display.
    
    Prints lines of text to the stdout.
    """

    def write(self,line=''):
        print line,
    def writeLine(self,line=''):
        print line
    def writeLines(self,lines):
        for line in lines:
            self.writeLine(line)

    
class ListDisplay(Display):
    """Default List Display.
    
    Prints list of items to the stdout in columnized format.
    """

    def writeList(self,items):
        for line in columnize(items):
            print line

class TableDisplay(Display):
    """Default Table Display.
    
    Prints data from iterable in tabular format according to layout specification.
    """
    
    def writeTable(self,description,data):
        """description: List of tuples that describe the table layout.
                        (COLUMN_NAME,WIDTH)
           data:        Iterable. Each yield must return list or tuple of items.
        """
        for col in description:
            print col[0].ljust(col[1]),
        print
        for col in description:
            print '=' * col[1],
        print
        for row in data:
            for index in xrange(len(description)):
                value = row[index]
                if isinstance(value,types.StringTypes):
                    value = value.ljust(description[index][1])
                else:
                    value = str(value).rjust(description[index][1])
                print value,
            print

class ObjectDisplay(Display,ObjectVisitor):
    """Default Object Display.
    
    Prints string representation of object to the stdout.
    """

    def __init__(self):
        self.visitor_extension_list = [renderer(self) for renderer in renderers]

    def write(self,line=''):
        print line,
    def writeLine(self,line=''):
        print line
    def writeLines(self,lines):
        for line in lines:
            self.writeLine(line)

    def visitDefaultAction(self,obj):
        """Prints string representation of 'obj' to the stdout.
        
        Does nothing if 'obj' is None.
        """

        print str(obj)

    def writeObject(self,obj):
        """Display object.
        
        If 'obj' supports PowerConsole visitor pattern (obj.acceptVisitor), 
        it's used. Otherwise it calls self.visitDefaultAction(obj).
        """

        if hasattr(obj,'acceptVisitor'):
            obj.acceptVisitor(self)
        else:
            self.visitDefaultAction(obj)


class ObjectListDisplay(Display,ObjectVisitor):
    """Default Object List Display.
    
    Prints string representation of objects in list to the stdout.
    """

    def __init__(self):
        self.visitor_extension_list = [renderer(self) for renderer in renderers]

    def write(self,line=''):
        print line,
    def writeLine(self,line=''):
        print line
    def writeLines(self,lines):
        for line in lines:
            self.writeLine(line)

    def visitDefaultAction(self,obj):
        """Prints string representation of 'obj' to the stdout."""

        print str(obj)

    def writeList(self,items):
        """Display list of objects.
        
        If particular object in list supports PowerConsole visitor pattern 
        (obj.acceptVisitor), it's used. Otherwise it calls 
        self.visitDefaultAction(obj).
        """

        for obj in items:
            if hasattr(obj,'acceptVisitor'):
                obj.acceptVisitor(self)
            else:
                self.visitDefaultAction(obj)


class UserInterfaceProvider(object):
    """Base class for PowerConsole user interface providers.
    
    Default implementation returns default displays and input dialogs.

    All methods returning UI instances take optional 'context' parameter
    that define use context for requested UI widget. It's typically a command 
    name optionally followed by internal context specification(s) separated by
    dot, for example: 'show.detail', 'show.list' etc.
    If context is not specified, then 'main' context is assumed.
    
    The rationale behind it is to allow UI customization for various
    contexts via configuration.
    """

    def __init__(self):
        for renderer in pkg_resources.iter_entry_points("powerconsole.text_renderers"):
            try:
                obj = renderer.load()
            except:
                ## ToDo: Handle exceptions in provider initialization
                raise
            else:
                renderers.append(obj)
        
    def getLineDisplay(self,context='main'):
        """Returns a LineDisplay instance.
        
        Optional 'context' is string that define use context for requested 
        display. See class docstring for details.
        """

        return displays.setdefault('LineDisplay.' + context, LineDisplay())

    def getListDisplay(self,context='main'):
        """Returns a ListDisplay instance.
        
        Optional 'context' is string that define use context for requested 
        display. See class docstring for details.
        """

        return displays.setdefault('ListDisplay.' + context, ListDisplay())

    def getTableDisplay(self,context='main'):
        """Returns a TableDisplay instance.
        
        Optional 'context' is string that define use context for requested 
        display. See class docstring for details.
        """

        return displays.setdefault('TableDisplay.' + context, TableDisplay())

    def getObjectDisplay(self,context='main'):
        """Returns a ObjectDisplay instance.
        
        Optional 'context' is string that define use context for requested 
        display. See class docstring for details.
        """

        return displays.setdefault('ObjectDisplay.' + context, ObjectDisplay())

    def getObjectListDisplay(self,context='main'):
        """Returns a ObjectListDisplay instance.
        
        Optional 'context' is string that define use context for requested 
        display. See class docstring for details.
        """

        return displays.setdefault('ObjectListDisplay.' + context, ObjectListDisplay())


