#
#   PROGRAM:     Power Console
#   MODULE:      base.py
#   DESCRIPTION: Base classes and functions
#
#  The contents of this file are subject to the Initial
#  Developer's Public License Version 1.0 (the "License");
#  you may not use this file except in compliance with the
#  License. You may obtain a copy of the License at
#  http://www.firebirdsql.org/index.php?op=doc&id=idpl
#
#  Software distributed under the License is distributed AS IS,
#  WITHOUT WARRANTY OF ANY KIND, either express or implied.
#  See the License for the specific language governing rights
#  and limitations under the License.
#
#  The Original Code was created by Pavel Cisar
#
#  Copyright (c) 2007 Pavel Cisar <pcisar@users.sourceforge.net>
#  and all contributors signed below.
#
#  All Rights Reserved.
#  Contributor(s): ______________________________________.

import pkg_resources
import itertools
import pyparsing as pp
import types
import sys

#--- Constants

HELP_PREFIX     = 'help_'
GRAMMAR_FULL    = 'full'
GRAMMAR_CHKCMD  = 'chkcmd'
CMD_NAME        = 'cmd_name' # Grammar ResultSet identifier for command name
UI_TEXT         = 1
UI_LIST         = 2
UI_OBJECT       = 3
UI_OBJECTLIST   = 4
UI_TABLE        = 5

#--- Globals

displays = {}   # Global cache of display instances. key is class name+context
renderers = []  # Global cache of object renderer classes

#--- Exceptions

class pcException(Exception):
    """An exception generated directly by PowerConsole.

    All exceptions thrown by PowerConsole should be derived from this class.
    """

    def __init__(self, message):
        """Construct a new 'PWException'.

        'message' -- A string describing the cause of the message.
        """

        Exception.__init__(self, message)

class pcError(pcException):
    """General programming error generated by PowerConsole."""
    pass


class pcParseError(pcException):
    """Wrongly specified internal command."""
    pass

#--- Functions

def isiterable(obj):
    """Returns True if 'obj' supports new iteration protocol."""
    return hasattr(obj, '__iter__') and callable(obj.__iter__)

def unquote(str):
    """Returns 'str' unquoted."""
    if (str[-1:] == '"') and (str[:1] == '"'):
        return str[1:-1]
    else:
        return str

def iif(expr,valueA,valueB):
    if expr: return valueA
    else: return valueB

def asString(expr):
    if isinstance(expr,types.StringTypes):
        return expr
    else:
        return str(expr)
    

def columnize(array, displaywidth=80, colsep = '  ', 
              arrange_vertical=True, ljust=True, lineprefix=''):
    """"Generator func that returns a list of strings as a compact set of columns
    arranged horizontally or vertically.

    For example, for a line width of 4 characters (arranged vertically):

        ['1', '2,', '3', '4'] => '1  3','2  4'
   
    or arranged horizontally:

        ['1', '2,', '3', '4'] => '1  2' '3  4'

    Each column is only as wide as necessary.  By default, columns are
    separated by two spaces - one was not legible enough. Set "colsep"
    to adjust the string separate columns. Set `displaywidth' to set
    the line width. 

    Normally, consecutive items go down from the top to bottom from
    the left-most column to the right-most. If "arrange_vertical" is
    set false, consecutive items will go across, left to right, top to
    bottom.
    
    :author: Rocky Bernstein, http://code.google.com/p/pycolumnize/
    Adapted to generator function by Pavel Cisar"""

    array = [str(i) for i in array]

    # Some degenerate cases
    size = len(array)
    if 0 == size: 
        yield "<empty>"
        return
    elif size == 1:
        yield '%s' % str(array[0])
        return

    displaywidth = max(4, displaywidth - len(lineprefix))
    if arrange_vertical:
        array_index = lambda nrows, row, col: nrows*col + row
        # Try every row count from 1 upwards
        for nrows in range(1, size):
            ncols = (size+nrows-1) // nrows
            colwidths = []
            totwidth = -len(colsep)
            for col in range(ncols):
                # get max column width for this column
                colwidth = 0
                for row in range(nrows):
                    i = array_index(nrows, row, col)
                    if i >= size: break
                    x = array[i]
                    colwidth = max(colwidth, len(x))
                colwidths.append(colwidth)
                totwidth += colwidth + len(colsep)
                if totwidth > displaywidth: 
                    break
            if totwidth <= displaywidth: 
                break
        # The smallest number of rows computed and the
        # max widths for each column has been obtained.
        # Now we just have to format each of the
        # rows.
        s = ''
        for row in range(nrows):
            texts = []
            for col in range(ncols):
                i = row + nrows*col
                if i >= size:
                    x = ""
                else:
                    x = array[i]
                texts.append(x)
            while texts and not texts[-1]:
                del texts[-1]
            for col in range(len(texts)):
                if ljust:
                    texts[col] = texts[col].ljust(colwidths[col])
                else:
                    texts[col] = texts[col].rjust(colwidths[col])
            yield "%s%s" % (lineprefix, str(colsep.join(texts)))
    else:
        array_index = lambda nrows, row, col: ncols*(row-1) + col
        # Try every column count from size downwards
        prev_colwidths = []
        colwidths = []
        for ncols in range(size, 0, -1):
            # Try every row count from 1 upwards
            min_rows = (size+ncols-1) // ncols
            for nrows in range(min_rows, size):
                rounded_size = nrows * ncols
                colwidths = []
                totwidth  = -len(colsep)
                for col in range(ncols):
                    # get max column width for this column
                    colwidth  = 0
                    for row in range(1, nrows+1):
                        i = array_index(nrows, row, col)
                        if i >= rounded_size: break
                        elif i < size:
                            x = array[i]
                            colwidth = max(colwidth, len(x))
                    colwidths.append(colwidth)
                    totwidth += colwidth + len(colsep)
                    if totwidth >= displaywidth: 
                        break
                if totwidth <= displaywidth and i >= rounded_size-1:
                    # Found the right nrows and ncols
                    nrows  = row
                    break
                elif totwidth >= displaywidth:
                    # Need to reduce ncols
                    break
            if totwidth <= displaywidth and i >= rounded_size-1:
                break
        # The smallest number of rows computed and the
        # max widths for each column has been obtained.
        # Now we just have to format each of the
        # rows.
        s = ''
        for row in range(1, nrows+1):
            texts = []
            for col in range(ncols):
                i = array_index(nrows, row, col)
                if i >= size:
                    break
                else: x = array[i]
                texts.append(x)
                pass
            for col in range(len(texts)):
                if ljust:
                    texts[col] = texts[col].ljust(colwidths[col])
                else:
                    texts[col] = texts[col].rjust(colwidths[col])
            yield "%s%s" % (lineprefix, str(colsep.join(texts)))

def update_meta (self, other):
    self.__name__ = other.__name__
    self.__doc__ = other.__doc__
    self.__dict__.update(other.__dict__)
    return self

#--- Classes

class struct(object):
    """Class to hold arbitrary data as attributes (structure).
    """
    pass

class LateBindingProperty (property):
    """Recipe from Tim Delaney, 2005/03/31
       http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/408713

::

    class C(object):
    
        def getx(self):
            print 'C.getx'
            return self._x
    
        def setx(self, x):
            print 'C.setx'
            self._x = x
    
        def delx(self):
            print 'C.delx'
            del self._x
    
        x = LateBindingProperty(getx, setx, delx)
    
    class D(C):
    
        def setx(self, x):
            print 'D.setx'
            super(D, self).setx(x)
    
        def delx(self):
            print 'D.delx'
            super(D, self).delx()
    
    c = C()
    c.x = 1
    c.x
    c.x
    del c.x
    
    print
    
    d = D()
    d.x = 1
    d.x
    d.x
    del d.x

This has the advantages that:

a. You get back an actual property object (with attendant memory savings, performance increases, etc);

b. It's the same syntax as using property(fget, fset, fdel, doc) except for the name;

c. It will fail earlier (when you define the class as opposed to when you use it).

d. It's shorter ;)

e. If you inspect the property you will get back functions with the correct __name__, __doc__, etc.

    """
    def __new__(self, fget=None, fset=None, fdel=None, doc=None):
        if fget is not None:
            def __get__(obj, objtype=None, name=fget.__name__):
                fget = getattr(obj, name)
                return fget()
            fget = update_meta(__get__, fget)
        if fset is not None:
            def __set__(obj, value, name=fset.__name__):
                fset = getattr(obj, name)
                return fset(value)
            fset = update_meta(__set__, fset)
        if fdel is not None:
            def __delete__(obj, name=fdel.__name__):
                fdel = getattr(obj, name)
                return fdel()
            fdel = update_meta(__delete__, fdel)
        return property(fget, fset, fdel, doc)


class HelpProvider(object):
    """Base class for all PowerConsole built-in help topic providers.
    
    By default, it provides all attributes which name starts with HELP_PREFIX
    as help topics (without HELP_PREFIX). These attributes are typically strings
    to be printed, but could be also callables that return the help text.
    
    For special help handling subclasses could rewrite getTopics, CanHandle and 
    getHelp methods.
    """

    def __init__(self, pythonhelp):
        self._help = pythonhelp
        self._topics = [x[len(HELP_PREFIX):] for x in dir(self) 
                        if x.startswith(HELP_PREFIX)]

    def getTopics(self):
        """Return list of topics handled by provider.

        It's possible that provider can handle additional topics 
        not included in this list, but all listed topics must be handled.

        By default, it returns list of names for instance attributes with name 
        matching HELP_PREFIX+topic pattern stripped from HELP_PREFIX.
        """

        return self._topics

    def canHandle(self,topic):
        """Return True if this instance can provide help for 'topic'.
        
        By default, instance can provide help if it has attribute with
        name matching HELP_PREFIX+topic pattern.
        """

        return topic in self._topics

    def getHelp(self, topic):
        """Return help for 'topic'.

        By default, it returns attribute with name matching HELP_PREFIX+topic.
        """

        return getattr(self, HELP_PREFIX+topic)


class Command(object):
    """Base class for all PowerConsole built-in commands."""
    
    usesTerminator = False
    
    # Handy basic grammar building blocks for use in descendants

    #: Grammar node for INTEGER value
    INTEGER      = pp.Word(pp.nums+"+-",pp.nums).setName('integer')
    INTEGER.setParseAction(lambda st, loc, toks:int(toks[0]))
    #: Grammar node for identificator
    IDENT        = pp.Word(pp.alphas,pp.alphanums+'_.').setName('ident')
    IDENT.errmsg = 'identifier expected'
    #: Grammar node for file name
    FILENAME     = (pp.QuotedString("'").setName('filename') | 
                    pp.Word(pp.alphas + pp.alphas8bit + 
                            pp.nums + '_/\-:.').setName('filename')).setName('filename')
    #: Grammar node for path
    PATH         = (pp.Word(pp.alphas + 
                    pp.alphas8bit + pp.nums + '_/\-:.').setName('path') | 
                    pp.QuotedString("'")).setName('path')
    #: Grammar node for arbitrary text up to the line separator
    CHUNK        = pp.CharsNotIn('\n')
    #: Grammar node for command argument. It may consist of several lines.
    ARG          = pp.White().setName('<arg>').suppress() + pp.OneOrMore(CHUNK + (
                    (pp.Optional(pp.lineEnd.setParseAction(pp.replaceWith(' '))) +
                    pp.FollowedBy(CHUNK)) ^
                    pp.Optional(pp.lineEnd.suppress())
                    )
        ).setName('arg')
    #: Grammar node for expression. It may consist of several lines.
    EXPR         = pp.Empty().setName('<expr>') + pp.OneOrMore(CHUNK + (
                    (pp.Optional(pp.lineEnd.setParseAction(pp.replaceWith(' '))) +
                    pp.FollowedBy(CHUNK)) ^
                    pp.Optional(pp.lineEnd.suppress())
                    )
        ).setName('expr')

    def __init__(self,controller):
        self.controller = controller
        self._name = self.__class__.__name__.replace('cmd','').lower()
        self.writeErr = controller.writeErr
        self.write = controller.write

    def _getUserNamespace(self):
        """Return dictionary of names defined as locals in controller."""

        return self.controller.locals

    def _getContextLocals(self,depth=0):
        """Return locals from specified frame.
        
        'depth' is number of frames up in stack; 0 means one frame above caller.
        """

        return sys._getframe(depth+2).f_locals

    def _tokenToInt(self, st, loc, toks):
        """Helper PyParsing parse action that convers parsed token to integer."""
        return int(toks[0])

    def _tokenTotrue(self, st, loc, toks):
        """Helper PyParsing parse action that convers parsed token to True value."""
        return True

    def _makeNameNode(self,node):
        """Helper functions that sets the specified PyParsing grammar node as
        command identification node.
        """
        return node.setName(self._name).setResultsName(CMD_NAME).setParseAction(pp.replaceWith(self._name))

    def _fail(self, s, loc, expr, err):
        """pyparsing fail action that raises ParseFatalException.

        It's used to stop parsing with error on required grammar elements.
        
        EXAMPLE:
        Next defined grammar element for USER 'username' will fail when 'username'
        is not specified::
            optUser = keyUser + \
                QuotedString("'").setResultsName('user').setFailAction(self._fail)
        """

        raise pp.ParseFatalException(s,loc,err,expr)

    def _compile(self, s, l, tokens):
        """pyparsing parse action that converts parsed command.
        
        Command is converted into string with python code to call it's 
        implementation function with parameters.

        The resulting code has format: _NAME_execute(PARAM_LIST)

        where:
        
        :NAME: is value of grammar token with ResultsName CMD_NAME
        :PARAM_LIST: is comma-separated list of named parameters (i.e. name=value)
                     constructed from named grammar tokens.
        
        Because the result must be valid Python function call, parameter values
        are quoted using next logic:
        
           1. IntType is converted to arg=value
           2. String types are converted to arg='value' using correct quotation for 
              given value.
           3. Value of any other type is converted to arg='value'
        
        :IMPORTANT:
            This parse action must be assigned to root (and only to root) token
            in defined grammar.
        
        EXAMPLE::

            Let's have next grammar: action [argument]
            
            So the definition in descendant's __init__ is:
                self.keyAction = CaselessKeyword('action')
                self.cmdAction = self.keyAction.setResultsName(CMD_NAME) + \
                    Optional(self.arg.setResultsName('arg'))
                self.cmdAction.setParseAction(self._compile)

            Then next commands will be translated into:
            >>>parseString('action')
            _action_execute()
            >>>parseString('action 1')
            _action_execute(arg=1)
            >>>parseString('action doIt')
            _action_execute(arg='doIt')
            >>>parseString("action 'doIt'")
            _action_execute(arg="'doIt'")
            >>>parseString('action "doIt"')
            _action_execute(arg='"doIt"')
            >>>parseString('''action "It's smile time!"''')
            _action_execute(arg=""""It's smile time!"""")
            >>>parseString('action obj')
            _action_execute(arg='obj')
        """

        params = []
        for par,value in tokens.items():
            if par == 'expr':
                s = "%s=%s" % (par,value[0])
            elif par == CMD_NAME:
                cmd = value
                continue
            else:
                if type(value) == pp.ParseResults:
                    value = ''.join(value)
                if isinstance(value,basestring):
                    single = value.find("'")
                    double = value.find('"')
                    if (single != -1) and (double != -1):
                        s = '%s="""%s"""' % (par,value)
                    elif (single != -1) and (double == -1):
                        s = '%s="%s"' % (par,value)
                    else:
                        s = "%s='%s'" % (par,value)
                elif type(value) == types.IntType:
                    s = "%s=%i" % (par,value)
                elif type(value) == types.BooleanType:
                    s = "%s=%s" % (par,value)
                else:
                    s = "%s='%s'" % (par,value)
            params.append(s)
        return '_%s_execute(%s)' % (cmd,','.join(params))

    def _getGrammar(self):
        """Return grammar for command.

        Descendant has to override it.
        """

        return None

    def _getCmdKeyword(self):
        """Return grammar element for command's keyword(s). 
        
        It's used by interpreter to identify input lines where internal command
        starts whithout fully parsing the command.
        
        Descendant has to override it.
        """

        return None

    def getName(self):
        """Returns the command name.
        
        By default, the name is lowercased class name without 'cmd' prefix (if it's present).
        
        """

        return self._name

    def getGrammar(self, grammarType):
        """Returns grammar definition of requested type.
        
        Return None if grammar of requested type is not available.
        
        There could be multiple 'types' of grammar for single command used for
        different purposes.
        
        Now recognized grammar types are:
        
        :GRAMMAR_FULL: Grammar used to parse and convert command to callable.
        :GRAMMAR_CHKCMD: Grammar for check whether line might be this command.
        """

        if grammarType == GRAMMAR_FULL:
            return self._getGrammar()
        elif grammarType == GRAMMAR_CHKCMD:
            return self._getCmdKeyword()
        else:
            return None

    def execute(self,**kwargs):
        """Main method that implements the command.
        
        Defined grammar (GRAMMAR_FULL) must convert the command text into
        python code to call this function under name '_<cmdname>_execute'
        with specified arguments.

        Descendant has to override it.
        """

        print "Method %s.execute not implemented" % self.__class__.__name__


class ObjectVisitor(object):
    """Base/mixin class for object visitors."""

    def __getattr__(self,name):
        """Dispatch unhandled visitor call.
        
        When visitXXX is called by visitee but it's not defined by visitor, 
        it first makes a lookup in list of externally defined visitor extensions
        and eventually returns visitDefaultAction if no specialised method is found.
        """

        if name.startswith('visit'):
            for extension in self.visitor_extension_list:
                if hasattr(extension,name):
                    return getattr(extension,name)
            return self.visitDefaultAction
        else:
            return super(ObjectVisitor,self).__getattr__(name)

    def handleObject(self,obj):
        """Handle objects that don't support the visitor pattern.
        
        Makes a lookup in list of externally defined visitor extensions for
        the first one that can handle object of this type.
        Calls visitDefaultAction if no specialised handler is found.
        """
        name = 'handle_'+type(obj).__name__
        for extension in self.visitor_extension_list:
            if hasattr(extension,name):
                getattr(extension,name)(obj)
                return
        self.visitDefaultAction(obj)
        
    def visitDefaultAction(self,obj):
        """Default visitor action. Does nothing in base implementation."""
        pass

class Display(ObjectVisitor):
    """Default Display.
    
    Supports all defined display interfaces.
    """

    def __init__(self):
        super(Display,self).__init__()
        self.visitor_extension_list = [renderer(self) for renderer in renderers]

    def provides(self):
        """Returns list of interface codes this display implements."""
        return [UI_TEXT,UI_LIST,UI_TABLE,UI_OBJECT,UI_OBJECTLIST]
    
    # UI_TEXT interface
    def write(self,text=''):
        """Writes text (without the line end)."""
        if not isinstance(text,types.StringTypes):
            text = str(text)
        sys.stdout.write(text)
    def writeLine(self,line=''):
        """Writes line of text (with the line end)."""
        print line
    def writeLines(self,lines):
        """Writes lines from iterable  (with the line end after each line)"""
        for line in lines:
            self.writeLine(line)
    
    # UI_LIST interface
    def writeList(self,items):
        """Writes list of items in columnized format.
        
        :items: List or iterable of elements. If item is not a string, then its
                str() representation is used.
        """
        for line in columnize(items):
            print line
    
    # UI_TABLE interface
    def writeTable(self,description,data):
        """Prints data from iterable in tabular format according to layout specification.
        
        :description: List of tuples that describe the table layout.
                        (COLUMN_NAME,WIDTH)
        :data:        Iterable. Each yield must return list or tuple of items.
        """
        for col in description:
            print col[0].ljust(max(col[1],len(col[0]))),
        print
        for col in description:
            print '=' * max(col[1],len(col[0])),
        print
        for row in data:
            for index in xrange(len(description)):
                value = row[index]
                col = description[index]
                if isinstance(value,types.StringTypes):
                    value = value.ljust(max(col[1],len(col[0])))
                else:
                    value = str(value).rjust(max(col[1],len(col[0])))
                print value,
            print
    
    def visitDefaultAction(self,obj):
        """Prints string representation of 'obj' to the stdout.
        
        """
        print str(obj)

    # UI_OBJECT interface
    def writeObject(self,obj):
        """Prints string representation of object to the stdout. 
        
        Uses Visitor pattern to write out the object. If object doesn't supports
        Visitors, then it attempts to use custom renderers that can handle 
        the object, otherwise str() representation is written.
        """
        if hasattr(obj,'acceptVisitor'):
            obj.acceptVisitor(self)
        else:
            self.handleObject(obj)
    
    # UI_OBJECTLIST interface
    def writeObjectList(self,items):
        """Display list of objects.
        
        Uses Visitor pattern to write out the object. If object doesn't supports
        Visitors, then str() representation is written.
        """

        for obj in items:
            if hasattr(obj,'acceptVisitor'):
                obj.acceptVisitor(self)
            else:
                self.handleObject(obj)
    
class UserInterfaceProvider(object):
    """Base class for PowerConsole user interface providers.
    
    Default implementation returns default displays and input dialogs.

    All methods returning UI instances take optional 'context' parameter
    that define use context for requested UI widget. It's typically a command 
    name optionally followed by internal context specification(s) separated by
    dot, for example: 'show.detail', 'show.list' etc.
    If context is not specified, then 'main' context is assumed.
    
    The rationale behind it is to allow UI customization for various
    contexts via configuration.
    """

    def __init__(self, packages, factory=None):
        self.packages = packages
        if factory:
            self.factory = factory
        else:
            self.factory = self._factory
        for renderer in itertools.chain(*(p.object_renderers() for p 
                                           in self.packages 
                                           if hasattr(p,'object_renderers'))):
            try:
                obj = renderer
            except:
                ## ToDo: Handle exceptions in provider initialization
                raise
            else:
                renderers.append(obj)
    
    def _factory(self,interface):
        return Display()
    
    def getDisplay(self,context='main',interface=UI_TEXT):
        """Returns a Display instance.
        
        :interface:  One or more interface specifications for requested display.
        
        :context: String that defines usage context for requested display. 
                  See class docstring for details. Defaults to 'main'.
        """

        return displays.setdefault(context, self.factory(interface))

    def raw_input(self, prompt=""):
        """Write a prompt and read a line.

        The returned line does not include the trailing newline.
        When the user enters the EOF key sequence, EOFError is raised.

        The base implementation uses the built-in function
        raw_input(); a subclass may replace this with a different
        implementation.
        """

        return raw_input(prompt)
    
    def promptYN(self,message):
        answer = self.raw_input(message+' (Y/n)?').lower()
        while answer != '' and answer not in 'yn':
            answer = self.raw_input(message).lower()
        else:
            return (answer == '' or answer == 'y')

class ExtensionPackage(object):
    """Base class for PowerConsole extension packages."""
    def __init__(self,pkgname):
        self.name = pkgname
        self.version = 'UNKNOWN'
        self.summary = 'UNKNOWN'
        self.url = 'UNKNOWN'
        self.download_url = 'UNKNOWN'
        self.description = 'UNKNOWN'
        self.author = 'UNKNOWN'
        self.author_email = 'UNKNOWN'
        self.license = 'UNKNOWN'
        pkg = pkg_resources.get_distribution(pkgname)
        for line in pkg.get_metadata_lines('PKG-INFO'):
            lowline = line.lower()
            if lowline.startswith('version:'):
                self.version = self._get_value(line)
            elif lowline.startswith('summary:'):
                self.summary = self._get_value(line)
            elif lowline.startswith('home-page:'):
                self.url = self._get_value(line)
            elif lowline.startswith('download-url:'):
                self.download_url = self._get_value(line)
            elif lowline.startswith('description:'):
                self.description = self._get_value(line)
            elif lowline.startswith('author:'):
                self.author = self._get_value(line)
            elif lowline.startswith('author-email:'):
                self.author_email = self._get_value(line)
            elif lowline.startswith('license:'):
                self.license = self._get_value(line)
    def _get_value(self,line):
        i = line.find(':')
        return line[i+1:].strip()
    
